//$Id: preferences.cc,v 1.21 2005/11/10 21:43:26 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2005 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "preferences.h"

#include <gconfmm/client.h>
#include <algorithm>

namespace Guikachu
{
    namespace Preferences
    {
	Glib::RefPtr<Gnome::Conf::Client> gconf_client;

	static std::string get_gconf_string (const std::string &key);
    }
}

#define GCONF_ROOT        "/apps/guikachu"
#define GCONF_COLORS_ROOT "/FormEditor/Colors"

using namespace Guikachu;

void Preferences::init ()
{
    // Set up GConf client
    gconf_client = Gnome::Conf::Client::get_default_client ();
    gconf_client->add_dir ("/apps/guikachu", Gnome::Conf::CLIENT_PRELOAD_RECURSIVE);
    
    // Set up GConf callbacks
    // User interface
    gconf_client->notify_add (GCONF_ROOT "/Interface",
                              sigc::hide (sigc::hide (Interface::recent_files_changed.make_slot ())));
    
    // Colors
    gconf_client->notify_add (GCONF_ROOT GCONF_COLORS_ROOT,
                              sigc::hide (sigc::hide (FormEditor::colors_changed.make_slot ())));
}

std::string Preferences::get_gconf_string (const std::string &key)
{
    std::string real_key = GCONF_ROOT + key;
    return gconf_client->get_string (real_key);;
}



sigc::signal0<void> Preferences::Interface::recent_files_changed;

Preferences::Interface::RecentFiles Preferences::Interface::get_recent_files ()
{
    return gconf_client->get_string_list (GCONF_ROOT "/Interface/recent_files");
}

void Preferences::Interface::add_recent_file (const Glib::ustring &uri)
{
    RecentFiles recent_list = get_recent_files ();

    // Remove latest addition from list
    RecentFiles::iterator old_pos =
	std::find (recent_list.begin (), recent_list.end (), uri);
    if (old_pos != recent_list.end ())
	recent_list.erase (old_pos);

    // Prepend latest addition to list
    recent_list.push_front (uri);
    
    // Truncate list
    int recent_list_size = gconf_client->get_int (GCONF_ROOT "/Interface/recent_files_size");
    int pos;
    RecentFiles::iterator truncate_begin;

    for (pos = 0, truncate_begin = recent_list.begin ();
	 pos < recent_list_size && truncate_begin != recent_list.end ();
	 pos++, truncate_begin++)
	;

    if (truncate_begin != recent_list.end ())
	recent_list.erase (truncate_begin, recent_list.end ());

    Gnome::Conf::Value gval (Gnome::Conf::VALUE_LIST);
    gval.set_list_type (Gnome::Conf::VALUE_STRING);
    gval.set_string_list (recent_list);

    gconf_client->set (GCONF_ROOT "/Interface/recent_files", gval);
}

int Preferences::Interface::get_undo_size ()
{
    return gconf_client->get_int (GCONF_ROOT "/Interface/undo_size");
}

bool Preferences::MainWin::get_default_grouped ()
{
    return gconf_client->get_bool (GCONF_ROOT "/MainWin/default_grouped");
}


sigc::signal0<void> Preferences::FormEditor::colors_changed;

void Preferences::FormEditor::set_colors (const std::string &color_fg,
					  const std::string &color_disabled,
					  const std::string &color_bg,
					  const std::string &color_selection)
{
#define SET_COLOR(key,value) gconf_client->set (GCONF_ROOT "/FormEditor/Colors/" key, value)

    SET_COLOR("foreground", color_fg);
    SET_COLOR("foreground_disabled", color_disabled);
    SET_COLOR("background", color_bg);
    SET_COLOR("selection", color_selection);

#undef SET_COLOR
    
}

std::string Preferences::FormEditor::get_color_fg ()
{
    return get_gconf_string (GCONF_COLORS_ROOT "/foreground");
}

std::string Preferences::FormEditor::get_color_disabled ()
{
    return get_gconf_string (GCONF_COLORS_ROOT "/foreground_disabled");
}

std::string Preferences::FormEditor::get_color_bg ()
{
    return get_gconf_string (GCONF_COLORS_ROOT "/background");
}

std::string Preferences::FormEditor::get_color_selection ()
{
    return get_gconf_string (GCONF_COLORS_ROOT "/selection");
}

float Preferences::FormEditor::get_default_zoom ()
{
    return gconf_client->get_float (GCONF_ROOT "/FormEditor/default_zoom");
}

void Preferences::FormEditor::set_default_zoom (float default_zoom)
{
    gconf_client->set (GCONF_ROOT "/FormEditor/default_zoom", default_zoom);
}
