//$Id: menu-res-ops.cc,v 1.6 2005/07/29 09:54:14 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2005 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"
#include <libgnome/libgnome.h>

#include "menu-res-ops.h"
#include <glib.h> // for g_assert

using namespace Guikachu;
using namespace Guikachu::ResourceOps::MenuOps;

MenuHolder::MenuHolder (Resources::Menu   *menu) :
    manager (menu->get_manager ()),
    menu_serial (menu->get_serial ())
{
}

Resources::Menu * MenuHolder::get_menu () const
{
    Resource *res = manager->get_resource (menu_serial);
    g_assert (res);

    return static_cast<Resources::Menu*> (res);
}




SubmenuCreateOp::SubmenuCreateOp (Resources::Menu *menu,
				  unsigned int     index_) :
    MenuHolder (menu),
    index (index_),
    label (menu->get_submenus ()[index].label)
{
    const std::string &menu_id = menu->id;
    
    char *label_str = g_strdup_printf (_("Create submenu of %s"), menu_id.c_str ());
    op_label = label_str;
    g_free (label_str);
}

void SubmenuCreateOp::undo ()
{
    Resources::Menu           *res = get_menu ();
    Resources::Menu::MenuTree  submenus = res->get_submenus ();

    submenus.erase (submenus.begin () + index);

    res->set_submenus (submenus);
}

void SubmenuCreateOp::redo ()
{
    Resources::Menu           *res = get_menu ();
    Resources::Menu::MenuTree  submenus = res->get_submenus ();

    submenus.insert (submenus.begin () + index, Resources::Menu::Submenu (label));
    
    res->set_submenus (submenus);
    
}




SubmenuRemoveOp::SubmenuRemoveOp (Resources::Menu *menu,
				  unsigned int     index_) :
    MenuHolder (menu),
    index (index_),
    submenu (menu->get_submenus ()[index])
{
    const std::string &menu_id = menu->id;
    const std::string &submenu_id = submenu.label;
    
    char *label_str = g_strdup_printf (_("Remove \"%s\" from %s"), submenu_id.c_str (), menu_id.c_str ());
    op_label = label_str;
    g_free (label_str);
}

void SubmenuRemoveOp::undo ()
{
    Resources::Menu           *res = get_menu ();
    Resources::Menu::MenuTree  submenus = res->get_submenus ();

    submenus.insert (submenus.begin () + index, submenu);

    res->set_submenus (submenus);    
}

void SubmenuRemoveOp::redo ()
{
    Resources::Menu           *res = get_menu ();
    Resources::Menu::MenuTree  submenus = res->get_submenus ();

    submenus.erase (submenus.begin () + index);

    res->set_submenus (submenus);
}



SubmenuChangeOp::SubmenuChangeOp (Resources::Menu   *menu,
				  unsigned int       index_,
				  const std::string &new_label_) :
    MenuHolder (menu),
    index (index_),
    old_label (menu->get_submenus ()[index].label),
    new_label (new_label_)
{
    const std::string &menu_id = menu->id;
    
    char *label_str = g_strdup_printf (_("Rename \"%s\" in %s"), old_label.c_str (), menu_id.c_str ());
    op_label = label_str;
    g_free (label_str);
}

void SubmenuChangeOp::undo ()
{
    Resources::Menu           *res = get_menu ();
    Resources::Menu::MenuTree  submenus = res->get_submenus ();

    submenus[index].label = old_label;

    res->set_submenus (submenus);
}

void SubmenuChangeOp::redo ()
{
    Resources::Menu           *res = get_menu ();
    Resources::Menu::MenuTree  submenus = res->get_submenus ();

    submenus[index].label = new_label;

    res->set_submenus (submenus);
}

UndoOp * SubmenuChangeOp::combine (UndoOp *other_op) const
{
    SubmenuChangeOp *op = dynamic_cast<SubmenuChangeOp*> (other_op);
    if (!op)
	return 0;

    if (op->menu_serial != menu_serial || op->index != index)
	return 0;

    Resources::Menu *menu = get_menu ();
    SubmenuChangeOp *new_op = new SubmenuChangeOp (menu, index, op->new_label);
    new_op->old_label = old_label;

    char *label_str = g_strdup_printf (_("Rename \"%s\" in %s"),
				       old_label.c_str (), menu->id ().c_str ());
    new_op->op_label = label_str;
    g_free (label_str);

    return new_op;
}




SubmenuMoveOp::SubmenuMoveOp (Resources::Menu *menu,
			      index_t          old_index,
			      index_t          new_index) :
    MenuHolder (menu)
{
    index_history.push_back (old_index);
    index_history.push_back (new_index);
    
    Resources::Menu::MenuTree submenus = menu->get_submenus ();
    const std::string  submenu_label = submenus[old_index].label;
    const std::string &menu_id = menu->id;

    char *label_str = g_strdup_printf (_("Move \"%s\" in %s"),
				       submenu_label.c_str (), menu_id.c_str ());
    op_label = label_str;
    g_free (label_str);
}

SubmenuMoveOp::SubmenuMoveOp (Resources::Menu    *menu,
			      const std::string  &op_label_,
			      const index_list_t &index_history_head,
			      const index_list_t &index_history_tail) :
    MenuHolder (menu),
    index_history (index_history_head),
    op_label (op_label_)
{
    index_list_t::const_iterator tail_begin = index_history_tail.begin ();
    index_list_t::const_iterator tail_end = index_history_tail.end ();

    g_assert (index_history_tail.front () == index_history_head.back ());
    
    index_history.insert (index_history.end (), ++tail_begin, tail_end);
}

void SubmenuMoveOp::undo ()
{
    // Undo/redo may seem patently stupid and over-complicated, but
    // it's the only way to make cascading work
    
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    index_list_t::const_reverse_iterator rbegin = index_history.rbegin ();
    index_list_t::const_reverse_iterator rend = index_history.rend ();
    index_list_t::const_reverse_iterator curr, next;
    
    for (curr = rbegin, next = ++rbegin; next != rend; ++curr, ++next)
	std::iter_swap (submenus.begin () + *curr, submenus.begin () + *next);

    res->set_submenus (submenus);
}

void SubmenuMoveOp::redo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    index_list_t::const_iterator begin = index_history.begin ();
    index_list_t::const_iterator end = index_history.end ();
    index_list_t::const_iterator curr, next;
    
    for (curr = begin, next = ++begin; next != end; ++curr, ++next)
	std::iter_swap (submenus.begin () + *curr, submenus.begin () + *next);

    res->set_submenus (submenus);
}

UndoOp * SubmenuMoveOp::combine (UndoOp *other_op) const
{
    SubmenuMoveOp *op = dynamic_cast<SubmenuMoveOp*> (other_op);
    if (!op)
	return 0;

    if (op->menu_serial != menu_serial)
	return 0;

    if (op->index_history.front () != index_history.back ())
	return 0;

    Resources::Menu *menu = get_menu ();
    SubmenuMoveOp *new_op = new SubmenuMoveOp (menu, op_label, index_history, op->index_history);

    return new_op;
}




MenuItemCreateOp::MenuItemCreateOp (Resources::Menu *menu,
				    unsigned int     submenu_index_,
				    unsigned int     index_) :
    MenuHolder (menu),
    submenu_index (submenu_index_),
    index (index_)
{
    item_data = menu->get_submenus ()[submenu_index].items[index];

    const std::string &menu_id = menu->id;
    const std::string  menuitem_id = item_data.separator ? _("separator") : item_data.id;

    char *label_str = g_strdup_printf (_("Create %s in %s"), menu_id.c_str (), menuitem_id.c_str ());
    op_label = label_str;
    g_free (label_str);
}

void MenuItemCreateOp::undo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    std::vector<Resources::Menu::MenuItem> &items = submenus[submenu_index].items;
    items.erase (items.begin () + index);

    res->set_submenus (submenus);
}
    
void MenuItemCreateOp::redo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    std::vector<Resources::Menu::MenuItem> &items = submenus[submenu_index].items;
    items.insert (items.begin () + index, item_data);

    res->set_submenus (submenus);
}
    
MenuItemRemoveOp::MenuItemRemoveOp (Resources::Menu *menu,
				    unsigned int     submenu_index_,
				    unsigned int     index_) :
    MenuHolder (menu),
    submenu_index (submenu_index_),
    index (index_)
{
    item_data = menu->get_submenus ()[submenu_index].items[index];

    const std::string &menu_id = menu->id;
    const std::string  menuitem_id = item_data.separator ? _("separator") : item_data.id;

    char *label_str = g_strdup_printf (_("Remove %s from %s"), menuitem_id.c_str (), menu_id.c_str ());
    op_label = label_str;
    g_free (label_str);
}




void MenuItemRemoveOp::undo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    std::vector<Resources::Menu::MenuItem> &items = submenus[submenu_index].items;
    items.insert (items.begin () + index, item_data);

    res->set_submenus (submenus);
}
    
void MenuItemRemoveOp::redo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    std::vector<Resources::Menu::MenuItem> &items = submenus[submenu_index].items;
    items.erase (items.begin () + index);

    res->set_submenus (submenus);
}
    



MenuItemRenameOp::MenuItemRenameOp (Resources::Menu   *menu,
				    unsigned int       submenu_index_,
				    unsigned int       index_,
				    const std::string &new_id_) :
    MenuHolder (menu),
    submenu_index (submenu_index_),
    index (index_),
    old_id (menu->get_submenus ()[submenu_index].items[index].id),
    new_id (new_id_)
{
    char *label_str = g_strdup_printf (_("Rename %s to %s"), old_id.c_str (), new_id.c_str ());
    op_label = label_str;
    g_free (label_str);
}

void MenuItemRenameOp::undo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    submenus[submenu_index].items[index].id = old_id;

    res->set_submenus (submenus);
}

void MenuItemRenameOp::redo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    submenus[submenu_index].items[index].id = new_id;

    res->set_submenus (submenus);
}




MenuItemLabelOp::MenuItemLabelOp (Resources::Menu   *menu,
				  unsigned int       submenu_index_,
				  unsigned int       index_,
				  const std::string &new_label_) :
    MenuHolder (menu),
    submenu_index (submenu_index_),
    index (index_),
    old_label (menu->get_submenus ()[submenu_index].items[index].label),
    new_label (new_label_)
{
    std::string id = menu->get_submenus ()[submenu_index].items[index].id;
    
    char *label_str = g_strdup_printf (_("Change label of %s"), id.c_str ());
    op_label = label_str;
    g_free (label_str);
}

void MenuItemLabelOp::undo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    submenus[submenu_index].items[index].label = old_label;

    res->set_submenus (submenus);
}

void MenuItemLabelOp::redo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    submenus[submenu_index].items[index].label = new_label;

    res->set_submenus (submenus);
}

UndoOp * MenuItemLabelOp::combine (UndoOp *other_op) const
{
    MenuItemLabelOp *op = dynamic_cast<MenuItemLabelOp*> (other_op);
    if (!op)
	return 0;

    if (op->menu_serial != menu_serial ||
	op->submenu_index != submenu_index || op->index != index)
	return 0;

    MenuItemLabelOp *new_op = new MenuItemLabelOp (get_menu (), submenu_index, index, op->new_label);
    new_op->old_label = old_label;
    
    return new_op;
}




MenuItemShortcutOp::MenuItemShortcutOp (Resources::Menu *menu,
					unsigned int     submenu_index_,
					unsigned int     index_,
					char             new_shortcut_) :
    MenuHolder (menu),
    submenu_index (submenu_index_),
    index (index_),
    old_shortcut (menu->get_submenus ()[submenu_index].items[index].shortcut),
    new_shortcut (new_shortcut_)
{
    std::string id = menu->get_submenus ()[submenu_index].items[index].id;
    
    char *label_str = g_strdup_printf (_("Change shortcut of %s"), id.c_str ());
    op_label = label_str;
    g_free (label_str);
}

void MenuItemShortcutOp::undo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    submenus[submenu_index].items[index].shortcut = old_shortcut;

    res->set_submenus (submenus);
}

void MenuItemShortcutOp::redo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    submenus[submenu_index].items[index].shortcut = new_shortcut;

    res->set_submenus (submenus);
}

UndoOp * MenuItemShortcutOp::combine (UndoOp *other_op) const
{
    MenuItemShortcutOp *op = dynamic_cast<MenuItemShortcutOp*> (other_op);
    if (!op)
	return 0;

    if (op->menu_serial != menu_serial ||
	op->submenu_index != submenu_index || op->index != index)
	return 0;

    // Combine shortcut changes only if it's an erase-type sequence
    // (since otherwise every shortcut change would result in two undo ops)
    if (op->old_shortcut != 0)
	return 0;

    MenuItemShortcutOp *new_op = new MenuItemShortcutOp (
	get_menu (), submenu_index, index, op->new_shortcut);
    new_op->old_shortcut = old_shortcut;
    
    return new_op;
}




MenuItemMoveOp::MenuItemMoveOp (Resources::Menu *menu,
				index_t          submenu_index_,
				index_t          old_index,
				index_t          new_index) :
    MenuHolder (menu),
    submenu_index (submenu_index_)
{
    index_history.push_back (old_index);
    index_history.push_back (new_index);
    
    Resources::Menu::MenuTree submenus = menu->get_submenus ();
    const std::string  item_id = submenus[submenu_index].items[old_index].id;
    const std::string &menu_id = menu->id;

    char *label_str = g_strdup_printf (_("Move %s in %s"), item_id.c_str (), menu_id.c_str ());
    op_label = label_str;
    g_free (label_str);
}

MenuItemMoveOp::MenuItemMoveOp (Resources::Menu    *menu,
				index_t             submenu_index_,
				const std::string  &op_label_,
				const index_list_t &index_history_head,
				const index_list_t &index_history_tail) :
    MenuHolder (menu),
    submenu_index (submenu_index_),
    index_history (index_history_head),
    op_label (op_label_)
{
    index_list_t::const_iterator tail_begin = index_history_tail.begin ();
    index_list_t::const_iterator tail_end = index_history_tail.end ();

    g_assert (index_history_tail.front () == index_history_head.back ());
    
    index_history.insert (index_history.end (), ++tail_begin, tail_end);
}

void MenuItemMoveOp::undo ()
{
    // Undo/redo may seem patently stupid and over-complicated, but
    // it's the only way to make cascading work
    
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();
    
    index_list_t::const_reverse_iterator rbegin = index_history.rbegin ();
    index_list_t::const_reverse_iterator rend = index_history.rend ();
    index_list_t::const_reverse_iterator curr, next;
    
    for (curr = rbegin, next = ++rbegin; next != rend; ++curr, ++next)
	std::iter_swap (submenus[submenu_index].items.begin () + *curr,
			submenus[submenu_index].items.begin () + *next);

    res->set_submenus (submenus);
}

void MenuItemMoveOp::redo ()
{
    Resources::Menu *res = get_menu ();
    Resources::Menu::MenuTree submenus = res->get_submenus ();

    index_list_t::const_iterator begin = index_history.begin ();
    index_list_t::const_iterator end = index_history.end ();
    index_list_t::const_iterator curr, next;
    
    for (curr = begin, next = ++begin; next != end; ++curr, ++next)
	std::iter_swap (submenus[submenu_index].items.begin () + *curr,
			submenus[submenu_index].items.begin () + *next);

    res->set_submenus (submenus);
}

UndoOp * MenuItemMoveOp::combine (UndoOp *other_op) const
{
    MenuItemMoveOp *op = dynamic_cast<MenuItemMoveOp*> (other_op);
    if (!op)
	return 0;

    if (op->menu_serial != menu_serial)
	return 0;

    if (op->index_history.front () != index_history.back ())
	return 0;

    Resources::Menu *menu = get_menu ();
    MenuItemMoveOp *new_op = new MenuItemMoveOp (menu, submenu_index, op_label,
						 index_history, op->index_history);
    return new_op;
}
