//$Id: resource-tree.cc,v 1.18 2004/07/19 13:51:51 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2004 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"
#include <libgnome/libgnome.h>

#include "resource-tree.h"

#include "resource-factory.h"
#include "resource-manager-ops.h"
#include "edit-cut-and-paste.h"

#include "cellrenderer-icontext.h"
#include "cellrenderer-indent.h"

#include <libgnomeuimm/app-helper.h>
#include <libgnomeuimm/ui-items-stock.h>
#include <libgnomeuimm/app-helper.h>
#include <gtkmm/stock.h>

#include <sigc++/adaptors/bind.h>
#include <sigc++/adaptors/bind_return.h>

#include <algorithm>
#include <functional>

using namespace Guikachu::GUI;
using namespace Guikachu;

namespace
{
    class AppResetOp: public UndoOp
    {
        Resources::Application *res;
        std::string iconname, version, vendor;
        
    public:
        AppResetOp (Resources::Application *res_) :
            res (res_),
            iconname (res->iconname),
            version (res->version),
            vendor (res->vendor)
            {};
        
        Glib::ustring get_label () const { return _("Reset application settings"); };

        void undo () {
            res->iconname = iconname;
            res->version = version;
            res->vendor = vendor;
        };
        
        void redo () { res->reset (); };
    };
} // Anonymous namespace

ResourceTree::ResourceTree ():
    manager (0),
    app_win (0),
    popup_menu (0)
{
    Gtk::TreeModel::ColumnRecord cols;
    cols.add (col_data);
    
    treestore = Gtk::TreeStore::create (cols);
    set_model (treestore);

    set_headers_visible (false);
    
    CellRendererIndent<CellRendererIconText>::setup_view (
        *this, "", sigc::mem_fun (*this, &ResourceTree::cell_label_cb));
    
    /* Connect to events */
    signal_button_press_event ().connect_notify (sigc::mem_fun (*this, &ResourceTree::button_press_cb));
    signal_key_press_event ().connect (sigc::bind_return (sigc::mem_fun (*this, &ResourceTree::key_press_cb), true));

    add_events (Gdk::BUTTON_PRESS_MASK | Gdk::KEY_PRESS_MASK);

    signal_row_activated ().connect (sigc::mem_fun (*this, &ResourceTree::row_activated_cb));    
}

ResourceTree::~ResourceTree ()
{
    delete popup_menu;
    delete app_win;
}

void ResourceTree::cell_label_cb (Gtk::CellRenderer *cell, const Gtk::TreeModel::iterator &iter) const
{
    const TreeData &td = (*iter)[col_data];
    td.apply_to_cell (cell);
}

Guikachu::Resource * ResourceTree::get_selected ()
{
    Glib::RefPtr<Gtk::TreeSelection> tree_selection = get_selection ();
    Gtk::TreeModel::iterator iter = tree_selection->get_selected ();

    if (!iter)
        return 0;

    const TreeData &td = (*iter)[col_data];
    return td.get_resource ();
}

void ResourceTree::set_manager (ResourceManager *manager_)
{
    delete app_win;
    app_win = 0;
    
    manager = manager_;

    treestore->clear ();
    subtrees.clear ();

    root_tree = *(treestore->append ());
    root_tree[col_data] = manager->get_application ();

    manager->get_application ()->changed.connect (
	sigc::mem_fun (*this, &ResourceTree::app_changed_cb));

    create_tree_for_type (Resources::RESOURCE_FORM);
    create_tree_for_type (Resources::RESOURCE_MENU);
    create_tree_for_type (Resources::RESOURCE_BITMAP);
    create_tree_for_type (Resources::RESOURCE_STRING);
    create_tree_for_type (Resources::RESOURCE_BLOB);
    
    manager->resource_created.connect (
	sigc::mem_fun (*this, &ResourceTree::resource_created_cb));
    manager->resource_removed.connect (
	sigc::mem_fun (*this, &ResourceTree::resource_removed_cb));
    
    // Create tree items for existing resources
    const std::set<Resource*> resources = manager->get_resources ();
    std::for_each (resources.begin (), resources.end (),
		   sigc::mem_fun (*this, &ResourceTree::resource_created_cb));

    expand_all ();
}

void ResourceTree::create_tree_for_type (Resources::Type type)
{
    if (subtrees.find (type) != subtrees.end ())
        return;

    Gtk::TreeRow row = *(treestore->append (root_tree.children ()));
    row[col_data] = type;

    subtrees[type] = row;

    switch (type)
    {
    case Resources::RESOURCE_FORM:
    case Resources::RESOURCE_DIALOG:
        subtrees[Resources::RESOURCE_FORM] = subtrees[Resources::RESOURCE_DIALOG] = row;
        break;

    case Resources::RESOURCE_STRING:
    case Resources::RESOURCE_STRINGLIST:
        subtrees[Resources::RESOURCE_STRING] = subtrees[Resources::RESOURCE_STRINGLIST] = row;
        break;

    case Resources::RESOURCE_BITMAP:
    case Resources::RESOURCE_BITMAPFAMILY:
        subtrees[Resources::RESOURCE_BITMAP] = subtrees[Resources::RESOURCE_BITMAPFAMILY] = row;
        break;

    default:
        break;
    }
}

void ResourceTree::app_changed_cb ()
{
    treestore->row_changed (treestore->get_path (root_tree), root_tree);
}

void ResourceTree::resource_removed_cb (Resource *res)
{
    row_map_t::iterator row_found = row_map.find (res);
    g_return_if_fail (row_found != row_map.end ());

    treestore->erase (row_found->second);
    row_map.erase (row_found);
}

void ResourceTree::resource_created_cb (Resource *res)
{
    create_tree_for_type (res->get_type ());
    Gtk::TreeRow &subtree_row = subtrees[res->get_type ()];

    Gtk::TreeRow row = *(treestore->append (subtree_row.children ()));
    row[col_data] = res;

    row_map[res] = row;
    
    res->changed.connect (sigc::bind (
	sigc::mem_fun (*this, &ResourceTree::resource_changed_cb), row));
    
    expand_row (treestore->get_path (subtree_row), true);
}

void ResourceTree::resource_changed_cb (Gtk::TreeRow row)
{
    treestore->row_changed (treestore->get_path (row), row);
}

void ResourceTree::show_app_win ()
{
    if (!app_win)
        app_win = new AppWindow (manager->get_application ());

    app_win->show ();
}

void ResourceTree::show_res_win (Resource *res)
{
    ResourceWindow *res_win = get_resource_editor (res);
    
    if (res_win)
        res_win->show ();
}

void ResourceTree::row_activated_cb (const Gtk::TreeModel::Path &path,
				     Gtk::TreeView::Column      *col)
{
    Gtk::TreeModel::iterator iter = treestore->get_iter (path);
    if (iter == treestore->children ().begin ())
    {
        show_app_win ();
	return;
    }

    const TreeData &td = (*iter)[col_data];
    Resource *res = td.get_resource ();
    if (res)
        show_res_win (res);
}

void ResourceTree::button_press_cb (GdkEventButton *e)
{
    if (e->button != 3)
	return;

    int cell_x, cell_y;
    Gtk::TreeModel::Path  path;
    Gtk::TreeViewColumn  *column;
    Guikachu::Resource   *resource = 0;

    // FIXME: e->x and e->y are doubles
    if (get_path_at_pos (e->x, e->y, path, column, cell_x, cell_y))
    {
	Gtk::TreeModel::iterator iter = treestore->get_iter (path);
        const TreeData &td = (*iter)[col_data];
    
        if (iter == treestore->children ().begin ())
            popup_app_menu (e->button, e->time);
        else
            resource = td.get_resource ();
    }
    
    if (resource)
        popup_res_menu (e->button, e->time, resource);
}

void ResourceTree::key_press_cb (GdkEventKey *e)
{
    Gtk::TreeModel::iterator iter = get_selection ()->get_selected ();
    if (!iter)
        return;

    const TreeData &td = (*iter)[col_data];
    Resource *res = td.get_resource ();
    if (!res && iter != treestore->children ().begin ())
        return;

    switch (e->keyval)
    {
    case GDK_Return:
    case GDK_KP_Enter:
        if (res)
            show_res_win (res);
        else
            show_app_win ();
        break;
        
    case GDK_Delete:
    case GDK_KP_Delete:
        if (res)
            res_remove_cb (res);
        else
            app_reset_cb ();
        break;

    case GDK_Menu:
        if (res)
            popup_res_menu (0, e->time, res);
        else
            popup_app_menu (0, e->time);
        break;
        
    default:
        break;
    }
}

void ResourceTree::popup_app_menu (guint button, guint32 time)
{
    if (popup_menu)
        delete popup_menu;

    popup_menu = new Gtk::Menu;
    std::list <Gnome::UI::Items::Info> popup_menu_list;
    
    popup_menu_list.push_back (
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::PROPERTIES),
				_("_Edit"),
				sigc::mem_fun (*this, &ResourceTree::show_app_win)));

    Gnome::UI::Items::fill (*popup_menu, popup_menu_list,
			    popup_menu->get_accel_group ());
    
    popup_menu->popup (button, time);
}

void ResourceTree::app_reset_cb ()
{
    Resources::Application *app = manager->get_application ();
    manager->get_undo_manager ().push (new AppResetOp (app));
    app->reset ();
    if (app_win)
        app_win->hide ();

}

void ResourceTree::popup_res_menu (guint button, guint32 time, Resource *res)
{
    if (popup_menu)
        delete popup_menu;

    popup_menu = new Gtk::Menu;
    
    Gnome::UI::Items::Info popup_menu_list[] = {
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::PROPERTIES),
				_("_Edit"),
				sigc::bind (sigc::mem_fun (*this, &ResourceTree::show_res_win), res)),
	Gnome::UI::Items::Separator (),
	Gnome::UI::MenuItems::Cut   (sigc::bind (sigc::mem_fun (*this, &ResourceTree::res_cut_cb), res)),
	Gnome::UI::MenuItems::Copy  (sigc::bind (sigc::mem_fun (*this, &ResourceTree::res_copy_cb), res)),
	Gnome::UI::Items::Separator (),
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::DELETE),
				_("_Remove"),
				sigc::bind (sigc::mem_fun (*this, &ResourceTree::res_remove_cb), res)),
    };

    Gnome::UI::Items::fill (*popup_menu, popup_menu_list,
			    popup_menu->get_accel_group ());
    
    popup_menu->popup (button, time);
}

void ResourceTree::res_cut_cb (Resource *res)
{
    res_copy_cb (res);
    res_remove_cb (res);
}

void ResourceTree::res_copy_cb (Resource *res)
{
    Edit::copy_resource (res);
}

void ResourceTree::res_remove_cb (Resource *res)
{
    manager->get_undo_manager ().push (new ResourceOps::RemoveOp (res));
    manager->remove_resource (res);
}
