//$Id: io.cc,v 1.10 2004/08/20 18:22:17 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2004 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "io/io.h"

#include "io/xml-loader.h"
#include "io/xml-saver.h"
#include "io/rcp-loader.h"
#include "io/rcp-saver.h"

#include <libgnomevfsmm/handle.h>
#include <libgnomevfsmm/uri.h>
#include <libgnomevfsmm/utils.h>

using namespace Guikachu;

Glib::ustring IO::create_canonical_uri (const Glib::ustring &filename)
{
    if (filename == "")
	return "";

    if (filename.find ("://") < filename.find ("/"))
	// Don't touch URI's
	return filename;
    
    Glib::ustring full_filename;
    if (g_path_is_absolute (filename.c_str ()))
	full_filename = filename;
    else {
	char *current_dirname = g_get_current_dir ();
	full_filename = Glib::ustring (current_dirname) + "/" + filename;
	g_free (current_dirname);
    }

    return Gnome::Vfs::make_uri_canonical (full_filename);
}

Glib::ustring IO::get_mime_type (const Glib::ustring &uri) throw (Glib::Exception)
{
    Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri = Gnome::Vfs::Uri::create (uri);
    Glib::RefPtr<Gnome::Vfs::FileInfo> file_info = vfs_uri->get_file_info (Gnome::Vfs::FILE_INFO_GET_MIME_TYPE);
    return file_info->get_mime_type ();
}

void IO::load_uri (const Glib::ustring &uri, char *&data, Gnome::Vfs::FileSize &len) throw (Glib::Exception)
{
    Gnome::Vfs::Handle f;
    f.open (uri, Gnome::Vfs::OPEN_READ);

    Glib::RefPtr<Gnome::Vfs::FileInfo> vfs_info = f.get_file_info (Gnome::Vfs::FILE_INFO_FOLLOW_LINKS);
        
    data = new char[vfs_info->get_size ()];
    len = 0;        
    char                 buffer[10240];
    Gnome::Vfs::FileSize bytes_read;

    do {
        bytes_read = f.read (buffer, sizeof buffer - 1);
        if (bytes_read)
            memcpy (data + len, buffer, bytes_read);
	len += bytes_read;
    } while (bytes_read);
}

void IO::save_uri (const Glib::ustring &uri, const char *data, Gnome::Vfs::FileSize len) throw (Glib::Exception)
{
    // Create file
    Gnome::Vfs::Handle f;
    f.create (uri, Gnome::Vfs::OPEN_WRITE, 0, 06444);
	
    // Write data to VFS stream
    const Gnome::Vfs::FileSize chunk_size = len;
    
    for (Gnome::Vfs::FileSize bytes_written = 0; len; data += bytes_written)
    {
        bytes_written = f.write (data, std::min (chunk_size, len));
        len -= bytes_written;
    }
}

IO::IOFactory * IO::IOFactory::instance_ = 0;

namespace
{
    template<class T>
    IO::Loader * loader_factory ()
    {
	return new T;
    }

    template<class T>
    IO::Saver * saver_factory ()
    {
	return new T;
    }
    
} // anonymous namespace

IO::IOFactory::IOFactory ()
{
    loader_map["application/x-guikachu"] = loader_factory<IO::XMLLoader>;
    loader_map["text/x-rcp"] = loader_factory<IO::RCPLoader>;

    saver_map["application/x-guikachu"] = saver_factory<IO::XMLSaver>;
    saver_map["text/x-rcp"] = saver_factory<IO::RCPSaver>;
}

IO::IOFactory * IO::IOFactory::instance ()
{
    if (!instance_)
	instance_ = new IOFactory;
	
    return instance_;
}

IO::Loader * IO::IOFactory::create_loader (const std::string &mime_type) const
{
    loader_map_t::const_iterator found = loader_map.find (mime_type);
    if (found == loader_map.end ())
	return 0;

    return found->second ();
}

IO::Saver * IO::IOFactory::create_saver (const std::string &mime_type) const
{
    saver_map_t::const_iterator found = saver_map.find (mime_type);
    if (found == saver_map.end ())
	return 0;

    return found->second ();
}
