//$Id: mainwin.cc,v 1.53 2004/07/12 22:07:57 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2004 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"
#include <libgnome/libgnome.h>

#include "mainwin.h"
#include "ui-gui.h"

#include <gtkmm/statusbar.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/stock.h>

#include <list>

#include "resource-manager.h"

#include "add-resource.h"
#include "preferences-win.h"
#include "preferences.h"
#include "session.h"

#include <sigc++/adaptors/bind.h>

using namespace Guikachu;

namespace {

    Glib::ustring escape_underscores (const Glib::ustring &text);
    
} // Anonymous namespace

GUI::MainWin::MainWin ():
    Gnome::UI::App ("Guikachu"),
    manager (0),
    uri (""),
    in_exit (false),
    recent_files_num (0)   
{
    /* File menu */
    Gnome::UI::Items::Info m_file[] = {
	Gnome::UI::MenuItems::New    (_("_New project"), _("Create a new project"),
				      sigc::mem_fun (*this, &MainWin::new_cb)),
	
	Gnome::UI::MenuItems::Open   (sigc::mem_fun (*this, &MainWin::load_cb)),
        Gnome::UI::Items::Separator  (),
	Gnome::UI::MenuItems::Save   (sigc::mem_fun (*this, &MainWin::save_cb)),
	Gnome::UI::MenuItems::SaveAs (sigc::mem_fun (*this, &MainWin::save_as_cb)),
	Gnome::UI::Items::Item       (Gnome::UI::Items::Icon (Gtk::Stock::CONVERT),
				      _("Export _RCP..."),
				      sigc::mem_fun (*this, &MainWin::export_cb),
				      _("Create an RCP file for compilation")),
	Gnome::UI::Items::Separator  (),
	Gnome::UI::Items::Item       (Gnome::UI::Items::Icon (Gtk::Stock::ADD),
				      _("Add _Resource..."),
				      sigc::mem_fun (*this, &MainWin::add_cb),
				      _("Add new component to current project")),
	Gnome::UI::Items::Separator  (),
	Gnome::UI::MenuItems::Exit   (sigc::mem_fun (*this, &MainWin::exit_cb))
    };
    
    /* Edit menu */
    Gnome::UI::Items::Info m_edit[] = {
        Gnome::UI::Items::Item       (Gnome::UI::Items::Icon (Gtk::Stock::UNDO),
				      "Undo placeholder"),
        Gnome::UI::Items::Item       (Gnome::UI::Items::Icon (Gtk::Stock::REDO),
				      "Redo placeholder"),
	Gnome::UI::Items::Separator  (),
	Gnome::UI::MenuItems::Cut    (sigc::mem_fun (*this, &MainWin::cut_cb)),
	Gnome::UI::MenuItems::Copy   (sigc::mem_fun (*this, &MainWin::copy_cb)),
	Gnome::UI::MenuItems::Paste  (sigc::mem_fun (*this, &MainWin::paste_cb)),
    };
    
    /* Settings menu */
    Gnome::UI::Items::Info m_settings[] = {
	Gnome::UI::MenuItems::Preferences (sigc::mem_fun (*this, &MainWin::preferences_cb))
    };
    
    /* Help menu */
    Gnome::UI::Items::Info m_help[] = {
	Gnome::UI::MenuItems::About (sigc::ptr_fun (&UI::show_about))
    };

    /* Create the menu bar structure */
    Gnome::UI::Items::SubTree menubar[] = {
	Gnome::UI::Menus::File     (m_file),
	Gnome::UI::Menus::Edit     (m_edit),
	Gnome::UI::Menus::Settings (m_settings),
	Gnome::UI::Menus::Help     (m_help)
    };    
    create_menus (menubar);

    
    /* Create status bar */
    set_statusbar (*manage (new Gtk::Statusbar));
    install_menu_hints ();

    
    /* Create the toolbar */
    Gnome::UI::Items::Info toolbar [] = {
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::ADD),
				_("Add"),
				sigc::mem_fun (*this, &MainWin::add_cb),
				_("Add new component to current project")),
	Gnome::UI::Items::Separator (),
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::NEW),
				_("New"),
				sigc::mem_fun (*this, &MainWin::new_cb),
				_("New project")),
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::OPEN),
				_("Open"),
				sigc::mem_fun (*this, &MainWin::load_cb),
				_("Open project")),
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::SAVE),
				_("Save"),
				sigc::mem_fun (*this, &MainWin::save_cb),
				_("Save project"))
    };
    create_toolbar (toolbar);

    // Listen for changes in recent file list
    Preferences::Interface::recent_files_changed.connect (
	sigc::mem_fun (*this, &MainWin::update_recent_files));
    update_recent_files ();

    // Scrollable viewport
    Gtk::ScrolledWindow *scrollwin = new Gtk::ScrolledWindow ();
    scrollwin->set_policy (Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
    scrollwin->add (resource_tree);
    scrollwin->show_all ();
    set_contents (*manage (scrollwin));
    set_size_request (300, 300);
}

void GUI::MainWin::set_manager (ResourceManager *manager_)
{
    manager = manager_;

    update_title ();

    resource_tree.set_manager (manager);

    // Listen for changes in the undo stack
    manager->get_undo_manager ().changed.connect (sigc::mem_fun (*this, &MainWin::update_undo));
    update_undo ();

    // Listen for changes in dirty state
    manager->dirty_state_changed.connect (sigc::mem_fun (*this, &MainWin::update_title));
}

void GUI::MainWin::update_recent_files ()
{
    std::string insertion_file = dgettext ("libgnomeui-2.0", "_File");

    // Clear old entries
    if (recent_files_num)
	remove_menu_range (insertion_file + '/', 8, recent_files_num + 1);

    Preferences::Interface::RecentFiles recent_files =
	Preferences::Interface::get_recent_files ();
    recent_files_num = recent_files.size ();

    if (recent_files_num)
    {
	std::list<Gnome::UI::Items::Info> recent_items;
	recent_items.push_back (Gnome::UI::Items::Separator ());

	Preferences::Interface::RecentFiles::const_iterator i;
	int num;
	for (i = recent_files.begin (), num = 0;
	     i != recent_files.end (); i++, num++)
	{
	    Glib::ustring vis_filename = escape_underscores (UI::visible_filename (*i));
	    Glib::ustring menu_label;
	    
	    if (num < 9)
		menu_label = Glib::ScopedPtr<char> (
		    g_strdup_printf ("_%d %s", num + 1, vis_filename.c_str ())).get ();
	    else
		menu_label = Glib::ScopedPtr<char> (
		    g_strdup_printf ("%d %s", num + 1, vis_filename.c_str ())).get ();
	    
	    recent_items.push_back (
		Gnome::UI::Items::Item (menu_label, sigc::bind (
                    sigc::mem_fun (*this, &MainWin::open), *i)));
	}

        std::string insertion_res = _("Add _Resource...");
        insert_menus (insertion_file + '/' + insertion_res, recent_items);
    }
}

void GUI::MainWin::update_undo ()
{
    UndoManager &undo_manager = manager->get_undo_manager ();
    
    std::list<Glib::ustring> undo_labels = undo_manager.get_undo_labels ();
    std::list<Glib::ustring> redo_labels = undo_manager.get_redo_labels ();
    
    std::string top_undo_label = undo_labels.size () ?
	escape_underscores (undo_labels.front ()) : _("No operation");
    std::string top_redo_label = redo_labels.size () ?
	escape_underscores (redo_labels.front ()) : _("No operation");
	
    Glib::ScopedPtr<char> undo_label (
	g_strdup_printf (_("_Undo: %s"), top_undo_label.c_str ()));
    Glib::ScopedPtr<char> redo_label (
	g_strdup_printf (_("_Redo: %s"), top_redo_label.c_str ()));
    
    std::string edit_str = dgettext ("libgnomeui-2.0", "_Edit");
    edit_str += '/';
    remove_menu_range (edit_str, 0, 2);
    
    Gnome::UI::Items::Info undo_items[] = {
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::UNDO), undo_label.get (),
				sigc::mem_fun (*this, &MainWin::undo_cb)),
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::REDO), redo_label.get (),
				sigc::mem_fun (*this, &MainWin::redo_cb)),
    };

    Gnome::UI::Items::Array<Gnome::UI::Items::Info> menubar_widgets =
	insert_menus (edit_str, undo_items);

    // Disable menu items if undo and/or redo stack is empty
    menubar_widgets[0].get_widget ()->set_sensitive (undo_labels.size ());
    menubar_widgets[1].get_widget ()->set_sensitive (redo_labels.size ());
}

void GUI::MainWin::undo_cb ()
{
    manager->get_undo_manager ().undo ();
}

void GUI::MainWin::redo_cb ()
{
    manager->get_undo_manager ().redo ();
}

void GUI::MainWin::add_cb ()
{
    AddResourceWin add_win (manager);

    add_win.run (this);
}

void GUI::MainWin::preferences_cb ()
{
    static PreferencesWin pref_win;

    pref_win.run ();
}

void GUI::MainWin::update_title ()
{
    gchar *title_buf;

    if (uri != "")
	title_buf = g_strdup_printf (_("Guikachu: %s%c"),
				     UI::visible_filename (uri).c_str (),
				     manager->is_dirty () ? '*' : '\0');
    else
	title_buf = g_strdup_printf (_("Guikachu: %s%c"), _("<unnamed>"),
				     manager->is_dirty () ? '*' : '\0');
	
    set_title (title_buf);

    g_free (title_buf);
}

namespace {

Glib::ustring escape_underscores (const Glib::ustring &text)
{
    // Change _ to __ to avoid underscore accelerators

    Glib::ustring escaped_text;
    escaped_text.reserve (text.length () * 2);
    
    for (Glib::ustring::const_iterator i = text.begin (); i != text.end (); ++i)
	if (*i == '_')
	    escaped_text += "__";
	else
	    escaped_text += *i;
	    
    return escaped_text;
}
    
} // Anonymous namespace
