//$Id: menu-win.cc,v 1.42 2004/07/08 11:27:49 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2004 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "menu-win.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include "property-ops-resource.h"
#include "menu-res-ops.h"

#include <sigc++/adaptors/hide.h>
#include <sigc++/adaptors/bind.h>
#include <sigc++/adaptors/bind_return.h>

#include "ui.h"
#include "property-ops.h"

#include "cellrenderer-indent.h"

using namespace Guikachu;
using namespace Guikachu::ResourceOps::MenuOps;

GUI::MenuWindow::MenuWindow (Guikachu::Resources::Menu *res_):
    res (res_),
    gui (Gnome::Glade::Xml::create (GUIKACHU_GLADEDIR "/guikachu.glade", "res_menu")),
    current_entry (0),
    update_block (0)
{
    gui->get_widget ("res_menu", window);
    window->signal_delete_event ().connect (sigc::bind_return (
        sigc::mem_fun (*this, &MenuWindow::delete_event_impl), true));    
    
    gui->get_widget ("menu_id", id_entry);

    // Menu items list
    Gtk::TreeModel::ColumnRecord cols;
    cols.add (col_is_submenu);
    cols.add (col_index);
    cols.add (col_parent_index);
    treestore = Gtk::ListStore::create (cols);

    gui->get_widget ("menu_tree", treeview);
    treeview->set_model (treestore);

    CellRendererIndent<Gtk::CellRendererText> *indent_renderer = new CellRendererIndent<Gtk::CellRendererText>;
    Gtk::CellRendererText *label_renderer = static_cast<Gtk::CellRendererText*> (indent_renderer->get_real_cell ());
    Gtk::TreeView::Column *label_col = new Gtk::TreeView::Column (_("Label"), *manage (indent_renderer));
    label_col->set_cell_data_func (*indent_renderer, sigc::mem_fun (*this, &MenuWindow::cell_label_cb));
    label_renderer->signal_edited ().connect (sigc::mem_fun (*this, &MenuWindow::cell_label_edit_cb));
    treeview->append_column (*manage (label_col));

    Gtk::CellRendererText *id_renderer = new Gtk::CellRendererText;
    Gtk::TreeView::Column *id_col = new Gtk::TreeView::Column (_("ID"), *manage (id_renderer));
    id_col->set_cell_data_func (*id_renderer, sigc::mem_fun (*this, &MenuWindow::cell_id_cb));
    id_renderer->signal_edited ().connect (sigc::mem_fun (*this, &MenuWindow::cell_id_edit_cb));
    treeview->append_column (*manage (id_col));

    Gtk::CellRendererText *shortcut_renderer = new Gtk::CellRendererText;
    Gtk::TreeView::Column *shortcut_col = new Gtk::TreeView::Column (_("Shortcut"), *manage (shortcut_renderer));
    shortcut_col->set_cell_data_func (*shortcut_renderer, sigc::mem_fun (*this, &MenuWindow::cell_shortcut_cb));
    shortcut_renderer->signal_edited ().connect (sigc::mem_fun (*this, &MenuWindow::cell_shortcut_edit_cb));
    treeview->append_column (*manage (shortcut_col));
    
    treeview->get_selection ()->set_mode (Gtk::SELECTION_BROWSE);
    treeview->get_selection ()->signal_changed ().connect (
	sigc::mem_fun (*this, &MenuWindow::selection_changed_cb));
    
    gui->get_widget ("menu_btn_up", btnUp);
    gui->get_widget ("menu_btn_down", btnDown);
    gui->get_widget ("menu_btn_remove", btnRemove);

    // New item buttons
    gui->get_widget ("menu_btn_submenu", btnSubmenu);
    gui->get_widget ("menu_btn_item", btnItem);
    gui->get_widget ("menu_btn_separator", btnSeparator);

    // Menu item properties
    gui->get_widget ("menuitem_label", item_label);
    gui->get_widget ("menuitem_label_label", lbl_item_label);
    gui->get_widget ("menuitem_id", item_id);
    gui->get_widget ("menuitem_id_label", lbl_item_id);
    gui->get_widget ("menuitem_shortcut", item_shortcut);
    gui->get_widget ("menuitem_shortcut_label", lbl_item_shortcut);

    id_entry->signal_activate ().connect (sigc::mem_fun (*this, &MenuWindow::id_entry_cb));
    id_entry->signal_focus_out_event ().connect (
	sigc::hide (sigc::bind_return (sigc::mem_fun (*this, &MenuWindow::id_entry_cb), false)));
    
    item_id->signal_activate ().connect (sigc::mem_fun (*this, &MenuWindow::item_id_entry_cb));
    item_id->signal_focus_out_event ().connect (
        sigc::hide (sigc::bind_return (sigc::mem_fun (*this, &MenuWindow::item_id_entry_cb), false)));

    item_label->signal_changed ().connect (sigc::mem_fun (*this, &MenuWindow::item_label_entry_cb));
    item_label->signal_insert_text ().connect_notify (
        sigc::bind (sigc::mem_fun (*this, &MenuWindow::entry_insert_cb), item_label));
    item_label->signal_delete_text ().connect_notify (
        sigc::bind (sigc::mem_fun (*this, &MenuWindow::entry_delete_cb), item_label));

    item_shortcut->signal_changed ().connect (sigc::mem_fun (*this, &MenuWindow::item_shortcut_entry_cb));
    item_shortcut->signal_insert_text ().connect_notify (
        sigc::bind (sigc::mem_fun (*this, &MenuWindow::entry_insert_cb), item_shortcut));
    item_shortcut->signal_delete_text ().connect_notify (
        sigc::bind (sigc::mem_fun (*this, &MenuWindow::entry_delete_cb), item_shortcut));

    btnUp->signal_clicked ().connect (sigc::mem_fun (*this, &MenuWindow::btn_up_cb));
    btnDown->signal_clicked ().connect (sigc::mem_fun (*this, &MenuWindow::btn_down_cb));
    btnRemove->signal_clicked ().connect (sigc::mem_fun (*this, &MenuWindow::btn_remove_cb));

    btnSubmenu->signal_clicked ().connect (sigc::mem_fun (*this, &MenuWindow::btn_submenu_cb));
    btnItem->signal_clicked ().connect (sigc::bind (sigc::mem_fun (*this, &MenuWindow::btn_item_cb), false));
    btnSeparator->signal_clicked ().connect (sigc::bind (sigc::mem_fun (*this, &MenuWindow::btn_item_cb), true));
    
    res->changed.connect (sigc::mem_fun (*this, &MenuWindow::update));
    update ();
}

GUI::MenuWindow::~MenuWindow ()
{
    delete window;
}

void GUI::MenuWindow::cell_label_cb (Gtk::CellRenderer *cell, const Gtk::TreeModel::iterator &iter) const
{
    typedef Gtk::CellRendererText text_cell_t;
    typedef CellRendererIndent<text_cell_t> indent_cell_t;
    
    indent_cell_t *cell_indent = dynamic_cast<indent_cell_t*> (cell);
    g_return_if_fail (cell_indent);
    
    text_cell_t *cell_text = dynamic_cast<text_cell_t*> (cell_indent->get_real_cell ());
    g_return_if_fail (cell_text);

    if ((*iter)[col_is_submenu])
    {
        const Resources::Menu::Submenu &submenu = submenus[(*iter)[col_index]];

        cell_indent->set_depth (0);
        cell_text->property_text () = submenu.label;
        cell_text->property_weight () = Pango::WEIGHT_BOLD;
        cell_text->property_editable () = true;
    } else {
        const Resources::Menu::MenuItem &item = submenus[(*iter)[col_parent_index]].items[(*iter)[col_index]];

        cell_indent->set_depth (1);
        cell_text->property_weight () = Pango::WEIGHT_NORMAL;

        if (item.separator)
        {
            cell_text->property_text () = "------";
            cell_text->property_editable () = false;
        } else {
            cell_text->property_text () = item.label;
            cell_text->property_editable () = true;
        }
    }
}

void GUI::MenuWindow::cell_id_cb (Gtk::CellRenderer *cell, const Gtk::TreeModel::iterator &iter) const
{
    Gtk::CellRendererText *cell_id = dynamic_cast<Gtk::CellRendererText*> (cell);
    g_return_if_fail (cell_id);

    if ((*iter)[col_is_submenu])
    {
        cell_id->property_text () = "";
        cell_id->property_editable () = false;
    } else {
        const Resources::Menu::MenuItem &item = submenus[(*iter)[col_parent_index]].items[(*iter)[col_index]];
        cell_id->property_text () = item.separator ? "" : item.id;
        cell_id->property_editable () = !item.separator;
    }
}

void GUI::MenuWindow::cell_shortcut_cb (Gtk::CellRenderer *cell, const Gtk::TreeModel::iterator &iter) const
{
    Gtk::CellRendererText *cell_shortcut = dynamic_cast<Gtk::CellRendererText*> (cell);
    g_return_if_fail (cell_shortcut);

    cell_shortcut->property_text () = "";
    cell_shortcut->property_editable () = false;
    
    if ((*iter)[col_is_submenu])
        return;

    const Resources::Menu::MenuItem &item = submenus[(*iter)[col_parent_index]].items[(*iter)[col_index]];
    cell_shortcut->property_editable () = !item.separator;
    
    if (!item.separator && item.shortcut)
    {
        Glib::ScopedPtr<char> shortcut_str (g_strdup_printf ("%c", item.shortcut));
        cell_shortcut->property_text () = shortcut_str.get ();
    }
}

void GUI::MenuWindow::cell_label_edit_cb (const Glib::ustring &path, const Glib::ustring &new_text)
{
    current_entry = 0;
    
    Gtk::TreeModel::iterator iter = treestore->get_iter (path);    
    set_item_label (iter, new_text);
}

void GUI::MenuWindow::cell_id_edit_cb (const Glib::ustring &path, const Glib::ustring &new_text)
{
    current_entry = 0;
    
    Gtk::TreeModel::iterator iter = treestore->get_iter (path);    
    set_item_id (iter, new_text);
}

void GUI::MenuWindow::cell_shortcut_edit_cb (const Glib::ustring &path, const Glib::ustring &new_text)
{
    current_entry = 0;
    
    Gtk::TreeModel::iterator iter = treestore->get_iter (path);    
    set_item_shortcut (iter, new_text);
}

void GUI::MenuWindow::delete_event_impl (GdkEventAny *e)
{
    window->hide ();
}

void GUI::MenuWindow::show ()
{
    window->show_all ();
    window->raise ();
}

void GUI::MenuWindow::update ()
{
    ++update_block;
    
    gchar *title_buf = g_strdup_printf (_("Menu: %s"), res->id ().c_str ());
    window->set_title (title_buf);
    g_free (title_buf);

    id_entry->set_text (Glib::ustring (res->id));

    submenus = res->get_submenus ();
    unsigned int submenu_num = 0;

    Gtk::TreeModel::Path store_path ("0");
    for (Resources::Menu::MenuTree::const_iterator i = submenus.begin ();
	 i != submenus.end (); ++i, ++submenu_num)
    {
        Gtk::TreeModel::iterator store_iter = treestore->get_iter (store_path);
        if (!store_iter)
            store_iter = treestore->append ();
        Gtk::TreeRow row (*store_iter);

	row[col_is_submenu] = true;
	row[col_index] = submenu_num;

        ++store_path[0];
        
        for (unsigned int j = 0; j < i->items.size (); ++j)
	{
            Gtk::TreeModel::iterator store_iter = treestore->get_iter (store_path);
            if (!store_iter)
                store_iter = treestore->append ();
            row = *store_iter;

	    row[col_is_submenu] = false;
	    row[col_index] = j;
	    row[col_parent_index] = submenu_num;

            ++store_path[0];
	}
    }
    
    // Remove excessive treestore rows
    for (Gtk::TreeModel::iterator store_iter = treestore->get_iter (store_path); store_iter;
         store_iter = treestore->get_iter (store_path))
        treestore->erase (store_iter);

    selection_changed_cb ();

    if (current_entry)
    {
        current_entry->grab_focus ();
        current_entry->select_region (0, 0);
        current_entry->set_position (current_entry_pos);
    }
    
    --update_block;
}

void GUI::MenuWindow::reset_controls ()
{
    ++update_block;
    
    btnDown->set_sensitive (false);
    btnRemove->set_sensitive (false);
    btnUp->set_sensitive (false);

    btnItem->set_sensitive (false);
    btnSeparator->set_sensitive (false);

    item_label->set_text ("");
    item_label->set_sensitive (false);
    lbl_item_label->set_sensitive (false);

    item_id->set_text ("");
    item_id->set_sensitive (false);
    lbl_item_id->set_sensitive (false);

    item_shortcut->set_text ("");
    item_shortcut->set_sensitive (false);
    lbl_item_shortcut->set_sensitive (false);

    --update_block;
}

void GUI::MenuWindow::selection_changed_cb ()
{
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
	
    reset_controls ();

    if (!iter)
        return;
    
    btnRemove->set_sensitive (true);

    if ((*iter)[col_is_submenu])
    {
	// This is a submenu
	unsigned int index = (*iter)[col_index];
	
	Resources::Menu::Submenu &submenu = submenus[index];
	
        btnItem->set_sensitive (true);
        btnSeparator->set_sensitive (true);
        
	btnUp->set_sensitive (index != 0);
	btnDown->set_sensitive (index < (submenus.size () - 1));

	item_label->set_text (submenu.label);
	item_label->set_sensitive (true);
	lbl_item_label->set_sensitive (true);
    } else {
	// This is a menu item
	unsigned int index = (*iter)[col_index];
	unsigned int parent_index = (*iter)[col_parent_index];

	Resources::Menu::Submenu& submenu = submenus[parent_index];
	Resources::Menu::MenuItem& menuitem = submenu.items[index];

        btnItem->set_sensitive (true);
        btnSeparator->set_sensitive (true);
    
	btnUp->set_sensitive (index != 0);
	btnDown->set_sensitive (index < (submenu.items.size () - 1));
	
	if (!menuitem.separator)
	{
	    item_label->set_text (menuitem.label);
	    item_label->set_sensitive (true);
	    lbl_item_label->set_sensitive (true);
	    
	    item_id->set_text (menuitem.id);
	    item_id->set_sensitive (true);
	    lbl_item_id->set_sensitive (true);

	    gchar *shortcut_str = g_strdup_printf ("%c", menuitem.shortcut);
	    item_shortcut->set_text (shortcut_str);
	    item_shortcut->set_sensitive (true);
	    lbl_item_shortcut->set_sensitive (true);
	    g_free (shortcut_str);
	}
    }
}

void GUI::MenuWindow::entry_insert_cb (const Glib::ustring &text, int *pos, Gtk::Entry *entry)
{
    if (update_block)
        return;
    
    current_entry = entry;
    current_entry_pos = entry->get_position () + text.length ();
}

void GUI::MenuWindow::entry_delete_cb (int start_pos, int end_pos, Gtk::Entry *entry)
{
    if (update_block)
        return;
    
    current_entry = entry;
    current_entry_pos = start_pos;
}

void GUI::MenuWindow::id_entry_cb ()
{
    if (update_block)
	return;

    ResourceOps::RenameOpFactory op_factory (res);

    const std::string new_id = id_entry->get_text ();
    op_factory.push_change (new_id);

    id_entry->set_text (Glib::ustring (res->id));
}

void GUI::MenuWindow::item_label_entry_cb ()
{
    if (update_block)
	return;
    
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    const std::string new_label = item_label->get_text ();

    set_item_label (iter, new_label);
}

void GUI::MenuWindow::item_id_entry_cb ()
{
    if (update_block)
	return;
    
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    const std::string new_id = item_id->get_text ();

    set_item_id (iter, new_id);
}

void GUI::MenuWindow::item_shortcut_entry_cb ()
{
    if (update_block)
	return;
    
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    set_item_shortcut (iter, item_shortcut->get_text ());
}

void GUI::MenuWindow::set_item_label (const Gtk::TreeModel::iterator &iter, const std::string &new_label)
{
    if (!iter)
        return;

    unsigned int index = (*iter)[col_index];

    if ((*iter)[col_is_submenu])
    {
	if (submenus[index].label == new_label)
	    return;
	
	submenus[index].label = new_label;

	UndoOp *op = new SubmenuChangeOp (res, index, new_label);
	res->set_submenus (submenus);
	res->get_manager ()->get_undo_manager ().push (op);

    } else {
        unsigned int parent_index = (*iter)[col_parent_index];    

	std::vector<Resources::Menu::MenuItem>& items = submenus[parent_index].items;
	if (items[index].label == new_label)
	    return;
	
	items[index].label = new_label;
	
	UndoOp *op = new MenuItemLabelOp (res, parent_index, index, new_label);
	res->set_submenus (submenus);
	res->get_manager ()->get_undo_manager ().push (op);
    }
}

void GUI::MenuWindow::set_item_id (const Gtk::TreeModel::iterator &iter, const std::string &new_id)
{
    if (!iter)
        return;

    unsigned int index = (*iter)[col_index];
    unsigned int parent_index = (*iter)[col_parent_index];    

    if ((*iter)[col_is_submenu])
	return;

    std::vector<Resources::Menu::MenuItem>& items = submenus[parent_index].items;

    if (items[index].id == new_id)
	return;
    
    items[index].id = new_id;

    UndoOp *op = new MenuItemRenameOp (res, parent_index, index, new_id);
    res->set_submenus (submenus);
    res->get_manager ()->get_undo_manager ().push (op);
}

void GUI::MenuWindow::set_item_shortcut (const Gtk::TreeModel::iterator &iter, const std::string &new_shortcut)
{
    if (!iter)
        return;

    g_return_if_fail (!(*iter)[col_is_submenu]);
    
    unsigned int index = (*iter)[col_index];
    unsigned int parent_index = (*iter)[col_parent_index];

    std::vector<Resources::Menu::MenuItem>& items = submenus[parent_index].items;
    char shortcut = new_shortcut == "" ? 0 : new_shortcut[0];
    
    if (shortcut == items[index].shortcut)
	return;
    
    items[index].shortcut = shortcut;
    
    UndoOp *op = new MenuItemShortcutOp (res, parent_index, index, shortcut);
    res->set_submenus (submenus);
    res->get_manager ()->get_undo_manager ().push (op);
}

void GUI::MenuWindow::btn_up_cb ()
{
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    g_return_if_fail (iter);

    unsigned int index = (*iter)[col_index];
    unsigned int parent_index = (*iter)[col_parent_index];
    
    if ((*iter)[col_is_submenu])
        submenu_up (index);
    else
        menuitem_up (parent_index, index);
}

void GUI::MenuWindow::btn_down_cb ()
{
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    g_return_if_fail (iter);

    unsigned int index = (*iter)[col_index];
    unsigned int parent_index = (*iter)[col_parent_index];
    
    if ((*iter)[col_is_submenu])
        submenu_down (index);
    else
        menuitem_down (parent_index, index);
}

void GUI::MenuWindow::btn_remove_cb ()
{
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    g_return_if_fail (iter);

    unsigned int index = (*iter)[col_index];
    unsigned int parent_index = (*iter)[col_parent_index];
    
    if ((*iter)[col_is_submenu])
        submenu_remove (index);
    else
        menuitem_remove (parent_index, index);
}

void GUI::MenuWindow::btn_submenu_cb ()
{
    static const std::string submenu_name = _("New submenu");
    current_entry = item_label;
    current_entry_pos = submenu_name.length ();
    
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    if (!iter)
    {
	Resources::Menu::Submenu submenu (submenu_name);
	submenus.push_back (submenu);
        
	res->set_submenus (submenus);
        
        UndoOp *op = new SubmenuCreateOp (res, submenus.size () - 1);
        res->get_manager ()->get_undo_manager ().push (op);

        Gtk::TreeModel::Path path ("0");
        int new_row = 0;
        for (Resources::Menu::MenuTree::const_iterator i = submenus.begin ();
             i != submenus.end (); ++i)
            new_row += i->items.size () + 1;
        path[0] = new_row - 1;
        
        UI::flush_events ();
        treeview->get_selection ()->select (path);
        
	return;
    }
    
    Gtk::TreeModel::Path path = treestore->get_path (iter);
    
    unsigned int index = (*iter)[col_index];
    unsigned int parent_index = (*iter)[col_parent_index];

    std::vector<Resources::Menu::Submenu>::iterator new_index;
    Resources::Menu::Submenu submenu (submenu_name);
    
    if ((*iter)[col_is_submenu])
    {
	new_index = submenus.begin () + index + 1;
	path[0] += submenus[index].items.size () + 1;
    } else {
	new_index = submenus.begin () + parent_index + 1;        
	path[0] -= index - submenus[parent_index].items.size ();
    }

    submenus.insert (new_index, submenu);    
    res->set_submenus (submenus);
    UndoOp *op = new SubmenuCreateOp (res, index);
    res->get_manager ()->get_undo_manager ().push (op);

    UI::flush_events ();
    treeview->get_selection ()->select (path);
}

void GUI::MenuWindow::btn_item_cb (bool separator)
{
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    g_return_if_fail (iter);
    
    Gtk::TreeModel::Path path = treestore->get_path (iter);
    
    unsigned int index = (*iter)[col_index];
    unsigned int parent_index = (*iter)[col_parent_index];
    Resources::Menu::MenuItems::iterator pos;

    Resources::Menu::MenuItem menuitem;
    menuitem.separator = separator;

    if (!separator)
    {
	menuitem.id = res->create_id ("MENUITEM_");
	menuitem.label = menuitem.id;
	menuitem.shortcut = 0;
    }
    
    if ((*iter)[col_is_submenu])
    {
	pos = submenus[index].items.end ();
	submenus[index].items.insert (pos, menuitem);
	path[0] += submenus[index].items.size ();

	parent_index = index;
	index = submenus[parent_index].items.size () - 1;
	
    } else {
	
	pos = submenus[parent_index].items.begin () + index;
	submenus[parent_index].items.insert (pos, menuitem);
    }

    current_entry = item_id;
    current_entry_pos = menuitem.id.length ();
    
    res->set_submenus (submenus);
    UndoOp *op = new MenuItemCreateOp (res, parent_index, index);
    res->get_manager ()->get_undo_manager ().push (op);

    UI::flush_events ();    
    treeview->get_selection ()->select (path);
}

void GUI::MenuWindow::submenu_up (unsigned int index)
{
    g_return_if_fail (index > 0);

    Resources::Menu::MenuTree::iterator curr = submenus.begin () + index;
    Resources::Menu::MenuTree::iterator prev = curr - 1;
    int offset = prev->items.size() + 1;
    
    std::iter_swap (curr, prev);

    UndoOp *op = new SubmenuMoveOp (res, index, index - 1);
    res->set_submenus (submenus);
    res->get_manager ()->get_undo_manager ().push (op);

    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    Gtk::TreeModel::Path path = treestore->get_path (iter);
    path[0] -= offset;

    UI::flush_events ();
    treeview->get_selection ()->select (path);
}

void GUI::MenuWindow::submenu_down (unsigned int index)
{
    g_return_if_fail (index < (submenus.size() - 1));
    
    Resources::Menu::MenuTree::iterator curr = submenus.begin() + index;
    Resources::Menu::MenuTree::iterator next = curr + 1;
    int offset = next->items.size() + 1;

    std::iter_swap (curr, next);

    UndoOp *op = new SubmenuMoveOp (res, index, index + 1);
    res->set_submenus (submenus);
    res->get_manager ()->get_undo_manager ().push (op);

    
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    Gtk::TreeModel::Path path = treestore->get_path (iter);
    path[0] += offset;

    UI::flush_events ();
    treeview->get_selection ()->select (path);
}

void GUI::MenuWindow::submenu_remove (unsigned int index)
{
    submenus.erase (submenus.begin () + index);

    UndoOp *op = new SubmenuRemoveOp (res, index);
    res->set_submenus (submenus);
    res->get_manager ()->get_undo_manager ().push (op);
}
	
void GUI::MenuWindow::menuitem_up (unsigned int parent_index,
				   unsigned int index)
{
    Resources::Menu::MenuItems &items = submenus[parent_index].items;
    Resources::Menu::MenuItems::iterator curr = items.begin() + index;
    Resources::Menu::MenuItems::iterator prev = curr - 1;
    
    std::iter_swap (curr, prev);

    UndoOp *op = new MenuItemMoveOp (res, parent_index, index, index - 1);
    res->set_submenus (submenus);
    res->get_manager ()->get_undo_manager ().push (op);
    
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    Gtk::TreeModel::Path path = treestore->get_path (iter);
    --path[0];

    UI::flush_events ();
    treeview->get_selection ()->select (path);
}

void GUI::MenuWindow::menuitem_down (unsigned int parent_index,
				     unsigned int index)
{
    Resources::Menu::MenuItems& items = submenus[parent_index].items;
    Resources::Menu::MenuItems::iterator curr = items.begin() + index;
    Resources::Menu::MenuItems::iterator next = curr + 1;

    g_return_if_fail (index < (items.size () - 1));
    
    std::iter_swap (curr, next);

    UndoOp *op = new MenuItemMoveOp (res, parent_index, index, index + 1);
    res->set_submenus (submenus);
    res->get_manager ()->get_undo_manager ().push (op);
    
    Gtk::TreeModel::iterator iter = treeview->get_selection ()->get_selected ();
    Gtk::TreeModel::Path path = treestore->get_path (iter);
    ++path[0];

    UI::flush_events ();
    treeview->get_selection ()->select (path);
}

void GUI::MenuWindow::menuitem_remove (unsigned int parent_index,
				       unsigned int index)
{
    std::vector<Resources::Menu::MenuItem>& items = submenus[parent_index].items;
    items.erase (items.begin () + index);
    
    UndoOp *op = new MenuItemRemoveOp (res, parent_index, index);
    res->set_submenus (submenus);
    res->get_manager ()->get_undo_manager ().push (op);
}
