//$Id: io.cc,v 1.9 2004/07/12 22:07:58 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2004 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "io/io.h"

#include "io/xml-loader.h"
#include "io/xml-saver.h"
#include "io/rcp-loader.h"
#include "io/rcp-saver.h"

#include <libgnomevfsmm/uri.h>
#include <libgnomevfsmm/utils.h>

using namespace Guikachu;

Glib::ustring IO::create_canonical_uri (const Glib::ustring &filename)
{
    if (filename == "")
	return "";

    if (filename.find ("://") < filename.find ("/"))
	// Don't touch URI's
	return filename;
    
    Glib::ustring full_filename;
    if (g_path_is_absolute (filename.c_str ()))
	full_filename = filename;
    else {
	char *current_dirname = g_get_current_dir ();
	full_filename = Glib::ustring (current_dirname) + "/" + filename;
	g_free (current_dirname);
    }

    return Gnome::Vfs::make_uri_canonical (full_filename);
}

Glib::ustring IO::get_mime_type (const Glib::ustring &uri)
{
    Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri = Gnome::Vfs::Uri::create (uri);
    Glib::RefPtr<Gnome::Vfs::FileInfo> file_info = vfs_uri->get_file_info (Gnome::Vfs::FILE_INFO_GET_MIME_TYPE);
    return file_info->get_mime_type ();
}

IO::IOResult IO::load_uri (const Glib::ustring &uri, char *&data, GnomeVFSFileSize &len)
{
    GnomeVFSHandle *vfs_handle = 0;
    GnomeVFSResult  res = gnome_vfs_open (&vfs_handle, uri.c_str (), GNOME_VFS_OPEN_READ);
    
    if (res != GNOME_VFS_OK)
	return res;
        
    GnomeVFSFileInfo *vfs_info = gnome_vfs_file_info_new ();
    res = gnome_vfs_get_file_info_from_handle (
	vfs_handle, vfs_info, GNOME_VFS_FILE_INFO_FOLLOW_LINKS);
        
    if (res != GNOME_VFS_OK)
    {
	gnome_vfs_file_info_unref (vfs_info);
	gnome_vfs_close (vfs_handle);
	return res;
    }
        
    data = new char[vfs_info->size];
    len = 0;        
    char             buffer[10240];
    GnomeVFSFileSize bytes_read;
    
    gnome_vfs_file_info_unref (vfs_info);
    
    while (res == GNOME_VFS_OK)
    {
	res = gnome_vfs_read (vfs_handle, buffer, sizeof buffer - 1, &bytes_read);
	
	if(!bytes_read)
	    break;

	memcpy(data + len, buffer, bytes_read);
	len += bytes_read;
    }
    gnome_vfs_close (vfs_handle);
    
    return GNOME_VFS_OK;
}

IO::IOResult IO::save_uri (const Glib::ustring &uri, const char *data, GnomeVFSFileSize len)
{
    // Create file
    GnomeVFSHandle *vfs_handle = 0;
    GnomeVFSResult res = gnome_vfs_create (
	&vfs_handle, uri.c_str (), GNOME_VFS_OPEN_WRITE, 0, 0644);
    if (res != GNOME_VFS_OK)
	return res;
	
    // Write data to VFS stream
    const GnomeVFSFileSize chunk_size = len;
    
    for (GnomeVFSFileSize bytes_written = 0; len; data += bytes_written)
    {
        res = gnome_vfs_write (vfs_handle, data, std::min (chunk_size, len), &bytes_written);
        len -= bytes_written;
        
        if (res != GNOME_VFS_OK)
            break;
    }
    
    if (vfs_handle)
        gnome_vfs_close (vfs_handle);

    return res;
}

IO::IOFactory * IO::IOFactory::instance_ = 0;

namespace
{
    template<class T>
    IO::Loader * loader_factory ()
    {
	return new T;
    }

    template<class T>
    IO::Saver * saver_factory ()
    {
	return new T;
    }
    
} // anonymous namespace

IO::IOFactory::IOFactory ()
{
    loader_map["application/x-guikachu"] = loader_factory<IO::XMLLoader>;
    loader_map["text/x-rcp"] = loader_factory<IO::RCPLoader>;

    saver_map["application/x-guikachu"] = saver_factory<IO::XMLSaver>;
    saver_map["text/x-rcp"] = saver_factory<IO::RCPSaver>;
}

IO::IOFactory * IO::IOFactory::instance ()
{
    if (!instance_)
	instance_ = new IOFactory;
	
    return instance_;
}

IO::Loader * IO::IOFactory::create_loader (const std::string &mime_type) const
{
    loader_map_t::const_iterator found = loader_map.find (mime_type);
    if (found == loader_map.end ())
	return 0;

    return found->second ();
}

IO::Saver * IO::IOFactory::create_saver (const std::string &mime_type) const
{
    saver_map_t::const_iterator found = saver_map.find (mime_type);
    if (found == saver_map.end ())
	return 0;

    return found->second ();
}
