//$Id: resource-tree.cc,v 1.10 2004/06/10 15:51:15 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2004 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"
#include <libgnome/libgnome.h>

#include "resource-tree.h"

#include "resource-factory.h"
#include "resource-manager-ops.h"
#include "edit-cut-and-paste.h"

#include <libgnomeuimm/app-helper.h>
#include <libgnomeuimm/ui-items-stock.h>
#include <libgnomeuimm/app-helper.h>
#include <gtkmm/stock.h>

#include <sigc++/bind.h>
#include <sigc++/bind_return.h>

#include <algorithm>
#include <functional>

using namespace Guikachu::GUI;
using namespace Guikachu;

namespace
{
    class AppResetOp: public UndoOp
    {
        Resources::Application *res;
        std::string iconname, version, vendor;
        
    public:
        AppResetOp (Resources::Application *res_) :
            res (res_),
            iconname (res->iconname),
            version (res->version),
            vendor (res->vendor)
            {};
        
        Glib::ustring get_label () const { return _("Reset application settings"); };

        void undo () {
            res->iconname = iconname;
            res->version = version;
            res->vendor = vendor;
        };
        
        void redo () { res->reset (); };
    };
} // Anonymous namespace

ResourceTree::ResourceTree ():
    manager (0),
    app_win (0),
    popup_menu (0)
{
    Gtk::TreeModel::ColumnRecord cols;
    cols.add (col_label);
    cols.add (col_res);
    
    treestore = Gtk::TreeStore::create (cols);
    set_model (treestore);
    
    append_column ("", col_label);
    set_headers_visible (false);

    /* Connect to events */
    signal_button_press_event ().connect_notify (slot (*this, &ResourceTree::button_press_cb));
    signal_key_press_event ().connect (bind_return (slot (*this, &ResourceTree::key_press_cb), true));

    add_events (Gdk::BUTTON_PRESS_MASK | Gdk::KEY_PRESS_MASK);

    signal_row_activated ().connect (slot (*this, &ResourceTree::row_activated_cb));    
}

ResourceTree::~ResourceTree ()
{
    delete popup_menu;
    delete app_win;
}

Guikachu::Resource * ResourceTree::get_selected ()
{
    Glib::RefPtr<Gtk::TreeSelection> tree_selection = get_selection ();
    Gtk::TreeModel::iterator iter = tree_selection->get_selected ();

    if (!iter)
        return 0;

    return (*iter)[col_res];
}

void ResourceTree::set_manager (ResourceManager *manager_)
{
    delete app_win;
    app_win = 0;
    
    manager = manager_;

    treestore->clear ();
    subtrees.clear ();

    root_tree = *(treestore->append ());
    root_tree[col_res] = 0;

    manager->get_application ()->changed.connect (
	SigC::slot (*this, &ResourceTree::app_changed_cb));
    app_changed_cb ();
    
    create_tree_for_type (Resources::RESOURCE_FORM,         _("Forms"));
    create_tree_for_type (Resources::RESOURCE_DIALOG,       _("Dialogs"));
    create_tree_for_type (Resources::RESOURCE_MENU,         _("Menus"));
    create_tree_for_type (Resources::RESOURCE_BITMAP,       _("Bitmaps"));
    create_tree_for_type (Resources::RESOURCE_BITMAPFAMILY, _("Bitmap groups"));
    create_tree_for_type (Resources::RESOURCE_STRING,       _("Strings"));
    create_tree_for_type (Resources::RESOURCE_STRINGLIST,   _("String lists"));
    create_tree_for_type (Resources::RESOURCE_BLOB,         _("Blobs"));

    manager->resource_created.connect (
	SigC::slot (*this, &ResourceTree::resource_created_cb));
    manager->resource_removed.connect (
	SigC::slot (*this, &ResourceTree::resource_removed_cb));

    // Create tree items for existing resources
    const std::set<Resource*> resources = manager->get_resources ();
    std::for_each (resources.begin (), resources.end (),
		   SigC::slot (*this, &ResourceTree::resource_created_cb));

    expand_all ();
}

void ResourceTree::create_tree_for_type (Resources::Type      type,
					 const Glib::ustring &category_name)
{
    g_return_if_fail (subtrees.find (type) == subtrees.end ());
    
    Gtk::TreeRow row = *(treestore->append (root_tree.children ()));
    row[col_label] = category_name;
    row[col_res] = 0;

    subtrees[type] = row;
}

void ResourceTree::app_changed_cb ()
{
    std::string app_id = manager->get_application ()->iconname;
    if (app_id == "")
	app_id = _("Application");
    
    root_tree[col_label] = app_id;
}

namespace {
    struct ResourceFinder: public std::unary_function<Gtk::TreeModel::iterator, bool> 
    {
	typedef Guikachu::Resource *             target_t;
	typedef Gtk::TreeModelColumn<target_t> & col_t;
	
	target_t target;
	col_t    col;
	
	ResourceFinder (target_t target_, col_t col_) :
	    target (target_), col (col_) {};
	
	bool operator() (const Gtk::TreeRow &row) {
	    return row[col] == target;
	}
    };
} // anonymous namespace

void ResourceTree::resource_removed_cb (Resource *res)
{
    Gtk::TreeModel::Row &subtree_row = subtrees[res->get_type ()];
    Gtk::TreeModel::Children rows = subtree_row->children ();
    Gtk::TreeModel::iterator begin = rows.begin ();
    Gtk::TreeModel::iterator end = rows.end ();

    // This is an O(n) lookup, where n is the number of resources
    Gtk::TreeModel::iterator res_found = std::find_if (begin, end, ResourceFinder (res, col_res));
    g_return_if_fail (res_found != end);

    treestore->erase (res_found);
}

void ResourceTree::resource_created_cb (Resource *res)
{
    Gtk::TreeRow &subtree_row = subtrees[res->get_type ()];

    Gtk::TreeRow row = *(treestore->append (subtree_row.children ()));
    row[col_label] = Glib::ustring (res->id);
    row[col_res] = res;

    expand_row (treestore->get_path (subtree_row), true);
    
    res->changed.connect (SigC::bind (
	SigC::slot (*this, &ResourceTree::resource_changed_cb), row));
}

void ResourceTree::resource_changed_cb (Gtk::TreeRow row)
{
    Resource *res = row[col_res];
    
    row[col_label] = Glib::ustring (res->id);
}

void ResourceTree::show_app_win ()
{
    if (!app_win)
        app_win = new AppWindow (manager->get_application ());

    app_win->show ();
}

void ResourceTree::show_res_win (Resource *res)
{
    ResourceWindow *res_win = get_resource_editor (res);
    
    if (res_win)
        res_win->show ();
}

void ResourceTree::row_activated_cb (const Gtk::TreeModel::Path &path,
				     Gtk::TreeView::Column      *col)
{
    Gtk::TreeModel::iterator iter = treestore->get_iter (path);
    if (iter == treestore->children ().begin ())
    {
        show_app_win ();
	return;
    }
    
    if ((*iter)[col_res])
    {
	Resource *res = (*iter)[col_res];
        show_res_win (res);
    }
}

void ResourceTree::button_press_cb (GdkEventButton *e)
{
    if (e->button != 3)
	return;

    int cell_x, cell_y;
    Gtk::TreeModel::Path  path;
    Gtk::TreeViewColumn  *column;
    Guikachu::Resource   *resource = 0;

    // FIXME: e->x and e->y are doubles
    if (get_path_at_pos (e->x, e->y, path, column, cell_x, cell_y))
    {
	Gtk::TreeModel::iterator iter = treestore->get_iter (path);
        if (iter == treestore->children ().begin ())
            popup_app_menu (e->button, e->time);
        else
            resource = (*iter)[col_res];
    }
    
    if (resource)
        popup_res_menu (e->button, e->time, resource);
}

void ResourceTree::key_press_cb (GdkEventKey *e)
{
    Gtk::TreeModel::iterator iter = get_selection ()->get_selected ();
    if (!iter)
        return;

    Resource *res = (*iter)[col_res];
    if (!res && iter != treestore->children ().begin ())
        return;

    switch (e->keyval)
    {
    case GDK_Return:
    case GDK_KP_Enter:
        if (res)
            show_res_win (res);
        else
            show_app_win ();
        break;
        
    case GDK_Delete:
    case GDK_KP_Delete:
        if (res)
            res_remove_cb (res);
        else
            app_reset_cb ();
        break;

    case GDK_Menu:
        if (res)
            popup_res_menu (0, e->time, res);
        else
            popup_app_menu (0, e->time);
        break;
        
    default:
        break;
    }
}

void ResourceTree::popup_app_menu (guint button, guint32 time)
{
    if (popup_menu)
        delete popup_menu;

    popup_menu = new Gtk::Menu;
    std::list <Gnome::UI::Items::Info> popup_menu_list;
    
    popup_menu_list.push_back (
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::PROPERTIES),
				_("_Edit"),
				SigC::slot (*this, &ResourceTree::show_app_win)));

    Gnome::UI::Items::fill (*popup_menu, popup_menu_list,
			    popup_menu->get_accel_group ());
    
    popup_menu->popup (button, time);
}

void ResourceTree::app_reset_cb ()
{
    Resources::Application *app = manager->get_application ();
    manager->get_undo_manager ().push (new AppResetOp (app));
    app->reset ();
    if (app_win)
        app_win->hide ();

}

void ResourceTree::popup_res_menu (guint button, guint32 time, Resource *res)
{
    if (popup_menu)
        delete popup_menu;

    popup_menu = new Gtk::Menu;
    
    Gnome::UI::Items::Info popup_menu_list[] = {
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::PROPERTIES),
				_("_Edit"),
				SigC::bind (SigC::slot (*this, &ResourceTree::show_res_win), res)),
	Gnome::UI::Items::Separator (),
	Gnome::UI::MenuItems::Cut   (SigC::bind (SigC::slot (*this, &ResourceTree::res_cut_cb), res)),
	Gnome::UI::MenuItems::Copy  (SigC::bind (SigC::slot (*this, &ResourceTree::res_copy_cb), res)),
	Gnome::UI::Items::Separator (),
	Gnome::UI::Items::Item (Gnome::UI::Items::Icon (Gtk::Stock::DELETE),
				_("_Remove"),
				SigC::bind (SigC::slot (*this, &ResourceTree::res_remove_cb), res)),
    };

    Gnome::UI::Items::fill (*popup_menu, popup_menu_list,
			    popup_menu->get_accel_group ());
    
    popup_menu->popup (button, time);
}

void ResourceTree::res_cut_cb (Resource *res)
{
    res_copy_cb (res);
    res_remove_cb (res);
}

void ResourceTree::res_copy_cb (Resource *res)
{
    Edit::copy_resource (res);
}

void ResourceTree::res_remove_cb (Resource *res)
{
    manager->get_undo_manager ().push (new ResourceOps::RemoveOp (res));
    manager->remove_resource (res);
}
