//$Id: preferences-win.cc,v 1.10 2004/06/10 15:51:15 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2004 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "preferences-win.h"
#include "preferences.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include <gtkmm/adjustment.h>

#include <sigc++/bind.h>
#include <sigc++/hide.h>

namespace {
    static std::string rgb_to_color (guint8 r, guint8 g, guint8 b);
    static void color_to_rgb (const std::string &color,
			      guint8 &r, guint8 &g, guint8 &b);
}

using namespace Guikachu;

GUI::PreferencesWin::PreferencesWin ():
    gui (Gnome::Glade::Xml::create (GUIKACHU_GLADEDIR "/guikachu.glade", "preferences"))
{
    Gtk::Button *ok_btn, *cancel_btn;
    
    gui->get_widget ("preferences",         prop_box);
    gui->get_widget ("pref_color_fg",       color_fg);
    gui->get_widget ("pref_color_disabled", color_disabled);
    gui->get_widget ("pref_color_bg",       color_bg);
    gui->get_widget ("pref_color_sel",      color_selection);
    gui->get_widget ("pref_zoom",           spin_zoom);
    gui->get_widget ("pref_apply",          apply_btn);
    gui->get_widget ("pref_ok",             ok_btn);
    gui->get_widget ("pref_cancel",         cancel_btn);
    
    prop_box->set_title (_("Preferences"));

    using namespace SigC;
    
    // Set up 'changed' signals so that the 'Apply' button is made sensitive
    color_fg->signal_color_set ().connect (
	slot (*this, &PreferencesWin::color_changed_cb));
    color_disabled->signal_color_set ().connect (
	slot (*this, &PreferencesWin::color_changed_cb));
    color_bg->signal_color_set ().connect (
	slot (*this, &PreferencesWin::color_changed_cb));
    color_selection->signal_color_set ().connect (
	slot (*this, &PreferencesWin::color_changed_cb));

    spin_zoom->get_adjustment ()->signal_value_changed ().connect (
	bind (slot (*this, &PreferencesWin::set_modified), true));
    
    // Update controls when window's shown
    prop_box->signal_show ().connect (SigC::slot (*this, &PreferencesWin::update));
    
    // Set up buttons
    apply_btn->signal_clicked ().connect (SigC::slot (*this, &PreferencesWin::apply_cb));
    ok_btn->signal_clicked ().connect (SigC::slot (*this, &PreferencesWin::apply_cb));
    ok_btn->signal_clicked ().connect (SigC::slot (*prop_box, &Gtk::Dialog::hide));
    cancel_btn->signal_clicked ().connect (SigC::slot (*prop_box, &Gtk::Dialog::hide));
}

void GUI::PreferencesWin::color_changed_cb (guint r, guint g, guint b, guint a)
{
    set_modified (true);
}

void GUI::PreferencesWin::set_modified (bool modified)
{
    apply_btn->set_sensitive (modified);
}

void GUI::PreferencesWin::run ()
{
    prop_box->show ();
}

void GUI::PreferencesWin::update ()
{
    // Update color pickers
    guint8 r, g, b;
    std::string color;

    color = Preferences::FormEditor::get_color_fg ();
    color_to_rgb (color, r, g, b);
    color_fg->set (r, g, b, 255);

    color = Preferences::FormEditor::get_color_disabled ();
    color_to_rgb (color, r, g, b);
    color_disabled->set (r, g, b, 255);

    color = Preferences::FormEditor::get_color_bg ();
    color_to_rgb (color, r, g, b);
    color_bg->set (r, g, b, 255);

    color = Preferences::FormEditor::get_color_selection ();
    color_to_rgb (color, r, g, b);
    color_selection->set (r, g, b, 255);

    // Update default zoom entry
    spin_zoom->get_adjustment ()->set_value (Preferences::FormEditor::get_default_zoom ());

    // Reset property box state
    set_modified (false);
}

void GUI::PreferencesWin::apply_cb ()
{
    // Set form editor colors
    guint8 r, g, b, dummy;
    std::string color_fg_, color_disabled_, color_bg_, color_selection_;

    color_fg->get (r, g, b, dummy);
    color_fg_ = rgb_to_color (r, g, b);
    
    color_disabled->get (r, g, b, dummy);
    color_disabled_ = rgb_to_color (r, g, b);
    
    color_bg->get (r, g, b, dummy);
    color_bg_ = rgb_to_color (r, g, b);
    
    color_selection->get (r, g, b, dummy);
    color_selection_ = rgb_to_color (r, g, b);    
    
    Preferences::FormEditor::set_colors (color_fg_, color_disabled_,
					 color_bg_, color_selection_);

    // Set default form editor zoom factor
    Preferences::FormEditor::set_default_zoom (spin_zoom->get_value ());

    set_modified (false);
}

namespace {

// This converts from "#rrggbb" format
static void color_to_rgb (const std::string &color,
			  guint8 &r, guint8 &g, guint8 &b)
{
    g_return_if_fail (color.length () == 7);
    g_return_if_fail (color[0] == '#');
    
    unsigned int r_, g_, b_;    
    sscanf (color.c_str (), "#%02x%02x%02x", &r_, &g_, &b_);
    r = r_;
    g = g_;
    b = b_;
}

static std::string rgb_to_color (guint8 r, guint8 g, guint8 b)
{
    char *tmp = g_strdup_printf ("#%02x%02x%02x", r, g, b);
    std::string ret_val = tmp;
    g_free (tmp);

    return ret_val;
}

} // Anonymous namespace
