//$Id: form-win-widgettree.cc,v 1.3 2004/06/10 15:51:15 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2004 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "form-win-widgettree.h"

#include "form-editor/widget-util.h"

#include <gdk/gdkkeysyms.h>

using namespace Guikachu::GUI::FormWindow_Helpers;

WidgetTreeWrapper::WidgetTreeWrapper (Gtk::TreeView    &treeview_,
				      Resources::Form  *form_):
    treeview (treeview_),
    form (form_),
    selection_block (false),
    menu_block (false)
{
    Gtk::TreeModel::ColumnRecord cols;
    cols.add (col_label);
    cols.add (col_widget);
    
    treestore = Gtk::TreeStore::create (cols);
    treeview.set_model (treestore);
    
    treeview.append_column ("", col_label);
    treeview.set_headers_visible (false);
    
    // Create root item: the form itself
    root_tree = *(treestore->append ());
    root_tree[col_widget] = 0;

    form->changed.connect (SigC::slot (*this, &WidgetTreeWrapper::form_changed_cb));
    form_changed_cb ();
    
    form->widget_created.connect (SigC::slot (*this, &WidgetTreeWrapper::widget_created_cb));
    form->widget_removed.connect (SigC::slot (*this, &WidgetTreeWrapper::widget_removed_cb));

    treeview.get_selection ()->set_select_function (
	SigC::slot (*this, &WidgetTreeWrapper::selection_changed_cb));
    
    // Create tree items for existing widgets
    const std::set<Guikachu::Widget*> widgets = form->get_widgets ();
    std::for_each (widgets.begin (), widgets.end (),
		   SigC::slot (*this, &WidgetTreeWrapper::widget_created_cb));

    treeview.expand_all ();

    // Set selection mode
    treeview.get_selection ()->set_mode (Gtk::SELECTION_MULTIPLE);

    treeview.add_events (Gdk::BUTTON_PRESS_MASK | Gdk::KEY_PRESS_MASK);
    treeview.signal_button_press_event ().connect_notify (
	SigC::slot (*this, &WidgetTreeWrapper::button_press_cb));
    treeview.signal_key_press_event ().connect (
	SigC::bind_return (SigC::slot (*this, &WidgetTreeWrapper::key_press_cb), false));        
}

void WidgetTreeWrapper::widget_removed_cb (Guikachu::Widget *widget)
{
    row_map_t::iterator row_found = row_map.find (widget);
    g_return_if_fail (row_found != row_map.end ());
    
    treestore->erase (row_found->second);
    row_map.erase (row_found);
}

void WidgetTreeWrapper::widget_created_cb (Guikachu::Widget *widget)
{
    Gtk::TreeRow row = *(treestore->append (root_tree.children ()));
    row[col_label] = Glib::ustring (widget->id);
    row[col_widget] = widget;

    row_map[widget] = row;
    
    widget->changed.connect (SigC::bind (
	SigC::slot (*this, &WidgetTreeWrapper::widget_changed_cb), row));
    widget->selected.connect (SigC::bind (
	SigC::slot (*this, &WidgetTreeWrapper::widget_selected_cb), widget));
    
    treeview.expand_all ();
}

void WidgetTreeWrapper::widget_changed_cb (Gtk::TreeRow row)
{
    Guikachu::Widget *widget = row[col_widget];
    
    row[col_label] = Glib::ustring (widget->id);
}

void WidgetTreeWrapper::form_changed_cb ()
{
    root_tree[col_label] = Glib::ustring (form->id);
}

void WidgetTreeWrapper::widget_selected_cb (bool selected,
					    Guikachu::Widget *widget)
{
    row_map_t::const_iterator row_iter = row_map.find (widget);
    if (row_iter == row_map.end ())
	return;
	
    selection_block = true;
    Glib::RefPtr<Gtk::TreeSelection> tree_selection = treeview.get_selection ();
    Gtk::TreeRow row = row_iter->second;

    if (selected)
    {
	tree_selection->select (row);
	tree_selection->unselect (root_tree);
    } else {
	tree_selection->unselect (row);
    }

    selection_block = false;
}

bool WidgetTreeWrapper::selection_changed_cb (const Glib::RefPtr<Gtk::TreeModel> &model,
					      const Gtk::TreeModel::Path         &path,
					      bool                                previously_selected)
{
    if (menu_block)
        return false;
    
    if (selection_block)
	return true;

    bool selected = !previously_selected;
    
    if (path.get_depth () < 2)
    {
	if (selected)
	    form_selected.emit ();
	return true;
    }
    
    Gtk::TreeModel::iterator iter = model->get_iter (path);
    Guikachu::Widget *widget = (*iter)[col_widget];

    widget_selected.emit (widget, selected);

    return false;
}

namespace
{
    void reset_menu_block (bool *menu_block)
    {
        *menu_block = false;
    }
} // anonymous namespace

void WidgetTreeWrapper::button_press_cb (GdkEventButton *e)
{
    if (e->button != 3)
	return;

    int cell_x, cell_y;
    Gtk::TreeModel::Path  path;
    Gtk::TreeViewColumn  *column;
    Guikachu::Widget *widget = 0;

    // FIXME: e->x and e->y are doubles
    if (treeview.get_path_at_pos (e->x, e->y, path, column, cell_x, cell_y))
    {
        if (treeview.get_selection ()->is_selected (path))
        {
            menu_block = true;
            Glib::signal_idle ().connect (
                SigC::bind_return (SigC::bind (SigC::slot (reset_menu_block), &menu_block), false));
        }
            
	Gtk::TreeModel::iterator iter = treestore->get_iter (path);
	widget = (*iter)[col_widget];
    }

    if (widget)
	widget_menu.emit (e->button, e->time, widget);
    else
        form_menu.emit (e->button, e->time);
}

namespace
{
    template<typename T, typename Container = std::list<T> >
    class SelectionCollector: public SigC::Object
    {
    public:
        typedef T                               element_t;
        typedef Container                       container_t;
        typedef Gtk::TreeModelColumn<element_t> column_t;
        
    private:
        column_t    column;
        container_t items;
        
    public:
        SelectionCollector (const column_t &column_): column (column_) {};
        container_t get_items () const { return items; };
        void selection_cb (const Gtk::TreeModel::iterator &iter) { items.push_back ((*iter)[column]); };
    };
} // anonymous namespace

void WidgetTreeWrapper::key_press_cb (GdkEventKey *e)
{
    Guikachu::Widget *widget = 0;
    
    typedef SelectionCollector<Guikachu::Widget*> collector_t;
    typedef collector_t::container_t              collector_list_t;

    collector_t collector (col_widget);
    treeview.get_selection ()->selected_foreach (SigC::slot (collector, &collector_t::selection_cb));
    collector_list_t items = collector.get_items ();

    // It doesn't really matter which widget we select when more than
    // one are selected
    if (items.size ())
        widget = items.front ();

    switch (e->keyval)
    {
    case GDK_Delete:
    case GDK_KP_Delete:
        if (widget)
            widget_remove.emit (widget);
        break;
        
    case GDK_Menu:
        if (widget)
            widget_menu.emit (0, e->time, widget);
        else
            form_menu.emit (0, e->time);
        break;
        
    default:
        break;
    }
}
