//$Id: bitmap-win.cc,v 1.10 2004/06/10 15:51:15 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2004 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "bitmap-win.h"

#include "bitmap-res-ops.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include <gtkmm/alignment.h>
#include <gtkmm/menu.h>
#include <gtkmm/fileselection.h>
#include <gtkmm/main.h>
#include <gtkmm/button.h>
#include <gtkmm/frame.h>
#include <gtkmm/box.h>

#include "ui-gui.h"

#include <sigc++/bind_return.h>
#include <sigc++/hide.h>

#include "io/io.h"

#include "property-ops-resource.h"

#include "widgets/propertytable.h"
#include "widgets/entry.h"

#include "form-editor/form-editor-canvas.h"
#include "preferences.h"

#include <gdk-pixbuf/gdk-pixbuf.h>

using namespace Guikachu;

GUI::BitmapWindow::BitmapWindow (Resources::Bitmap *res_):
    res (res_),
    update_block (true)
{
    using namespace SigC;
    using ResourceOps::PropChangeOpFactory;
    using ResourceOps::RenameOpFactory;
    
    window.signal_delete_event ().connect (slot (*this, &BitmapWindow::delete_event_impl));
    window.property_allow_grow () = true;
    
    GUI::PropertyTable *proptable = new GUI::PropertyTable;
    Gtk::Widget *control;

    /* Resource ID */
    control = new GUI::PropertyEditors::Entry (false, res->id, new RenameOpFactory (res));
    proptable->add (_("Resource _ID:"), *manage (control));

    /* Type */
    Gtk::Menu *type_menu = new Gtk::Menu;

#define BITMAP_TYPE(s,t)						\
	    type_menu->items ().push_back (				\
		Gtk::Menu_Helpers::MenuElem (				\
		    s,							\
		    bind (slot (*this, &BitmapWindow::type_menu_cb),	\
			  Resources::Bitmap::t)));

    BITMAP_TYPE (_("Monochrome"),              TYPE_MONO);
    BITMAP_TYPE (_("Greyscale (4 shades)"),    TYPE_GREY_4);
    BITMAP_TYPE (_("Greyscale (16 shades)"),   TYPE_GREY_16);
    BITMAP_TYPE (_("Color (16 colors)"),       TYPE_COLOR_16);
    BITMAP_TYPE (_("Color (256 colors)"),      TYPE_COLOR_256);
    BITMAP_TYPE (_("High Color (16K colors)"), TYPE_COLOR_16K);
#undef BITMAP_TYPE

    type_menu->show_all ();
    type_combo.set_menu (*manage (type_menu));
    proptable->add (_("T_ype:"), type_combo);

    /* Preview of the image */
    Gtk::VBox *vbox = new Gtk::VBox (false, 5);

    Gtk::Button *load_button = UI::create_stock_button (Gtk::Stock::OPEN, _("Load"));
    export_button = UI::create_stock_button (Gtk::Stock::SAVE_AS, _("Export"));

    Gtk::HBox *button_box = new Gtk::HBox (false, 5);    
    button_box->pack_start (*manage (load_button));
    button_box->pack_start (*manage (export_button));
    Gtk::Frame *preview_frame = new Gtk::Frame;

//    preview_pixmap.set_padding (5, 5);
    preview_frame->add (preview_pixmap);
    
    Gtk::Alignment *alignment = new Gtk::Alignment (0, 0.5, 0, 0);
    alignment->add (*manage (preview_frame));
    vbox->add (*manage (alignment));
    vbox->add (*manage (button_box));
    
    export_button->signal_clicked ().connect (slot (*this, &BitmapWindow::export_image_cb));
    load_button->signal_clicked ().connect (slot (*this, &BitmapWindow::load_image_cb));
    proptable->add (_("Image preview:"), *manage (vbox));
    
    window.add (*manage (proptable));

    res->changed.connect (slot (*this, &BitmapWindow::update));
    res->get_manager ()->get_target ()->changed.connect (
        slot (*this, &BitmapWindow::update_image));
    update ();
}

void GUI::BitmapWindow::show ()
{
    window.show_all ();
    window.raise ();
}

bool GUI::BitmapWindow::delete_event_impl (GdkEventAny *e)
{
    window.hide ();
    return true;
}

void GUI::BitmapWindow::update_image ()
{
    Magick::Image image = res->get_image ();
    if (!image.isValid ())
    {
        preview_pixmap.set ("");
        export_button->set_sensitive (false);
        return;
    }
    export_button->set_sensitive (true);
    
    Glib::RefPtr<Gdk::Pixbuf> pixbuf = FormEditor::render_bitmap (image, res->bitmap_type);

    if (pixbuf)
    {
        guint32 background_color = 0xFFFFFF;

        if (!res->get_manager ()->get_target ()->screen_color &&
            res->bitmap_type < Resources::Bitmap::TYPE_COLOR_16)
        {
            Gdk::Color c (Preferences::FormEditor::get_color_bg ());
            background_color = ((c.get_red () >> 8) << 16) +
                ((c.get_green () >> 8) << 8) + (c.get_blue () >> 8);
        }

        Glib::RefPtr<Gdk::Pixbuf> pixbuf_with_background =
            pixbuf->composite_color_simple (pixbuf->get_width (),
                                            pixbuf->get_height (),
                                            Gdk::INTERP_NEAREST, 255,
                                            1, background_color, background_color);

        preview_pixmap.set (pixbuf_with_background);
    }
}

void GUI::BitmapWindow::update ()
{
    // Update window title
    gchar *title_buf = g_strdup_printf (_("Bitmap: %s"), res->id ().c_str ());
    window.set_title (title_buf);
    g_free (title_buf);

    update_image ();
    
    update_block = true;    
    type_combo.set_history (res->bitmap_type);
    update_block = false;
}

void GUI::BitmapWindow::type_menu_cb (Resources::Bitmap::BitmapType type)
{
    if (res->bitmap_type == type)
	return;

    char *label_str = g_strdup_printf (_("Change type of %s"), res->id ().c_str ());
    
    UndoOp *op = new ResourceOps::PropChangeOp<Resources::Bitmap::BitmapType> (
	label_str, res, res->bitmap_type, type, false);
    g_free (label_str);
    
    res->bitmap_type = type;

    res->get_manager ()->get_undo_manager ().push (op);
}

namespace
{
    class ImageFileSelector: public SigC::Object
    {
        Gtk::FileSelection fs;
        Resources::Bitmap  *res;

    public:
        ImageFileSelector (Gtk::Window &parent, Resources::Bitmap *res, bool save);
        void run ();
        
    private:
        void close_cb ();
        void open_cb ();
	void export_cb ();
    };
} // Anonymous namespace

ImageFileSelector::ImageFileSelector (Gtk::Window &parent, Resources::Bitmap *res_, bool save) :
    res (res_)
{
    char *title;
    if (save)
    {
	title = g_strdup_printf (_("Exporting %s"), res->id ().c_str ());
	fs.show_fileop_buttons ();
	fs.get_ok_button ()->signal_clicked ().connect (
            SigC::slot (*this, &ImageFileSelector::export_cb));
    } else {
	title = g_strdup_printf (_("Loading image file to %s"), res->id ().c_str ());
	fs.hide_fileop_buttons ();
	fs.get_ok_button ()->signal_clicked ().connect (
            SigC::slot (*this, &ImageFileSelector::open_cb));
    }
    fs.set_title (title);
    g_free (title);
    
    fs.get_cancel_button ()->signal_clicked ().connect (SigC::slot (*this, &ImageFileSelector::close_cb));
    fs.signal_delete_event ().connect (
	SigC::hide<GdkEventAny*> (SigC::bind_return (
	    SigC::slot (*this, &ImageFileSelector::close_cb), true)));
    
    fs.set_transient_for (parent);
    fs.set_modal (true);
}

void ImageFileSelector::close_cb ()
{
    fs.hide ();
    Gtk::Main::quit ();
}

void ImageFileSelector::open_cb ()
{
    std::string uri = IO::create_canonical_uri (fs.get_filename ());

    Magick::Image old_image = res->get_image ();
    res->load_file (uri);

    UndoOp *op = new ResourceOps::BitmapOps::ImageChangeOp (res, old_image);
    res->get_manager ()->get_undo_manager ().push (op);
    
    close_cb ();
}

void ImageFileSelector::export_cb ()
{
    // TODO: Show a warning before overwriting files
    Magick::Image image = res->get_image ();    
    image.write (fs.get_filename ());
    
    close_cb ();
}

void ImageFileSelector::run ()
{
    fs.show ();
    Gtk::Main::run ();
}

void GUI::BitmapWindow::load_image_cb ()
{
    ImageFileSelector ifs (window, res, false);
    ifs.run ();
}

void GUI::BitmapWindow::export_image_cb ()
{
    ImageFileSelector ifs (window, res, true);
    ifs.run ();
}
