//$Id: form-win-helpers.cc,v 1.20.4.1 2004/06/01 14:12:59 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "form-win-helpers.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include <gtk--/box.h>
#include <gtk--/pixmap.h>
#include <gnome--/app-helper.h>
#include <libgnomeui/gnome-stock.h>

#include <sigc++/retbind.h>

#include <list>
#include <algorithm>

#include "form-editor/widget-util.h"
#include "form-editor/textual.h"

namespace {
#include "pixmaps/form.xpm"
}

using namespace Guikachu::GUI::FormWindow_Helpers;


/***********************************
 * CanvasSelectionWrapper
 ***********************************/

CanvasSelectionWrapper::CanvasSelectionWrapper (Gnome::Canvas &canvas_):
    canvas (canvas_),
    rubber_box (0),
    dragging (false),
    active (true)
{
    canvas.button_press_event.connect_after (SigC::retbind (
	SigC::slot (this, &CanvasSelectionWrapper::selection_begin), 1));
    canvas.motion_notify_event.connect_after (SigC::retbind (
	SigC::slot (this, &CanvasSelectionWrapper::selection_drag), 1));
    canvas.button_release_event.connect_after (SigC::retbind (
	SigC::slot (this, &CanvasSelectionWrapper::selection_end), 1));
    
}

CanvasSelectionWrapper::~CanvasSelectionWrapper ()
{
    delete rubber_box;
}

void CanvasSelectionWrapper::set_active (bool active_)
{
    active = active_;
}

void CanvasSelectionWrapper::selection_begin (GdkEventButton *e)
{
    if (e->button != 1 || dragging)
	return;

    if (!active)
	return;

    double x (0), y (0);
    canvas.window_to_world (e->x, e->y, x, y);
    
    static const char stipple_bits[] = { 0x02, 0x01 };
    GdkBitmap *stipple = gdk_bitmap_create_from_data (NULL,
						      stipple_bits,
						      2, 2);

    selection_start_x = x;
    selection_start_y = y;
    
    rubber_box = new Gnome::CanvasRect (*(canvas.root ()), x, y, x, y);
    rubber_box->set_width_pixels (1);
    rubber_box->set_outline_color ("black");
    rubber_box->set_outline_stipple (stipple);
    
    dragging = true;
}

void CanvasSelectionWrapper::selection_drag (GdkEventMotion *e)
{
    if (!active)
	return;
    
    if (!dragging)
	return;

    g_return_if_fail (rubber_box != 0);
    
    double x (0), y (0);
    
    canvas.window_to_world (e->x, e->y, x, y);

    double x1 = std::min (x, selection_start_x);
    double x2 = std::max (x, selection_start_x);

    double y1 = std::min (y, selection_start_y);
    double y2 = std::max (y, selection_start_y);

    rubber_box->set_x1 (x1);
    rubber_box->set_y1 (y1);

    rubber_box->set_x2 (x2);
    rubber_box->set_y2 (y2);
}

void CanvasSelectionWrapper::selection_end (GdkEventButton *e)
{
    if (!active)
	return;
    
    if (e->button != 1 || !dragging)
	return;

    g_return_if_fail (rubber_box != 0);

    double x (0), y (0);
    
    canvas.window_to_world (e->x, e->y, x, y);

    selection.emit (int (selection_start_x), int (selection_start_y),
		    int (x), int (y));
    
    delete rubber_box;
    rubber_box = 0;

    dragging = false;
}

/***********************************
 * WidgetTreeItem
 ***********************************/

WidgetTreeItem::WidgetTreeItem (Guikachu::Widget *widget_):
    widget (widget_)
{
    using namespace Gnome;

    Gtk::Pixmap *pixmap = new Gtk::Pixmap (Widgets::type_icon (widget->get_type ()));
    Gtk::Box    *box    = new Gtk::HBox   (false, 5);

    label.set_alignment (0, 0.5);
    box->pack_start (*manage (pixmap), false);
    box->pack_start (label);
    box->show_all ();    
    add (*manage (box));

    set_user_data (widget);

    std::list <UI::Info> popup_menu_list;
    popup_menu_list.push_back (UI::Item (UI::Icon (GNOME_STOCK_MENU_PROP),_("_Edit"), menu_edit.slot ()));
    popup_menu_list.push_back (UI::Separator ());
    popup_menu_list.push_back (MenuItems::Cut (menu_cut.slot ()));
    popup_menu_list.push_back (MenuItems::Copy (menu_copy.slot ()));
    popup_menu_list.push_back (MenuItems::Paste (menu_paste.slot ()));
    popup_menu_list.push_back (UI::Separator ());
    popup_menu_list.push_back (UI::Item (UI::Icon (GNOME_STOCK_MENU_TRASH), _("_Remove"), menu_remove.slot ()));
    UI::fill (popup_menu, popup_menu_list, *(popup_menu.get_accel_group ()));
    
    button_press_event.connect (slot (this, &WidgetTreeItem::button_press_cb));
    key_press_event.connect (slot (this, &WidgetTreeItem::key_press_cb));
    
    widget->selected.connect (slot (this, &WidgetTreeItem::widget_selected_cb));
    widget->changed.connect  (slot (this, &WidgetTreeItem::update));
    update ();
}

void WidgetTreeItem::update ()
{
    // I'm not sure this is looks good, so it's disabled for now
#if 0
    if (Widgets::Textual *textual = dynamic_cast<Widgets::Textual*> (widget))
    {
	const std::string widget_text = textual->text;
	std::string txt = widget->id () + ": ";

	for (std::string::const_iterator i = widget_text.begin ();
	     i != widget_text.end (); ++i)
	    if (*i == '\n')
		txt += "\\n";
	    else
		txt += *i;

//	txt += "\")";
	
	label.set_text (txt);
    } else {
	label.set_text (widget->id);
    }
#else
    label.set_text (widget->id);
#endif
}

int WidgetTreeItem::button_press_cb (GdkEventButton *e)
{
    if (e->type == GDK_BUTTON_PRESS && e->button == 1)
	activate ();
    
    if (e->button == 3)
	popup_menu.popup (e->button, e->time);
    
    return false;
}

int WidgetTreeItem::key_press_cb (GdkEventKey *e)
{
    switch (e->keyval)
    {
    case GDK_Return:
    case GDK_KP_Enter:
        menu_edit.emit ();
        break;
        
    case GDK_Delete:
    case GDK_KP_Delete:
        menu_remove.emit ();
        break;

    case GDK_Menu:
        popup_menu.popup (0, e->time);
        break;
    }
    
    return false;
}

void WidgetTreeItem::widget_selected_cb (bool selected)
{
    if (selected)
	select ();
    else
	deselect ();
}

/***********************************
 * TopTreeItem
 ***********************************/

TopTreeItem::TopTreeItem (Resources::Form *res_):
    res (res_)
{
    using namespace Gnome;
    
    Gtk::Pixmap *pixmap = new Gtk::Pixmap (form_xpm);
    Gtk::Box    *box    = new Gtk::HBox   (false, 5);

    label.set_alignment (0, 0.5);
    box->pack_start (*manage (pixmap), false);
    box->pack_start (label);
    box->show_all ();
    add (*manage (box));
    
    set_user_data (res);
    
    std::list <UI::Info> popup_menu_list;
    popup_menu_list.push_back (UI::Item (UI::Icon (GNOME_STOCK_MENU_PROP), _("_Edit form"), menu_edit.slot ()));
    popup_menu_list.push_back (UI::Item (_("_Select all"), menu_select_all.slot ()));
    popup_menu_list.push_back (UI::Separator ());
    popup_menu_list.push_back (MenuItems::Paste (menu_paste.slot ()));
    UI::fill (popup_menu, popup_menu_list, *(popup_menu.get_accel_group ()));

    res->changed.connect (slot (this, &TopTreeItem::update));
    button_press_event.connect (slot (this, &TopTreeItem::button_press_cb));
    key_press_event.connect (slot (this, &TopTreeItem::key_press_cb));

    update ();
}

void TopTreeItem::update ()
{
    label.set_text (res->id);
}

int TopTreeItem::button_press_cb (GdkEventButton *e)
{
    if (e->type == GDK_BUTTON_PRESS && e->button == 1)
	activate ();
    
    if (e->button == 3)
	popup_menu.popup (e->button, e->time);

    return false;
}

int TopTreeItem::key_press_cb (GdkEventKey *e)
{
    switch (e->keyval)
    {
    case GDK_Return:
    case GDK_KP_Enter:
        menu_edit.emit ();
        break;
        
    case GDK_Menu:
        popup_menu.popup (0, e->time);
        break;
    }
    
    return false;
}

/***********************************
 * FormMoveOp
 ***********************************/

FormMoveOp::FormMoveOp (Resources::Form *form, int new_x_, int new_y_) :
    manager (form->get_manager ()),
    form_id (form->id),
    old_x (form->x), old_y (form->y),
    new_x (new_x_), new_y (new_y_)
{
    char *label_str = g_strdup_printf (_("Move %s"), form_id.c_str ());
    label = label_str;
    g_free (label_str);
}

void FormMoveOp::undo ()
{
    Resource *resource = manager->get_resource (form_id);
    Resources::Form *form = static_cast<Resources::Form*> (resource);
    g_assert (form);

    form->x = old_x;
    form->y = old_y;
}

void FormMoveOp::redo ()
{
    Resource *resource = manager->get_resource (form_id);
    Resources::Form *form = static_cast<Resources::Form*> (resource);
    g_assert (form);
    
    form->x = new_x;
    form->y = new_y;
}
    
