//$Id: io.cc,v 1.5 2004/03/31 17:55:01 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "io/io.h"

#include "io/xml-loader.h"
#include "io/xml-saver.h"
#include "io/rcp-loader.h"
#include "io/rcp-saver.h"

#include <libgnomevfs/gnome-vfs-mime.h>

using namespace Guikachu;

std::string IO::create_canonical_uri (const std::string &filename)
{
    if (filename == "")
	return "";

    if (filename.find ("://") < filename.find ("/"))
	// Don't touch URI's
	return filename;
    
    std::string full_filename;
    if (g_path_is_absolute (filename.c_str ()))
	full_filename = filename;
    else {
	char *current_dirname = g_get_current_dir ();
	full_filename = std::string (current_dirname) + "/" + filename;
	g_free (current_dirname);
    }
    
    char *uri_tmp = gnome_vfs_make_uri_canonical (full_filename.c_str ());
    std::string uri = uri_tmp;
    g_free (uri_tmp);
    
    return uri;
}

IO::IOResult IO::load_uri (const std::string &uri, char *&data, GnomeVFSFileSize &len)
{
    GnomeVFSHandle *vfs_handle = 0;
    GnomeVFSResult  res = gnome_vfs_open (&vfs_handle, uri.c_str (), GNOME_VFS_OPEN_READ);
    
    if (res != GNOME_VFS_OK)
	return res;
        
    GnomeVFSFileInfo *vfs_info = gnome_vfs_file_info_new ();
    res = gnome_vfs_get_file_info_from_handle (
	vfs_handle, vfs_info, GNOME_VFS_FILE_INFO_FOLLOW_LINKS);
        
    if (res != GNOME_VFS_OK)
    {
	gnome_vfs_file_info_unref (vfs_info);
	gnome_vfs_close (vfs_handle);
	return res;
    }
        
    data = new char[vfs_info->size];
    len = 0;        
    char             buffer[10240];
    GnomeVFSFileSize bytes_read;
    
    gnome_vfs_file_info_unref (vfs_info);
    
    while (res == GNOME_VFS_OK)
    {
	res = gnome_vfs_read (vfs_handle, buffer, sizeof buffer - 1, &bytes_read);
	
	if(!bytes_read)
	    break;

	memcpy(data + len, buffer, bytes_read);
	len += bytes_read;
    }
    gnome_vfs_close (vfs_handle);
    
    return GNOME_VFS_OK;
}

IO::IOResult IO::save_uri (const std::string &uri, const char *data, GnomeVFSFileSize len)
{
    // Create file
    GnomeVFSHandle *vfs_handle = 0;
    GnomeVFSResult res = gnome_vfs_create (
	&vfs_handle, uri.c_str (), GNOME_VFS_OPEN_WRITE, 0, 0644);
    if (res != GNOME_VFS_OK)
	return res;
	
    // Write data to VFS stream
    const GnomeVFSFileSize chunk_size = len;
    
    for (GnomeVFSFileSize bytes_written = 0; len; data += bytes_written)
    {
        res = gnome_vfs_write (vfs_handle, data, std::min (chunk_size, len), &bytes_written);
        len -= bytes_written;
        
        if (res != GNOME_VFS_OK)
            break;
    }
    
    if (vfs_handle)
        gnome_vfs_close (vfs_handle);

    return res;
}

std::string IO::get_mime_type (const std::string &uri)
{
#if 0
    GnomeVFSURI *vfs_uri = gnome_vfs_uri_new (uri.c_str ());
    if (!vfs_uri)
	return "";
    
    std::string ret = gnome_vfs_get_mime_type (vfs_uri);
    gnome_vfs_uri_unref (vfs_uri);
    return ret;
#else
    return gnome_vfs_mime_type_from_name ("foo.guikachu");
#endif
}

IO::IOFactory * IO::IOFactory::instance_ = 0;

namespace
{
    template<class T>
    IO::Loader * loader_factory ()
    {
	return new T;
    }

    template<class T>
    IO::Saver * saver_factory ()
    {
	return new T;
    }
    
} // anonymous namespace

IO::IOFactory::IOFactory ()
{
    loader_map["application/x-guikachu"] = loader_factory<IO::XMLLoader>;
    loader_map["text/x-rcp"] = loader_factory<IO::RCPLoader>;

    saver_map["application/x-guikachu"] = saver_factory<IO::XMLSaver>;
    saver_map["text/x-rcp"] = saver_factory<IO::RCPSaver>;
}

IO::IOFactory * IO::IOFactory::instance ()
{
    if (!instance_)
	instance_ = new IOFactory;
	
    return instance_;
}

IO::Loader * IO::IOFactory::create_loader (const std::string &mime_type) const
{
    loader_map_t::const_iterator found = loader_map.find (mime_type);
    if (found == loader_map.end ())
	return 0;

    return found->second ();
}

IO::Saver * IO::IOFactory::create_saver (const std::string &mime_type) const
{
    saver_map_t::const_iterator found = saver_map.find (mime_type);
    if (found == saver_map.end ())
	return 0;

    return found->second ();
}
