//$Id: mainwin.cc,v 1.48 2004/01/26 22:08:38 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"
#include <libgnome/libgnome.h>

#include "mainwin.h"
#include "ui.h"

#include <gtk--/statusbar.h>
#include <gtk--/scrolledwindow.h>

#include <gnome--/dialog.h>
#include <gnome--/about.h>
#include <glade/glade.h>

#include <list>

#include "resource-manager.h"

#include "add-resource.h"
#include "preferences-win.h"
#include "preferences.h"
#include "session.h"

#include <fstream>
#include <sigc++/rettype.h>
#include <sigc++/retbind.h>

using namespace Guikachu;

namespace {

    std::string escape_underscores (const std::string &text);
    
} // Anonymous namespace

GUI::MainWin::MainWin ():
    Gnome::App ("Guikachu", "Guikachu"),
    manager (0),
    filename (""),
    in_exit (false),
    recent_files_num (0)   
{
    using namespace SigC;
    
    /* File menu */
    Gnome::UI::Info m_file[] = {
	Gnome::MenuItems::New    (_("_New project"), _("Create a new project"),
				  slot (this, &MainWin::new_cb)),
    
	Gnome::MenuItems::Open   (slot (this, &MainWin::load_cb)),
	Gnome::MenuItems::Save   (slot (this, &MainWin::save_cb)),
	Gnome::MenuItems::SaveAs (slot (this, &MainWin::save_as_cb)),
	Gnome::UI::Item          (Gnome::UI::Icon (GNOME_STOCK_MENU_CONVERT),
				  _("Export _RCP..."),
				  slot (this, &MainWin::export_cb),
				  _("Create an RCP file for compilation")),
	
	Gnome::UI::Separator     (),
	Gnome::UI::Item          (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_ADD),
				  _("Add _Resource..."),
				  slot (this, &MainWin::add_cb),
				  _("Add new component to current project")),
	
	Gnome::UI::Separator     (),
	Gnome::MenuItems::Exit   (bind (rettype<void>(delete_event.slot()),
					(GdkEventAny*)0))
    };

    /* Edit menu */
    Gnome::UI::Info m_edit[] = {
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_MENU_UNDO),
			 "Undo placeholder"),
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_MENU_REDO),
			 "Redo placeholder"),
	Gnome::UI::Separator     (),
	Gnome::MenuItems::Cut    (SigC::slot (this, &MainWin::cut_cb)),
	Gnome::MenuItems::Copy   (SigC::slot (this, &MainWin::copy_cb)),
	Gnome::MenuItems::Paste  (SigC::slot (this, &MainWin::paste_cb)),
    };
    
    /* Settings menu */
    Gnome::UI::Info m_settings[] = {
	Gnome::MenuItems::Preferences (slot (this, &MainWin::preferences_cb))
    };
    
    /* Help menu */
    Gnome::UI::Info m_help[] = {
	Gnome::MenuItems::About (slot (&UI::show_about))
    };

    /* Create the menu bar structure */
    Gnome::UI::SubTree menubar[] = {
	Gnome::Menus::File     (m_file),
	Gnome::Menus::Edit     (m_edit),
	Gnome::Menus::Settings (m_settings),
	Gnome::Menus::Help     (m_help)
    };    
    create_menus (menubar);

    /* Create status bar */
    set_statusbar (*manage (new Gtk::Statusbar ()));
    install_menu_hints ();

    
    // Create the toolbar
    Gnome::UI::Info toolbar [] = {
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_ADD),
			 _("Add"),
			 slot (this, &MainWin::add_cb),
			 _("Add new component to current project")),
	Gnome::UI::Separator (),
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_NEW),
			 _("New"),
			 slot (this, &MainWin::new_cb),
			 _("New project")),
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_OPEN),
			 _("Open"),
			 slot (this, &MainWin::load_cb),
			 _("Open project")),
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_SAVE),
			 _("Save"),
			 slot (this, &MainWin::save_cb),
			 _("Save project"))
    };
    create_toolbar (toolbar);
    
    // Listen for changes in recent file list
    Preferences::Interface::recent_files_changed.connect (
	slot (this, &MainWin::update_recent_files));
    update_recent_files ();

    // Scrollable viewport
    Gtk::ScrolledWindow *scrollwin = new Gtk::ScrolledWindow ();
    scrollwin->set_policy (GTK_POLICY_AUTOMATIC,
			   GTK_POLICY_AUTOMATIC);
    scrollwin->add_with_viewport (resource_tree);
    scrollwin->show_all ();
    set_contents (*manage (scrollwin));
    set_usize (300, 300);
}

void GUI::MainWin::set_manager (ResourceManager *manager_)
{
    manager = manager_;

    update_title ();

    resource_tree.set_manager (manager);

    // Listen for changes in the undo stack
    manager->get_undo_manager ().changed.connect (slot (this, &MainWin::update_undo));
    update_undo ();

    // Listen for changes in dirty state
    manager->dirty_state_changed.connect (SigC::slot (this, &MainWin::update_title));
}

void GUI::MainWin::update_recent_files ()
{
    // Clear old entries
    if (recent_files_num)
	remove_menu_range ("File/", 7, recent_files_num + 1);

    std::list<std::string> recent_files = Preferences::Interface::get_recent_files ();
    recent_files_num = recent_files.size ();

    if (recent_files_num)
    {
	std::list<Gnome::UI::Info> recent_items;
	recent_items.push_back (Gnome::UI::Separator ());

	std::list<std::string>::const_iterator i;
	int num;
	for (i = recent_files.begin (), num = 0;
	     i != recent_files.end (); i++, num++)
	{
	    std::string vis_filename = escape_underscores (UI::visible_filename (*i));
	    char *menu_label;
	    
	    if (num < 9)
		menu_label = g_strdup_printf ("_%d %s", num + 1, vis_filename.c_str ());
	    else
		menu_label = g_strdup_printf ("%d %s", num + 1, vis_filename.c_str ());
	    
	    recent_items.push_back (
		Gnome::UI::Item (menu_label, SigC::bind (SigC::slot (this, &MainWin::open), *i)));
	    
	    g_free (menu_label);
	}

	insert_menus ("File/Add _Resource...", recent_items);
	//install_menu_hints ();
    }
}

void GUI::MainWin::update_undo ()
{
    UndoManager &undo_manager = manager->get_undo_manager ();
    
    std::list<std::string> undo_labels = undo_manager.get_undo_labels ();
    std::list<std::string> redo_labels = undo_manager.get_redo_labels ();
    
    std::string top_undo_label = undo_labels.size () ?
	escape_underscores (undo_labels.front ()) : _("No operation");
    std::string top_redo_label = redo_labels.size () ?
	escape_underscores (redo_labels.front ()) : _("No operation");
	
    char *undo_label = g_strdup_printf (_("_Undo: %s"), top_undo_label.c_str ());
    char *redo_label = g_strdup_printf (_("_Redo: %s"), top_redo_label.c_str ());

    remove_menu_range ("Edit/", 0, 2);
    
    Gnome::UI::Info undo_items[] = {
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_MENU_UNDO), undo_label,
			 SigC::slot (this, &MainWin::undo_cb)),
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_MENU_REDO), redo_label,
			 SigC::slot (this, &MainWin::redo_cb)),
    };

    g_free (undo_label);
    g_free (redo_label);
    
    Gnome::UI::Array<Gnome::UI::Info> menubar_widgets = insert_menus ("Edit/", undo_items);

    // Disable menu items if undo and/or redo stack is empty
    menubar_widgets[0].get_widget ()->set_sensitive (undo_labels.size ());
    menubar_widgets[1].get_widget ()->set_sensitive (redo_labels.size ());
    
//    install_menu_hints (); // FIXME
}

void GUI::MainWin::undo_cb ()
{
    manager->get_undo_manager ().undo ();
}

void GUI::MainWin::redo_cb ()
{
    manager->get_undo_manager ().redo ();
}

void GUI::MainWin::add_cb ()
{
    AddResourceWin add_win (manager);

    add_win.run (this);
}

void GUI::MainWin::preferences_cb ()
{
    static PreferencesWin pref_win;

    pref_win.run ();
}

void GUI::MainWin::update_title ()
{
    gchar *title_buf;

    if (filename != "")
	title_buf = g_strdup_printf (_("Guikachu: %s%c"),
				     UI::visible_filename (filename).c_str (),
				     manager->is_dirty () ? '*' : '\0');
    else
	title_buf = g_strdup_printf (_("Guikachu: %s%c"), _("<unnamed>"),
				     manager->is_dirty () ? '*' : '\0');
	
    set_title (title_buf);

    g_free (title_buf);
}

namespace {

std::string escape_underscores (const std::string &text)
{
    // Change _ to __ to avoid underscore accelerators

    std::string escaped_text;
    for (std::string::const_iterator i = text.begin (); i != text.end (); ++i)
	if (*i == '_')
	    escaped_text += "__";
	else
	    escaped_text += *i;
	    
    return escaped_text;
}
    
} // Anonymous namespace
