//$Id: storage.cc,v 1.10 2003/06/30 14:07:56 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "storage.h"

#include <libxml/parser.h>
#include <libxml/xmlmemory.h>
#include <glib.h>

using namespace Guikachu;

Storage::Storage () :
    doc (xmlNewDoc ((xmlChar*)"1.0"))
{
}

Storage::Storage (xmlDocPtr doc_) :
    doc (doc_)
{
}

Storage::~Storage ()
{
    if (doc)
	xmlFreeDoc (doc);
}

Storage::Storage (const Storage &other) :
    doc (xmlCopyDoc (other.doc, 1))
{
}

Storage & Storage::operator= (const Storage &other)
{
    if (doc)
	xmlFreeDoc (doc);

    doc = xmlCopyDoc (other.doc, 1);
    return *this;
}

GnomeVFSResult Storage::load (const std::string &filename)
{
    if (doc)
	xmlFreeDoc (doc);
    
    doc = xmlParseFile (filename.c_str ());

    return doc ? GNOME_VFS_OK : GNOME_VFS_ERROR_GENERIC;
}

GnomeVFSResult Storage::load_uri (const std::string &uri)
{
    xmlParserCtxtPtr parser_ctxt = 0;

    GnomeVFSHandle *vfs_handle = 0;
    GnomeVFSResult  res = gnome_vfs_open (&vfs_handle, uri.c_str (), GNOME_VFS_OPEN_READ);

    if (res != GNOME_VFS_OK)
	goto error; // FIXME: Yeah, right

    char              buffer[1024];
    GnomeVFSFileSize  bytes_read;
    
    // The first chunk is needed to grab encoding information
    res = gnome_vfs_read (vfs_handle, buffer, sizeof buffer - 1, &bytes_read);
    if (!bytes_read || res != GNOME_VFS_OK)
    {
	goto error;
    }
    
    parser_ctxt = xmlCreatePushParserCtxt (0, 0, buffer, bytes_read, uri.c_str ());
    
    while (res == GNOME_VFS_OK)
    {
	res = gnome_vfs_read (vfs_handle, buffer, sizeof buffer - 1, &bytes_read);

	xmlParseChunk (parser_ctxt, buffer, bytes_read, 0);
	
	if(!bytes_read)
	    break;
    }
    
    // Last chunk
    xmlParseChunk (parser_ctxt, buffer, 0, 1);
    res = gnome_vfs_close (vfs_handle);
    vfs_handle = 0;
    if (res != GNOME_VFS_OK)
	goto error;

    if (!parser_ctxt->myDoc || !parser_ctxt->wellFormed)
    {
	res = GNOME_VFS_ERROR_WRONG_FORMAT;
	goto error;
    }

    xmlFreeDoc (doc);
    doc = parser_ctxt->myDoc;
    
    xmlFreeParserCtxt (parser_ctxt);

    return GNOME_VFS_OK;

 error:
    if (vfs_handle)
	gnome_vfs_close (vfs_handle);

    if (parser_ctxt)
    {
	if (parser_ctxt->myDoc)
	    xmlFreeDoc (parser_ctxt->myDoc);

	xmlFreeParserCtxt (parser_ctxt);
    }

    return res;
}

GnomeVFSResult Storage::save_uri (const std::string &uri) const
{
    g_assert (doc != 0);

    xmlChar *buffer = 0;
    int      buffer_size;
    
    GnomeVFSHandle *vfs_handle = 0;
    GnomeVFSResult  res = gnome_vfs_create (&vfs_handle, uri.c_str (),
					    GNOME_VFS_OPEN_WRITE, 0, 0644);
    if (res != GNOME_VFS_OK)
	goto error;

    save_buffer (buffer, buffer_size);

    GnomeVFSFileSize bytes_written;
    res = gnome_vfs_write (vfs_handle, buffer, buffer_size, &bytes_written);

    xmlFree (buffer);
    
    if (res != GNOME_VFS_OK)
	goto error;
    
    res = gnome_vfs_close (vfs_handle);

    return res;

 error:
    if (vfs_handle)
	gnome_vfs_close (vfs_handle);

    return res;
}

void Storage::save_buffer (xmlChar *&buffer, int &buffer_size) const
{
    xmlDocDumpMemory (doc, &buffer, &buffer_size);    
}

StorageNode Storage::get_root ()
{
    g_assert (doc != 0);
    
    return StorageNode (xmlDocGetRootElement (doc), this);
}

void Storage::set_root (StorageNode &node)
{
    g_assert (doc != 0);
    
    xmlDocSetRootElement (doc, node.c_node ());
}

StorageNode Storage::create_root (const std::string &name)
{
    g_assert (doc != 0);
    
    xmlNodePtr root_node = xmlNewNode (NULL, (xmlChar*) name.c_str ());
    xmlDocSetRootElement (doc, root_node);

    return StorageNode (root_node, this);
}
