//$Id: bitmap-win.cc,v 1.4 2003/12/23 12:36:40 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "bitmap-win.h"

#include "config.h"
#include <libgnome/libgnome.h>

#ifdef GUIKACHU_USE_WM_ICONS
#include <libgnomeui/gnome-window-icon.h>
#endif

#include <gtk--/menu.h>
#include <gtk--/fileselection.h>
#include <gtk--/main.h>
#include <gtk--/button.h>
#include <gtk--/frame.h>
#include <gtk--/box.h>
#include <gnome--/stock.h>

#include <sigc++/retbind.h>
#include "argumentsink.h"

#include "io/io.h"

#include "property-ops-resource.h"

#include "widgets/propertytable.h"
#include "widgets/entry.h"

#include "form-editor/form-editor-canvas.h"
#include "preferences.h"

#include <gdk-pixbuf/gdk-pixbuf.h>

using namespace Guikachu;

GUI::BitmapWindow::BitmapWindow (Resources::Bitmap *res_):
    res (res_),
    update_block (true)
{
    using namespace SigC;
    using ResourceOps::PropChangeOpFactory;
    using ResourceOps::RenameOpFactory;
    
    window.delete_event.connect (slot (this, &BitmapWindow::delete_event_impl));
    window.set_policy (false, true, false);
    
#ifdef GUIKACHU_USE_WM_ICONS
    gnome_window_icon_set_from_file (window.gtkobj (), GNOME_ICONDIR "/guikachu.png");
#endif

    GUI::PropertyTable *proptable = new GUI::PropertyTable;
    Gtk::Widget *control;

    /* Resource ID */
    control = new GUI::PropertyEditors::Entry (false, res->id, new RenameOpFactory (res));
    proptable->add (_("Resource _ID:"), *manage (control));

    /* Type */
    Gtk::Menu *type_menu = new Gtk::Menu;

#define BITMAP_TYPE(s,t)						\
	    type_menu->items ().push_back (				\
		Gtk::Menu_Helpers::MenuElem (				\
		    s,							\
		    bind (slot (this, &BitmapWindow::type_menu_cb),	\
			  Resources::Bitmap::t)));

    BITMAP_TYPE (_("Monochrome"),            TYPE_MONO);
    BITMAP_TYPE (_("Greyscale (4 shades)"),  TYPE_GREY_4);
    BITMAP_TYPE (_("Greyscale (16 shades)"), TYPE_GREY_16);
    BITMAP_TYPE (_("Color (16 colors)"),     TYPE_COLOR_16);
    BITMAP_TYPE (_("Color (256 colors)"),    TYPE_COLOR_256);

#undef BITMAP_TYPE
    
    type_menu->show_all ();
    type_combo.set_menu (*manage (type_menu));
    proptable->add (_("T_ype:"), type_combo);

    /* Preview of the image */
    Gtk::VBox *vbox = new Gtk::VBox (false, 5);
    Gnome::StockPixmap *stock_pm;
    stock_pm = new Gnome::StockPixmap (GNOME_STOCK_PIXMAP_OPEN);
    Gtk::Button *load_button = Gtk::wrap (
        GTK_BUTTON (gnome_pixmap_button (GTK_WIDGET (manage (stock_pm)->gtkobj ()), _("Load"))));

    stock_pm = new Gnome::StockPixmap (GNOME_STOCK_PIXMAP_SAVE_AS);
    Gtk::Button *export_button = Gtk::wrap (
        GTK_BUTTON (gnome_pixmap_button (GTK_WIDGET (manage (stock_pm)->gtkobj ()), _("Export"))));
    Gtk::HBox *button_box = new Gtk::HBox (false, 5);
    Gtk::Frame *preview_frame = new Gtk::Frame;

    preview_pixmap.set_padding (5, 5);
    preview_frame->add (preview_pixmap);
    
    button_box->pack_start (*manage (load_button));
    button_box->pack_start (*manage (export_button));
    
    vbox->add (*manage (preview_frame));
    vbox->add (*manage (button_box));
    
    load_button->clicked.connect (slot (this, &BitmapWindow::load_image_cb));
    proptable->add (_("Image preview:"), *manage (vbox));
    
    window.add (*manage (proptable));
    
    res->changed.connect (slot (this, &BitmapWindow::update));
    res->get_manager ()->get_target ()->changed.connect (slot (this, &BitmapWindow::update_image));
    update ();
}

void GUI::BitmapWindow::show ()
{
    window.show_all ();
    window.get_window ().raise ();
}

int GUI::BitmapWindow::delete_event_impl (GdkEventAny *e)
{
    window.hide ();
    return true;
}

void GUI::BitmapWindow::update_image ()
{
    Magick::Image image = res->get_image ();
    
    GdkPixbuf *pixbuf = FormEditor::render_bitmap (res->bitmap_type, image);

    if (pixbuf)
    {
        GdkPixmap *pm;
        GdkBitmap *bm;
        
        guint32 background_color = 0xFFFFFF;

        if (!res->get_manager ()->get_target ()->screen_color && res->bitmap_type < Resources::Bitmap::TYPE_COLOR_16)
        {
            Gdk_Color c (Preferences::FormEditor::get_color_bg ());
            background_color = ((c.red >> 8) << 16) + ((c.green >> 8) << 8) + (c.blue  >> 8);
        }
        
        GdkPixbuf *pixbuf_with_background = gdk_pixbuf_composite_color_simple (
            pixbuf, gdk_pixbuf_get_width (pixbuf), gdk_pixbuf_get_height (pixbuf),
            GDK_INTERP_NEAREST, 255, 1, background_color, background_color);
        gdk_pixbuf_unref (pixbuf);

        gdk_pixbuf_render_pixmap_and_mask (pixbuf_with_background, &pm, &bm, 127);
        preview_pixmap.set (pm, bm);
    
        gdk_pixbuf_unref (pixbuf_with_background);
    }
}

void GUI::BitmapWindow::update ()
{
    // Update window title
    gchar *title_buf = g_strdup_printf (_("Bitmap: %s"), res->id ().c_str ());
    window.set_title (title_buf);
    g_free (title_buf);

    update_image ();
    
    update_block = true;    
    type_combo.set_history (res->bitmap_type);
    update_block = false;
}

void GUI::BitmapWindow::type_menu_cb (Resources::Bitmap::BitmapType type)
{
    if (res->bitmap_type == type)
	return;

    char *label_str = g_strdup_printf (_("Change type of %s"), res->id ().c_str ());
    
    UndoOp *op = new ResourceOps::PropChangeOp<Resources::Bitmap::BitmapType> (
	label_str, res, res->bitmap_type, type, false);
    g_free (label_str);
    
    res->bitmap_type = type;

    res->get_manager ()->get_undo_manager ().push (op);
}

namespace
{
    class ImageFileSelector: public SigC::Object
    {
        Gtk::FileSelection fs;
        Resources::Bitmap  *res;

    public:
        ImageFileSelector (Gtk::Window &parent, Resources::Bitmap *res);
        void run ();
        
    private:
        void close_cb ();
        void open_cb ();
    };
} // Anonymous namespace

ImageFileSelector::ImageFileSelector (Gtk::Window &parent, Resources::Bitmap *res_) :
    res (res_)
{
    fs.hide_fileop_buttons ();
    fs.set_title (_("Select image file to load"));
    
    fs.get_ok_button ()->clicked.connect (SigC::slot (this, &ImageFileSelector::open_cb));
    fs.get_cancel_button ()->clicked.connect (SigC::slot (this, &ImageFileSelector::close_cb));
    fs.delete_event.connect (
	SigC::retbind (
	    SigC::hide<void, GdkEventAny*> (SigC::slot (this, &ImageFileSelector::close_cb)),
	    1));
    
    fs.set_transient_for (parent);
    fs.set_modal (true);
}

void ImageFileSelector::close_cb ()
{
    fs.hide ();
    Gtk::Main::quit ();
}

void ImageFileSelector::open_cb ()
{
    std::string uri = IO::create_canonical_uri (fs.get_filename ());
    res->load_file (uri);
    
    close_cb ();
}

void ImageFileSelector::run ()
{
    fs.show ();
    Gtk::Main::run ();
}

void GUI::BitmapWindow::load_image_cb ()
{
    ImageFileSelector ifs (window, res);
    ifs.run ();
}
