//$Id: storage-node.cc,v 1.23 2003/06/30 14:07:56 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "storage-node.h"

#include "config.h"
#include <libxml/xmlmemory.h>
#include <glib.h>
#include <libgnome/libgnome.h> // for gnome_i18n_get_language_list

#include "storage.h"

using namespace Guikachu;

StorageNode::StorageNode ():
    free_node (true),
    node_ptr (xmlNewNode (0, (xmlChar*) "")),
    parent_doc (0)
{
}

StorageNode::~StorageNode ()
{
    if (free_node)
	xmlFree (node_ptr);
}

StorageNode::StorageNode (xmlNodePtr node, Storage *parent_doc_):
    free_node (false),
    node_ptr (node),
    parent_doc (parent_doc_)
{   
}

StorageNode StorageNode::children () const
{
    return StorageNode (node_ptr->xmlChildrenNode, parent_doc);
}

StorageNode StorageNode::operator++ (int)
{
    StorageNode tmp = *this;
    node_ptr = next ().c_node ();
    return tmp;
}

StorageNode& StorageNode::operator++ ()
{
    node_ptr = next ().c_node ();
    return *this;
}

std::string StorageNode::name () const
{
    if (!node_ptr->name)
	return "";
    
    return (char*)(node_ptr->name);
}

StorageNode StorageNode::add_node (const std::string &name,
				   const std::string &content)
{
    g_assert (node_ptr != 0);
    
    xmlNodePtr new_node_ptr = 0;

    if (content != "")
	new_node_ptr = xmlNewChild (node_ptr, 0,
				    (xmlChar*)name.c_str (),
				    (xmlChar*)content.c_str ());
    else
	new_node_ptr = xmlNewChild (node_ptr, 0,
				    (xmlChar*)name.c_str (), 0);

    return StorageNode (new_node_ptr, parent_doc);
}

void StorageNode::set_content (const std::string &content)
{
    g_assert (node_ptr != 0);

    xmlNodeSetContent (node_ptr, (xmlChar*) content.c_str ());
}

std::string StorageNode::get_content_raw () const
{
    g_assert (node_ptr != 0);
    
    std::string ret;
    char *tmp = (char*)xmlNodeGetContent (node_ptr);
    if (tmp)
    {
	ret = tmp;
	g_free (tmp);
    } else {
	return "";
    }

    return ret;
}

namespace
{
    bool escape_string (std::string &raw_str)
    {
	std::string converted_str;
	bool converted = false;
	
	for (std::string::const_iterator i = raw_str.begin (); i != raw_str.end (); ++i)
	{
	    if ((signed char)*i < 0)
	    {
		converted = true;
		
		char *tmp = g_strdup_printf ("\\x%X", (unsigned char)*i);
		converted_str += tmp;
		g_free (tmp);
	    } else {
		converted_str += *i;
	    }
	}

	if (converted)
	    raw_str = converted_str;

	return converted;
    }
}

std::string StorageNode::get_content () const
{
    std::string ret_val = get_content_raw ();

    if (escape_string (ret_val) && parent_doc)
	parent_doc->encoding_notify.emit ();
    
    return ret_val;
}


void StorageNode::set_prop (const std::string &prop_name,
			    const std::string &value)
{
    g_assert (node_ptr != 0);

    xmlSetProp (node_ptr,
		(xmlChar*) prop_name.c_str (),
		(xmlChar*) value.c_str ());
}

void StorageNode::set_prop (const std::string &prop_name,
			    unsigned int       value)
{
    g_assert (node_ptr != 0);

    char *tmp = g_strdup_printf ("%d", value);
    set_prop (prop_name, tmp);
    g_free (tmp);
}

void StorageNode::set_prop (const std::string &prop_name,
			    int                value)
{
    g_assert (node_ptr != 0);
    
    char *tmp = g_strdup_printf ("%d", value);
    set_prop (prop_name, tmp);
    g_free (tmp);
}

void StorageNode::set_prop (const std::string &prop_name,
			    char               value)
{
    g_assert (node_ptr != 0);

    if (!value)
	return;
    
    static char *buffer = g_new0 (char, 2);

    buffer[0] = value;
    set_prop (prop_name, buffer);
}

std::string StorageNode::get_prop_string_raw (const std::string &prop_name) const
{
    g_assert (node_ptr != 0);

    std::string ret;
    
    xmlChar *xml_tmp = xmlGetProp (node_ptr, (xmlChar*) prop_name.c_str ());

    if (xml_tmp)
    {
	ret = (char*) xml_tmp;
	xmlFree (xml_tmp);
    }
    
    return ret;
}

std::string StorageNode::get_prop_string (const std::string &prop_name) const
{
    std::string ret_val = get_prop_string_raw (prop_name);
    
    if (escape_string (ret_val) && parent_doc)
	parent_doc->encoding_notify.emit ();
    
    return ret_val;
}

int StorageNode::get_prop_int (const std::string &prop_name) const
{
    g_assert (node_ptr != 0);

    int ret;

    xmlChar *xml_tmp = xmlGetProp (node_ptr, (xmlChar*) prop_name.c_str ());
    if (xml_tmp)
    {
	ret = atoi ((char*) xml_tmp);
	xmlFree (xml_tmp);
    } else {
	ret = 0;
    }

    return ret;
}

char StorageNode::get_prop_char (const std::string &prop_name) const
{
    g_assert (node_ptr != 0);

    char ret = 0;

    xmlChar *xml_tmp = xmlGetProp (node_ptr, (xmlChar*) prop_name.c_str ());
    if (xml_tmp)
    {
	if (strlen ((char*) xml_tmp))
	    ret = (char) xml_tmp[0];
	
	xmlFree (xml_tmp);
    }

    return ret;
}

StorageNode StorageNode::next () const
{
    return StorageNode (node_ptr->next, parent_doc);
}

StorageNode StorageNode::get_translated_child (const std::string &name) const
{
    // This one is lifted from GAL. It's pretty smart, actually.
    
    StorageNode best_node (0, parent_doc);

    GList *lang_list = gnome_i18n_get_language_list ("LC_MESSAGES");
    
    int best_score = INT_MAX;
    
    for (StorageNode node = children (); node; node++)
    {
	if (node.name () != name)
	    continue;

	xmlChar *lang = xmlNodeGetLang (node.c_node ());

	if (lang != 0)
	{
	    GList *lang_list_i;
	    int score;
	    
	    for (lang_list_i = lang_list, score = 0;
		 lang_list_i != 0 && score < best_score;
		 lang_list_i = lang_list_i->next, score++)
	    {
		if (strcmp ((char *)lang_list_i->data, (char*)lang) == 0)
		{
		    best_node = node;
		    best_score = score;
		}
	    }
	} else {
	    if (!best_node)
		best_node = node;
	}

	xmlFree (lang);

	if (best_score == 0) // It won't get better than that
	    return best_node;
    }

    return best_node;
}
