//$Id: form-editor-canvas.cc,v 1.2 2003/06/30 14:07:57 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "form-editor/form-editor-canvas.h"
#include "form-editor/form-editor.h"

#include <gnome--/canvas.h>
#include <gnome--/canvas-group.h>
#include <gnome--/canvas-text.h>
#include <gnome--/canvas-polygon.h>

#include <gdk-pixbuf/gnome-canvas-pixbuf.h>

#include <list>

using namespace Guikachu::GUI;
using Guikachu::GUI::FormEditor::Font;
using Guikachu::GUI::FormEditor::Glyph;

namespace
{
    GdkPixbuf* clip_pixbuf (GdkPixbuf    *src,
			    GtkAnchorType anchor,
			    int           clip_width,
			    int           clip_height);
    
    typedef std::list<const Glyph*> typeset_t;
    
    GdkPixbuf * render_text_impl (const Font &font,
				  const std::string &text,
				  const Gdk_Color   &color);
    
    GdkPixbuf * render_text_multi_line_impl (const Font        &font,
					     const std::string &text,
					     const Gdk_Color   &color);

    void render_raw_line (const typeset_t &typeset,
			  const Gdk_Color &color,
			  guint8          *target,
			  int              target_rowstride);
}

Gnome::CanvasItem* FormEditor::draw_text (Gnome::CanvasGroup &group,
					  const std::string &text, int font_num,
					  int x, int y,
					  bool usable,
					  bool multi_line,
					  GtkAnchorType anchor,
					  int clip_width, int clip_height)
{
    const Font &font = get_font (font_num);
    Gdk_Color color = get_foreground_color (usable);

    return render_text (group, text, font, color,
			x, y,
			multi_line,
			anchor,
			clip_width, clip_height);
}

Gnome::CanvasItem* FormEditor::render_text (Gnome::CanvasGroup &group,
					    const std::string  &text,
					    const Font         &font,
					    const Gdk_Color    &color,
					    int x, int y,
					    bool                multi_line,
					    GtkAnchorType       anchor,
					    int clip_width, int clip_height)
{
    Gnome::CanvasGroup *text_group = new Gnome::CanvasGroup (group, 0, 0);
    
    GdkPixbuf *pixbuf = multi_line ?
	render_text_multi_line_impl (font, text, color) : render_text_impl (font, text, color);

    if (!pixbuf)
	return 0;
    
    GdkPixbuf *clipped_pixbuf;
    if (clip_width || clip_height)
	clipped_pixbuf = clip_pixbuf (pixbuf, anchor, clip_width, clip_height);
    else
	clipped_pixbuf = gdk_pixbuf_ref (pixbuf);
    gdk_pixbuf_unref (pixbuf);
    
    int width = gdk_pixbuf_get_width (clipped_pixbuf);
    int height = gdk_pixbuf_get_height (clipped_pixbuf);

    // Horizontal offset
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_SOUTH_WEST:
	break;
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_SOUTH:
	x -= width / 2;
	break;
    case GTK_ANCHOR_NORTH_EAST:
    case GTK_ANCHOR_EAST:
    case GTK_ANCHOR_SOUTH_EAST:
	x -= width;
    }

    // Vertical offset
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_NORTH_EAST:
	break;
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_EAST:
	y -= height / 2;
	break;
    case GTK_ANCHOR_SOUTH_WEST:
    case GTK_ANCHOR_SOUTH:
    case GTK_ANCHOR_SOUTH_EAST:
	y -= height;
	break;
    }

    // Draw the text
    draw_pixbuf (*text_group, clipped_pixbuf, x, y);
    gdk_pixbuf_unref (clipped_pixbuf);

    // Draw a transparent background block (to get mouse events for
    // the whole area)
    FormEditor::draw_rectangle (*text_group,
				x, y,
				x + width, y + height);
    
    return text_group;
}

Gnome::CanvasItem* FormEditor::draw_pixbuf (Gnome::CanvasGroup &group,
					    GdkPixbuf          *pixbuf,
					    int                 x,
					    int                 y)
{
    return Gtk::wrap (gnome_canvas_item_new (group.gtkobj (),
					     gnome_canvas_pixbuf_get_type (),
					     "pixbuf", pixbuf,
					     "x", (double) x,
					     "y", (double) y,
					     0));
}

namespace
{

GdkPixbuf* clip_pixbuf (GdkPixbuf     *src,
			GtkAnchorType  anchor,
			int            clip_width,
			int            clip_height)
{
    GdkPixbuf *dest;
    
    int width = gdk_pixbuf_get_width (src);
    int height = gdk_pixbuf_get_height (src);
    
    if (clip_width <= 0 || clip_width > width)
	clip_width = width;
    
    if (clip_height <= 0 || clip_height > height)
	clip_height = height;
    
    int src_x = 0;
    int src_y = 0;
	
    /* Set src_x */
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_SOUTH_WEST:
	src_x = 0;
	break;
	
    case GTK_ANCHOR_NORTH_EAST:
    case GTK_ANCHOR_EAST:
    case GTK_ANCHOR_SOUTH_EAST:
	src_x = width - clip_width;
	break;
	
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_SOUTH:
	src_x = (width / 2) - (clip_width / 2);
	break;
    }
    
    /* Set src_y */
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_NORTH_EAST:
	src_y = 0;
	break;
	
    case GTK_ANCHOR_SOUTH_WEST:
    case GTK_ANCHOR_SOUTH:
    case GTK_ANCHOR_SOUTH_EAST:
	src_y = height - clip_height;
	break;
	
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_EAST:
	src_y = (height / 2) - (clip_height / 2);
	break;
    }
    
    dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, true, 8,
			   clip_width, clip_height);
    
    gdk_pixbuf_copy_area (src,
			  src_x, src_y,
			  clip_width, clip_height,
			  dest, 0, 0);
    
    return dest;
}
    
// Get next character from a string, with escape codes resolved
static char next_char (const std::string &text,
		       std::string::const_iterator &i)
{
    // Not an escape sequence
    if (*i != '\\')
	return *(i++);

    // Trailing '\'
    if (++i == text.end ())
	return '\\';

    // Check escape type
    char escape_type = *i;

    // Simple escape sequences
    switch (escape_type)
    {
    case 'a':
	i++;
	return '\x07';
    case 'b':
	i++;
	return '\x08';
    case 'e':
	i++;
	// FIXME: Figure out the value of '\e'
	return 'e';
    case 'f':
	i++;
	return '\x0c';
    case 'n':
	i++;
	return '\x0a';
    case 'r':
	i++;
	return '\n';
    case 't':
	i++;
	return '\x09';
    case 'v':
	i++;
	return '\x0b';
    case 'z':
	i++;
	// FIXME: Figure out the value of '\z'
	return 'z';
    }

#define IS_OCTAL(c)  ((c) >= '0' && (c) <= '7')
#define GET_OCTAL(c) ((c) - '0')

#define IS_HEX(c)  (((c) >= '0' && (c) <= '9') ||	\
		    ((c) >= 'a' && (c) <= 'f') ||	\
		    ((c) >= 'A' && (c) <= 'F'))

#define GET_HEX(c) (((c) >= '0' && (c) <= '9') ?	\
		     (c) - '0' :			\
		     ((c) >= 'a' && (c) <= 'f') ?	\
		      (c) - 'a' + 10 : (c) - 'A' + 10)
    
    
    // Octal numbers
    if (IS_OCTAL (escape_type))
    {
	char octal_value = 0;
	
	// Get next at most three octal numbers
	for (int j = 0; j < 3 && i != text.end (); i++, j++)
	{
	    if (!IS_OCTAL (*i))
		// Not an octal number
		break;

	    octal_value = octal_value * 8 + GET_OCTAL (*i);
	}

	return octal_value;
    }

    // Hexadecimal numbers
    if (escape_type == 'x')
    {
	char hex_value = 0;

	// Skip leading 0's
	while (*(++i) == '0' && i != text.end ());
	
	// Read at most two hexadecimal characters
	for (int j = 0; j < 2 && i != text.end (); i++, j++)
	{
	    if (!IS_HEX (*i))
		break;

	    hex_value = hex_value * 16 + GET_HEX (*i);
	}

	return hex_value;
    }

    // Unknown escape sequence: return escaped character
    return *(i++);
}

void set_pixel (guint8          *pixel,
		const Gdk_Color &color)
{
    pixel[0] = color.red   >> 8;
    pixel[1] = color.green >> 8;
    pixel[2] = color.blue  >> 8;
    pixel[3] = 255;
}

void free_pixdata_gfree (guint8 *pixels, gpointer data)
{
    g_free (pixels);
}

void render_raw_line (const Font      &font,
		      const typeset_t &typeset,
		      const Gdk_Color &color,
		      guint8          *target,
		      int              target_rowstride)
{
    int line_height = font.get_line_height ();
    
    for (typeset_t::const_iterator i = typeset.begin (); i != typeset.end (); i++)
    {
	const Glyph *glyph = *i;
	
	bool **curr_bitmap = glyph->bitmap;
	guint8 *curr_row = target;
	
	for (int y = 0; y < line_height; y++)
	{
	    guint8 *curr_pixel = curr_row;
	    
	    for (int x = 0; x < glyph->width; x++)
		{
		    if (curr_bitmap[y][x])
			set_pixel (curr_pixel, color);
		    
		    curr_pixel += 4;
		}
		
		curr_row += target_rowstride;
	    }
	    
	    target += glyph->width * 4;
	}
    }
    
GdkPixbuf* render_text_impl (const Font        &font,
			     const std::string &text,
			     const Gdk_Color   &color)
{
    if (text == "")
	return 0;

    int width = 0;
    int height = font.get_line_height ();
    typeset_t typeset;
    
    std::string::const_iterator i = text.begin ();
    while (i != text.end ())
    {
	char c = next_char (text, i);
	const Glyph &g = font.get_glyph (c);
	
	typeset.push_back (&g);
	width += g.width;
    }

    guint8 *pixbuf_data = g_new0 (guint8, width * height * 4);
    render_raw_line (font, typeset, color, pixbuf_data, width * 4);
    
    return gdk_pixbuf_new_from_data (pixbuf_data,
				     GDK_COLORSPACE_RGB,
				     true,
				     8,
				     width, height,
				     width * 4,
				     free_pixdata_gfree, 0);
}

GdkPixbuf* render_text_multi_line_impl (const Font        &font,
					const std::string &text,
					const Gdk_Color   &color)
{
    if (text == "")
	return 0;

    std::list <typeset_t> typesets;
    int width = 0;
    int height = 0;

    int current_width = 0;
    std::list <const Glyph*> current_typeset;
    int line_height = font.get_line_height ();
    
    std::string::const_iterator i = text.begin ();
    while (i != text.end ())
    {
	char c = next_char (text, i);
	if (c != '\n')
	{
	    const Glyph &g = font.get_glyph (c);
	    
	    current_typeset.push_back (&g);
	    current_width += g.width;
	} else {
	    typesets.push_back (current_typeset);
	    current_typeset.clear ();

	    width = std::max<int> (width, current_width);
	    current_width = 0;
	    height += line_height;
	}
    }
    
    // Add last line
    typesets.push_back (current_typeset);
    width = std::max<int> (width, current_width);
    height += font.get_line_height ();
    
    guint8 *pixbuf_data = g_new0 (guint8, width * height * 4);

    int offset = 0;
    for (std::list <typeset_t>::const_iterator i = typesets.begin ();
	 i != typesets.end (); i++)
    {
	render_raw_line (font, *i, color, pixbuf_data + offset, width * 4);

	offset += (line_height * width) * 4;
    }
    
    return gdk_pixbuf_new_from_data (pixbuf_data,
				     GDK_COLORSPACE_RGB,
				     true,
				     8,
				     width, height,
				     width * 4,
				     free_pixdata_gfree, 0);
}

} // anonymous namespace
    
Gnome::CanvasItem* FormEditor::draw_rectangle (Gnome::CanvasGroup &group,
					       int x1, int y1,
					       int x2, int y2)
{
    Gnome::CanvasPoints points;
    using Gnome::Art::Point;

    points.push_back (Point (x1, y1));
    points.push_back (Point (x2, y1));
    points.push_back (Point (x2, y2));
    points.push_back (Point (x1, y2));
    
    return new Gnome::CanvasPolygon (group, points);
}
