//$Id: edit-cut-and-paste.cc,v 1.4 2003/01/28 19:09:56 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "edit-cut-and-paste.h"

#include "resource-util.h"
#include "form-editor/widget-util.h"
#include "io/xml-loader.h"
#include "io/xml-saver.h"
#include "storage.h"
#include "edit-ops.h"

#include <libxml/parser.h>
#include <libxml/xmlmemory.h>

#include <gtk--/window.h>

using namespace Guikachu;

// Some awful hacks in the implementation -- but it's the price to pay
// for an interface that is very nice and easy to use and hides the
// asynchronity of the X selection mechanism

namespace
{
    
    const char *TARGET_RES    = "application/x-guikachu/resources";
    const char *TARGET_WIDGET = "application/x-guikachu/widgets";
    
    inline GdkAtom get_resource_target () { return gdk_atom_intern (TARGET_RES, false);    }
    inline GdkAtom get_widget_target ()   { return gdk_atom_intern (TARGET_WIDGET, false); }
    inline GdkAtom get_clipboard ()       { return gdk_atom_intern ("CLIPBOARD", false);   }
    
    struct
    {
	GdkAtom  target;
	xmlChar *buffer;
	int      length;
	
	void reset () {
	    xmlFree (buffer);
	    target = 0;
	    buffer = 0;
	    length = 0;
	}
    } clipboard_contents = { 0, 0, 0 };
    
    union
    {
	ResourceManager *manager;
	Resources::Form *form;
    } clipboard_state;
    
    void selection_received_cb (GtkSelectionData *data, guint format);
    void selection_requested_cb (GtkSelectionData *data, guint format, guint sel_closure);
    
    void paste_resources_impl (GtkSelectionData *data);
    void paste_widgets_impl   (GtkSelectionData *data);
    
    Gtk::Window * clipboard_window ();
    
} // anonymous namespace



void Edit::copy_resource (Resource *resource)
{
    // Create a new XML document in memory
    Storage storage;

    // The root node contains the type of the resource and its ID
    StorageNode root_node = storage.create_root ("guikachu-resources");

    std::string type_id = Resources::type_id_from_type (resource->get_type ());
    StorageNode resource_node = root_node.add_node (type_id);
    resource_node.set_prop ("id", resource->id);

    // Save resource data to document
    IO::XML::ResourceSaver saver (resource_node);
    resource->apply_visitor (saver);    

    // Serialize document into a string
    clipboard_contents.reset ();
    storage.save_buffer (clipboard_contents.buffer, clipboard_contents.length);
    
    // Set auxillary window as selection's owner
    GdkAtom target_atom = get_resource_target ();
    GdkAtom selection = get_clipboard ();

    clipboard_contents.target = target_atom;
    
    gtk_selection_add_target (GTK_WIDGET (clipboard_window ()->gtkobj ()),
			      selection, target_atom, 0);
    gtk_selection_owner_set (GTK_WIDGET (clipboard_window ()->gtkobj ()),
			     selection, GDK_CURRENT_TIME);
}

void Edit::copy_widget (Widget *widget)
{
    Storage storage;

    StorageNode root_node = storage.create_root ("guikachu-widgets");

    std::string type_id = Widgets::type_id_from_type (widget->get_type ());
    StorageNode widget_node = root_node.add_node (type_id);
    widget_node.set_prop ("id", widget->id);

    IO::XML::WidgetSaver saver (widget_node);
    widget->apply_visitor (saver);

    clipboard_contents.reset ();
    storage.save_buffer (clipboard_contents.buffer, clipboard_contents.length);
    
    GdkAtom target_atom = get_widget_target ();
    GdkAtom selection = get_clipboard ();

    clipboard_contents.target = target_atom;
    
    gtk_selection_add_target (GTK_WIDGET (clipboard_window ()->gtkobj ()),
			      selection, target_atom, 0);
    gtk_selection_owner_set (GTK_WIDGET (clipboard_window ()->gtkobj ()),
			     selection, GDK_CURRENT_TIME);
}

void Edit::copy_widgets (const std::set<Widget*> &widgets)
{
    Storage storage;

    StorageNode root_node = storage.create_root ("guikachu-widgets");

    for (std::set<Widget*>::const_iterator i = widgets.begin ();
	 i != widgets.end (); ++i)
    {
	Widget *widget = *i;
	
	std::string type_id = Widgets::type_id_from_type (widget->get_type ());
	StorageNode widget_node = root_node.add_node (type_id);
	widget_node.set_prop ("id", widget->id);

	IO::XML::WidgetSaver saver (widget_node);
	widget->apply_visitor (saver);
    }
    
    clipboard_contents.reset ();
    storage.save_buffer (clipboard_contents.buffer, clipboard_contents.length);
    
    GdkAtom target_atom = get_widget_target ();
    GdkAtom selection = get_clipboard ();

    clipboard_contents.target = target_atom;
    
    gtk_selection_add_target (GTK_WIDGET (clipboard_window ()->gtkobj ()),
			      selection, target_atom, 0);
    gtk_selection_owner_set (GTK_WIDGET (clipboard_window ()->gtkobj ()),
			     selection, GDK_CURRENT_TIME);
}



void Edit::paste_resources (ResourceManager *manager)
{
    clipboard_state.manager = manager;

    // Request selection
    GdkAtom target_atom = get_resource_target ();
    GdkAtom selection = get_clipboard ();

    gtk_selection_convert (GTK_WIDGET (clipboard_window ()->gtkobj ()),
			   selection, target_atom, GDK_CURRENT_TIME);
}

void Edit::paste_widgets (Resources::Form *form)
{
    clipboard_state.form = form;

    // Request selection
    GdkAtom target_atom = get_widget_target ();
    GdkAtom selection = get_clipboard ();

    gtk_selection_convert (GTK_WIDGET (clipboard_window ()->gtkobj ()),
			   selection, target_atom, GDK_CURRENT_TIME);
}



namespace
{
    
Gtk::Window * clipboard_window ()
{
    static Gtk::Window *window = 0;
    
    if (!window)
    {
	window = new Gtk::Window;
	window->selection_received.connect (SigC::slot (selection_received_cb));
	window->selection_get.connect (SigC::slot (selection_requested_cb));
    }
    
    return window;
}
    
void selection_requested_cb (GtkSelectionData *data, guint format, guint sel_closure)
{
    if (data->target != clipboard_contents.target)
    {
	data->length = 0;
	return;
    }

    gtk_selection_data_set (data, clipboard_contents.target, 8,
			    (guchar*) clipboard_contents.buffer,
			    clipboard_contents.length);
}

void selection_received_cb (GtkSelectionData *data, guint format)
{
    if (!data->length)
	return;
    
    if (data->target == get_resource_target ())
	paste_resources_impl (data);
    else if (data->target == get_widget_target ())
	paste_widgets_impl (data);
}

void paste_resources_impl (GtkSelectionData *data)
{
    // TODO: Check format and everything
    
    // Read XML from X selection and parse it
    xmlParserCtxtPtr parser_ctxt =
	xmlCreatePushParserCtxt (0, 0, (char*)data->data, data->length, "");
    xmlParseChunk (parser_ctxt, (char*)data->data, 0, 1);
    
    xmlDocPtr doc = parser_ctxt->myDoc;
    
    xmlFreeParserCtxt (parser_ctxt);

    if (!doc) // Parsing failed
    {
	g_warning ("X clipboard contains unparsable junk");
	return;
    }
    
    Storage storage (doc);

    StorageNode root_node = storage.get_root ();
    if (root_node.name () != "guikachu-resources")
    {
	g_warning ("X clipboard contains incorrect content");
	return;
    }

    std::list<Resource*> created_resources;
	
    for (StorageNode resource_node = root_node.children (); resource_node; ++resource_node)
    {
	// Create resource
	std::string type_id = resource_node.name ();
	Resources::Type type = Resources::type_from_type_id (type_id);
	std::string id = resource_node.get_prop_string ("id");
	
	Resource *resource = clipboard_state.manager->create_resource (type, id, true);
	g_assert (resource);
	
	// Load resource data from parsed selection
	IO::XML::ResourceLoader loader (resource_node);
	resource->apply_visitor (loader);

	created_resources.push_back (resource);
    }

    if (!created_resources.size ())
	return;

    if (created_resources.size () == 1)
    {
	UndoOp *op = new Edit::PasteResourceOp (*(created_resources.begin ()));
	clipboard_state.manager->get_undo_manager ().push (op);
    } else {
	UndoOp *op = new Edit::PasteResourcesOp (created_resources);
	clipboard_state.manager->get_undo_manager ().push (op);
    }
}

void paste_widgets_impl (GtkSelectionData *data)
{
    // TODO: Check format and everything
    
    // Read XML from X selection and parse it
    xmlParserCtxtPtr parser_ctxt =
	xmlCreatePushParserCtxt (0, 0, (char*)data->data, data->length, "");
    xmlParseChunk (parser_ctxt, (char*)data->data, 0, 1);
    
    xmlDocPtr doc = parser_ctxt->myDoc;
    
    xmlFreeParserCtxt (parser_ctxt);

    if (!doc) // Parsing failed
    {
	g_warning ("X clipboard contains unparsable junk");
	return;
    }
    
    Storage storage (doc);

    StorageNode root_node = storage.get_root ();
    if (root_node.name () != "guikachu-widgets")
    {
	g_warning ("X clipboard contains incorrect content");
	return;
    }

    std::list<Widget*> created_widgets;
	
    for (StorageNode widget_node = root_node.children (); widget_node; ++widget_node)
    {
	std::string type_id = widget_node.name ();
	Widgets::Type type = Widgets::type_from_type_id (type_id);
	std::string id = widget_node.get_prop_string ("id");
	
	Widget *widget = clipboard_state.form->create_widget (type, id, true);
	g_assert (widget);
	
	// Load resource data from parsed selection
	IO::XML::WidgetLoader loader (widget_node);
	widget->apply_visitor (loader);

	created_widgets.push_back (widget);
    }

    if (!created_widgets.size ())
	return;

    if (created_widgets.size () == 1)
    {
	UndoOp *op = new Edit::PasteWidgetOp (*(created_widgets.begin ()));
	clipboard_state.form->get_manager ()->get_undo_manager ().push (op);
    } else {
	UndoOp *op = new Edit::PasteWidgetsOp (created_widgets);
	clipboard_state.form->get_manager ()->get_undo_manager ().push (op);
    }
}

} // anonymous namespace
