//$Id: font.cc,v 1.20 2002/08/06 23:48:35 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "font.h"

#include <sstream>
#include <fstream>
#include <stdio.h> /* For sscanf */
#include <list>

namespace {
// Get next character from a string, with escape codes resolved
static char next_char (const std::string &text,
		       std::string::const_iterator &i)
{
    // Not an escape sequence
    if (*i != '\\')
	return *(i++);

    // Trailing '\'
    if (++i == text.end ())
	return '\\';

    // Check escape type
    char escape_type = *i;

    // Simple escape sequences
    switch (escape_type)
    {
    case 'a':
	i++;
	return '\x07';
    case 'b':
	i++;
	return '\x08';
    case 'e':
	i++;
	// FIXME: Figure out the value of '\e'
	return 'e';
    case 'f':
	i++;
	return '\x0c';
    case 'n':
	i++;
	return '\x0a';
    case 'r':
	i++;
	return '\n';
    case 't':
	i++;
	return '\x09';
    case 'v':
	i++;
	return '\x0b';
    case 'z':
	i++;
	// FIXME: Figure out the value of '\z'
	return 'z';
    }

#define IS_OCTAL(c)  ((c) >= '0' && (c) <= '7')
#define GET_OCTAL(c) ((c) - '0')

#define IS_HEX(c)  (((c) >= '0' && (c) <= '9') ||	\
		    ((c) >= 'a' && (c) <= 'f') ||	\
		    ((c) >= 'A' && (c) <= 'F'))

#define GET_HEX(c) (((c) >= '0' && (c) <= '9') ?	\
		     (c) - '0' :			\
		     ((c) >= 'a' && (c) <= 'f') ?	\
		      (c) - 'a' + 10 : (c) - 'A' + 10)
    
    
    // Octal numbers
    if (IS_OCTAL (escape_type))
    {
	char octal_value = 0;
	
	// Get next at most three octal numbers
	for (int j = 0; j < 3 && i != text.end (); i++, j++)
	{
	    if (!IS_OCTAL (*i))
		// Not an octal number
		break;

	    octal_value = octal_value * 8 + GET_OCTAL (*i);
	}

	return octal_value;
    }

    // Hexadecimal numbers
    if (escape_type == 'x')
    {
	char hex_value = 0;

	// Skip leading 0's
	while (*(++i) == '0' && i != text.end ());
	
	// Read at most two hexadecimal characters
	for (int j = 0; j < 2 && i != text.end (); i++, j++)
	{
	    if (!IS_HEX (*i))
		break;

	    hex_value = hex_value * 16 + GET_HEX (*i);
	}

	return hex_value;
    }

    // Unknown escape sequence: return escaped character
    return *(i++);
}

void set_pixel (guint8          *pixel,
		const Gdk_Color &color)
{
    pixel[0] = color.red   >> 8;
    pixel[1] = color.green >> 8;
    pixel[2] = color.blue  >> 8;
    pixel[3] = 255;
}

} // Anonymous namespace
    
using namespace Guikachu::GUI::FormEditor;

Font::Font (const std::string &filename):
    line_height (0)
{
    parse (filename);
}

Font::Font ():
    line_height (0)
{
}

void Font::parse (const std::string &filename)
{
    std::ifstream stream ((std::string (GUIKACHU_FONTDIR) + "/" + filename).c_str ());

    parse (stream);
}

void Font::parse (std::istream &stream)
{
    /* FIXME: Error handling, robustness issues regarding garbage
     * input */
    
    std::string curr_line;
    bool in_header = true;
    
    while (in_header)
    {
	getline (stream, curr_line);

	if (curr_line == "")
	    continue;
	
	std::stringstream curr_line_stream;
	std::string       param_name;
	int               param_val;
	
	curr_line_stream << curr_line;
	curr_line_stream >> param_name;
	if (!(curr_line_stream >> param_val))
	    param_val = 0;

	if (param_name == "ascent")
	    ascent = param_val;

	else if (param_name == "descent")
	    descent = param_val;
	
	else if (param_name == "GLYPH")
	{
	    in_header = false;
	    continue;
	}
    }

    while (!stream.eof ())
    {
	std::stringstream curr_line_stream;
	std::string       param_name;
	
	curr_line_stream << curr_line;
	curr_line_stream >> param_name;

	if (param_name == "GLYPH")
	{
	    std::string glyph_index;
	    char        glyph_char;

	    if (!(curr_line_stream >> glyph_index))
		glyph_index = "\\0";

	    /* Un-escape char index */
            if (glyph_index[0] == '\'')
            {
                sscanf (glyph_index.c_str (), "'%c'", &glyph_char);
            } else {
                glyph_char = atoi (glyph_index.c_str ());
            }

	    /* Parse current glyph */
	    Glyph curr_glyph (stream, line_height);

	    glyphs.insert (std::pair <char, Glyph> (glyph_char, curr_glyph));
	}
	
	getline (stream, curr_line);
    }
}

static void free_pixdata_gfree (guint8 *pixels, gpointer data)
{
    g_free (pixels);
}

const Glyph& Font::get_glyph (char c) const
{
    std::hash_map<char, Glyph>::const_iterator found_glyph = glyphs.find (c);
    if (found_glyph != glyphs.end ())
	return found_glyph->second;

    static Glyph empty_glyph (line_height, ascent);

    return empty_glyph;
}

GdkPixbuf* Font::render (const std::string &text,
			 const Gdk_Color   &color) const
{
    if (text == "")
	return 0;

    int width = 0;
    int height = line_height;
    typeset_t typeset;
    
    std::string::const_iterator i = text.begin ();
    while (i != text.end ())
    {
	char c = next_char (text, i);
	const Glyph &g = get_glyph (c);
	
	typeset.push_back (&g);
	width += g.width;
    }

    guint8 *pixbuf_data = g_new0 (guint8, width * height * 4);
    render_raw_line (typeset, color, pixbuf_data, width * 4);
    
    return gdk_pixbuf_new_from_data (pixbuf_data,
				     GDK_COLORSPACE_RGB,
				     true,
				     8,
				     width, height,
				     width * 4,
				     free_pixdata_gfree, 0);
}

GdkPixbuf* Font::render_multi_line (const std::string &text,
				    const Gdk_Color   &color) const
{
    if (text == "")
	return 0;

    std::list <typeset_t> typesets;
    int width = 0;
    int height = 0;

    int current_width = 0;
    std::list <const Glyph*> current_typeset;
    
    std::string::const_iterator i = text.begin ();
    while (i != text.end ())
    {
	char c = next_char (text, i);
	if (c != '\n')
	{
	    const Glyph &g = get_glyph (c);
	    
	    current_typeset.push_back (&g);
	    current_width += g.width;
	} else {
	    typesets.push_back (current_typeset);
	    current_typeset.clear ();

	    width = std::max<int> (width, current_width);
	    current_width = 0;
	    height += line_height;
	}
    }
    // Add last line
    typesets.push_back (current_typeset);
    width = std::max<int> (width, current_width);
    height += line_height;
    
    guint8 *pixbuf_data = g_new0 (guint8, width * height * 4);

    int offset = 0;
    for (std::list <typeset_t>::const_iterator i = typesets.begin ();
	 i != typesets.end (); i++)
    {
	render_raw_line (*i, color, pixbuf_data + offset, width * 4);

	offset += (line_height * width) * 4;
    }
    
    return gdk_pixbuf_new_from_data (pixbuf_data,
				     GDK_COLORSPACE_RGB,
				     true,
				     8,
				     width, height,
				     width * 4,
				     free_pixdata_gfree, 0);
}

void Font::render_raw_line (const typeset_t &typeset,
			    const Gdk_Color &color,
			    guint8          *target,
			    int              target_rowstride) const
{
    for (typeset_t::const_iterator i = typeset.begin (); i != typeset.end (); i++)
    {
        const Glyph *glyph = *i;

        bool **curr_bitmap = glyph->bitmap;
	guint8 *curr_row = target;
	
        for (int y = 0; y < line_height; y++)
        {
	    guint8 *curr_pixel = curr_row;
	    
            for (int x = 0; x < glyph->width; x++)
            {
                if (curr_bitmap[y][x])
		    set_pixel (curr_pixel, color);

		curr_pixel += 4;
	    }

	    curr_row += target_rowstride;
	}

	target += glyph->width * 4;
    }
}

int Font::get_line_width (const std::string &text) const
{
    int width = 0;

    std::string::const_iterator i = text.begin ();
    while (i != text.end ())
    {
	char c = next_char (text, i);
	const Glyph &g = get_glyph (c);
	
	width += g.width;
    }
    
    return width;
}

int Font::get_raw_line_width (const std::string &text) const
{
    int width = 0;
    
    for (std::string::const_iterator i = text.begin ();
	 i != text.end (); i++)
    {
	const Glyph &g = get_glyph (*i);
	
	width += g.width;
    }
    
    return width;
}

void Font::get_text_extents (const std::string &text,
			     int               &width,
			     int               &height) const
{
    width = 0;
    height = line_height; // Even if the string is empty, a height of
                          // at least one line is returned
    
    std::string::const_iterator i = text.begin ();
    std::string current_line;
    while (i != text.end ())
    {
	char c = next_char (text, i);

	if (c != '\r')
	    current_line += c;
	else
	{
	    width = std::max<int> (width, get_raw_line_width (current_line));
	    height += line_height;
	    current_line.clear ();
	}
    }

    // Add last line
    width = std::max<int> (width, get_raw_line_width (current_line));
    height += line_height;
}
