//$Id: form-editor.cc,v 1.25 2002/06/02 13:53:40 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "form-editor/form-editor.h"

#include <gnome--/canvas.h>
#include <gnome--/canvas-group.h>
#include <gnome--/canvas-text.h>
#include <gnome--/canvas-polygon.h>

#include <gdk-pixbuf/gnome-canvas-pixbuf.h>

#include "preferences.h"
#include "resource-manager.h"
#include "target.h"

using namespace Guikachu::GUI;
using Guikachu::GUI::FormEditor::Font;

#define GUIKACHU_TITLE_FONT 1

static GdkPixbuf* clip_pixbuf (GdkPixbuf    *src,
			       GtkAnchorType anchor,
			       int           clip_width,
			       int           clip_height);

const Font &FormEditor::get_font (int font_num)
{
    static Font standard_fonts[8];
    static bool init[8] = {false, false, false, false, false, false, false, false};

    if (!init[font_num])
    {
	std::string font_filename;
	
	switch (font_num)
	{
	case 0:
	    font_filename = "stdfont.txt";
	    break;
	case 1:
	    font_filename = "boldfont.txt";
	    break;
	case 2:
	    font_filename = "bigfont.txt";
	    break;
	case 3:
	    font_filename = "symbol.txt";
	    break;
	case 4:
	    font_filename = "symbol11.txt";
	    break;
	case 5:
	    font_filename = "symbol7.txt";
	    break;
	case 6:
	    font_filename = "ledfont.txt";
	    break;
	case 7:
	    font_filename = "bigbold.txt";
	    break;
	default:
	    g_assert_not_reached ();
	    break;
	}
	
	standard_fonts[font_num].parse (font_filename);
	init[font_num] = true;
    }
	
    return standard_fonts[font_num];
}

const Font &FormEditor::get_title_font ()
{
    return get_font (GUIKACHU_TITLE_FONT);
}

int FormEditor::get_font_height (int font_num)
{
    const Font &font = get_font (font_num);
    return font.get_line_height ();
}

int FormEditor::get_line_width (int font_num, const std::string &text)
{
    const Font &font = get_font (font_num);
    return font.get_line_width (text);
}

void FormEditor::get_text_extents (int font_num, const std::string &text,
				   int &width, int &height)
{
    const Font &font = get_font (font_num);
    font.get_text_extents (text, width, height);
}

Gnome::CanvasItem* FormEditor::draw_text (Gnome::CanvasGroup &group,
					  const std::string &text, int font_num,
					  int x, int y,
					  bool usable,
					  bool multi_line,
					  GtkAnchorType anchor,
					  int clip_width, int clip_height)
{
    Font font = get_font (font_num);
    Gdk_Color color = get_foreground_color (usable);

    return render_text (group, text, font, color,
			x, y,
			multi_line,
			anchor,
			clip_width, clip_height);
}

Gnome::CanvasItem* FormEditor::render_text (Gnome::CanvasGroup &group,
					    const std::string  &text,
					    const Font         &font,
					    const Gdk_Color    &color,
					    int x, int y,
					    bool multi_line,
					    GtkAnchorType anchor,
					    int clip_width, int clip_height)
{
    Gnome::CanvasGroup *text_group = new Gnome::CanvasGroup (group, 0, 0);
    
    GdkPixbuf *pixbuf = multi_line ?
	font.render_multi_line (text, color) : font.render (text, color);

    if (!pixbuf)
	return 0;
    
    GdkPixbuf *clipped_pixbuf;
    if (clip_width || clip_height)
	clipped_pixbuf = clip_pixbuf (pixbuf, anchor, clip_width, clip_height);
    else
	clipped_pixbuf = gdk_pixbuf_ref (pixbuf);
    gdk_pixbuf_unref (pixbuf);
    
    int width = gdk_pixbuf_get_width (clipped_pixbuf);
    int height = gdk_pixbuf_get_height (clipped_pixbuf);

    // Horizontal offset
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_SOUTH_WEST:
	break;
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_SOUTH:
	x -= width / 2;
	break;
    case GTK_ANCHOR_NORTH_EAST:
    case GTK_ANCHOR_EAST:
    case GTK_ANCHOR_SOUTH_EAST:
	x -= width;
    }

    // Vertical offset
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_NORTH_EAST:
	break;
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_EAST:
	y -= height / 2;
	break;
    case GTK_ANCHOR_SOUTH_WEST:
    case GTK_ANCHOR_SOUTH:
    case GTK_ANCHOR_SOUTH_EAST:
	y -= height;
	break;
    }

    // Draw the text
    draw_pixbuf (*text_group, clipped_pixbuf, x, y);
    gdk_pixbuf_unref (clipped_pixbuf);

    // Draw a transparent background block (to get mouse events for
    // the whole area)
    FormEditor::draw_rectangle (*text_group,
				x, y,
				x + width, y + height);
    
    return text_group;
}

Gnome::CanvasItem* FormEditor::draw_pixbuf (Gnome::CanvasGroup &group,
					    GdkPixbuf          *pixbuf,
					    int                 x,
					    int                 y)
{
    return Gtk::wrap (gnome_canvas_item_new (group.gtkobj (),
					     gnome_canvas_pixbuf_get_type (),
					     "pixbuf", pixbuf,
					     "x", (double) x,
					     "y", (double) y,
					     0));
}

static GdkPixbuf* clip_pixbuf (GdkPixbuf     *src,
			       GtkAnchorType  anchor,
			       int            clip_width,
			       int            clip_height)
{
    GdkPixbuf *dest;
    
    int width = gdk_pixbuf_get_width (src);
    int height = gdk_pixbuf_get_height (src);

    if (clip_width <= 0 || clip_width > width)
	clip_width = width;

    if (clip_height <= 0 || clip_height > height)
	clip_height = height;

    int src_x = 0;
    int src_y = 0;

    /* Set src_x */
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_SOUTH_WEST:
	src_x = 0;
	break;

    case GTK_ANCHOR_NORTH_EAST:
    case GTK_ANCHOR_EAST:
    case GTK_ANCHOR_SOUTH_EAST:
	src_x = width - clip_width;
	break;

    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_SOUTH:
	src_x = (width / 2) - (clip_width / 2);
	break;
    }

    /* Set src_y */
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_NORTH_EAST:
	src_y = 0;
	break;

    case GTK_ANCHOR_SOUTH_WEST:
    case GTK_ANCHOR_SOUTH:
    case GTK_ANCHOR_SOUTH_EAST:
	src_y = height - clip_height;
	break;

    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_EAST:
	src_y = (height / 2) - (clip_height / 2);
	break;
    }

    dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, true, 8,
			   clip_width, clip_height);

    gdk_pixbuf_copy_area (src,
			  src_x, src_y,
			  clip_width, clip_height,
			  dest, 0, 0);

    return dest;
}

Gnome::CanvasItem* FormEditor::draw_rectangle (Gnome::CanvasGroup &group,
					       int x1, int y1,
					       int x2, int y2)
{
    Gnome::CanvasPoints points;
    using Gnome::Art::Point;

    points.push_back (Point (x1, y1));
    points.push_back (Point (x2, y1));
    points.push_back (Point (x2, y2));
    points.push_back (Point (x1, y2));
    
    return new Gnome::CanvasPolygon (group, points);
}

std::string FormEditor::get_foreground_color (bool enabled)
{
    if (enabled)
	return Preferences::FormEditor::get_color_fg ();
    else
	return Preferences::FormEditor::get_color_disabled ();
}

std::string FormEditor::get_background_color ()
{
    return Preferences::FormEditor::get_color_bg ();
}

std::string FormEditor::get_selection_color ()
{
    return Preferences::FormEditor::get_color_selection ();
}
