//$Id: form-win.cc,v 1.54 2002/08/15 23:09:59 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "form-win.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include "form-editor/palette.h"

#include <gtk--/main.h>
#include <gtk--/treeitem.h>
#include <gtk--/table.h>
#include <gtk--/button.h>
#include <gtk--/pixmap.h>
#include <gtk--/menu.h>
#include <gtk--/scrolledwindow.h>

#include <gdk--/cursor.h>

#include <gnome--/canvas-group.h>
#include <gnome--/canvas-polygon.h>

#include <sigc++/retbind.h>

#include "form-win-helpers.h"
#include "form-editor/form-editor.h"
#include "preferences.h"
#include "target.h"

using namespace Guikachu::GUI;
using namespace Guikachu::Resources;

// Static members of FormWindow
static Guikachu::GUI::Palette *palette = 0;


FormWindow::FormWindow (Form *res_):
    res (res_),

    gui (glade_xml_new (GUIKACHU_GLADEDIR "/guikachu.glade", "res_form")),

    window     (Gtk::wrap (GTK_WINDOW      (glade_xml_get_widget (gui, "res_form")))),
    canvas     (Gtk::wrap (GNOME_CANVAS    (glade_xml_get_widget (gui, "form_canvas")))),
    zoom_entry (Gtk::wrap (GTK_SPIN_BUTTON (glade_xml_get_widget (gui, "form_zoom_entry")))),    

    edited_widget (0),
    last_widget (0),
    canvas_overlay (0),
    palette_selection (Widgets::WIDGET_NONE)
{
    using namespace SigC;
    using Gtk::wrap;
    
    window->delete_event.connect (slot (this, &FormWindow::delete_event_impl));

    // Property editor window
    property_window = new Gtk::Window (GTK_WINDOW_DIALOG);
    property_window->set_title (_("Widget properties"));
    property_window->delete_event.connect (slot (this, &FormWindow::property_delete_event_impl));
    property_window->set_policy (false, true, true);

    // Load widgets from Glade file
    Gtk::Tree *tree = wrap (GTK_TREE (glade_xml_get_widget (gui, "form_tree")));
    tree->map_ (); // FIXME: Isn't this evil?

    // Set up zoom factor entry
    Gtk::Adjustment *zoom_adj = zoom_entry->get_adjustment ();
    zoom_adj->value_changed.connect (slot (this, &FormWindow::zoom_entry_cb));
    zoom_adj->set_value (Preferences::FormEditor::get_default_zoom ());
    
    // Widget tree
    FormWindow_Helpers::TopTreeItem *item = new FormWindow_Helpers::TopTreeItem (res);
    item->show ();
    item->activate.connect (slot (this, &FormWindow::select_form));
    item->menu_edit.connect (slot (this, &FormWindow::select_form));
    tree->add (*manage (item));
    top_treeitem = item;

    form_widget = new Widgets::Form (res);
    form_canvas_item = form_widget->get_canvas_item (canvas->root ());
    form_canvas_item->clicked.connect (slot (this, &FormWindow::form_clicked_cb));
    form_canvas_item->released.connect (slot (this, &FormWindow::form_released_cb));
    form_canvas_item->drag_motion.connect (slot (this, &FormWindow::form_drag_motion_cb));
    form_canvas_item->drag_end.connect (slot (this, &FormWindow::form_drag_end_cb));
    
    // Set up initial widgets
    const std::set<Widget*>& widgets = res->get_widgets ();
    for (std::set<Widget*>::const_iterator i = widgets.begin ();
	 i != widgets.end (); i++)
	widget_created_cb (*i);
    
    // Window callbacks
    window->key_press_event.connect (SigC::retbind (
	SigC::slot (this, &FormWindow::key_press_cb), 1));

    // Canvas selection callback
    selection_canvas = new FormWindow_Helpers::CanvasSelectionWrapper (*canvas);
    selection_canvas->selection.connect (SigC::slot (this, &FormWindow::selection_box_cb));
    
    // Form resource callbacks
    res->changed.connect (slot (this, &FormWindow::update));
    res->widget_created.connect (slot (this, &FormWindow::widget_created_cb));
    res->widget_removed.connect (slot (this, &FormWindow::widget_removed_cb));
    update ();

    // Callbacks for global settings
    Preferences::FormEditor::colors_changed.connect (slot (this, &FormWindow::colors_changed_cb));
    res->get_manager ()->get_target ()->changed.connect (
	slot (this, &FormWindow::target_changed_cb));
    recenter_canvas ();
    
    if (!palette)
	palette = new Guikachu::GUI::Palette ();
    palette->palette_changed.connect (slot (this, &FormWindow::palette_cb));

}

FormWindow::~FormWindow ()
{
    gtk_widget_destroy (GTK_WIDGET (window->gtkobj ()));
    gtk_object_unref (GTK_OBJECT (gui));

    delete form_widget;
    delete property_window;
    delete selection_canvas;
}

void FormWindow::show ()
{
    window->show_all ();
    window->get_window ().raise ();
    palette->show_all ();
    palette->get_window ().raise ();

    if (edited_widget)
	show_property_editor (edited_widget);
    else
	show_form_property_editor ();
}

int FormWindow::delete_event_impl (GdkEventAny *e)
{
    window->hide ();
    property_window->hide ();
    return true;
}

int FormWindow::property_delete_event_impl (GdkEventAny *e)
{
    property_window->hide ();
    return true;
}

void FormWindow::update ()
{
    gchar *title_buf = g_strdup_printf (_("Form: %s"), res->id ().c_str ());
    window->set_title (title_buf);
    g_free (title_buf);

    if (!edited_widget)
    {
	gchar *prop_buf = g_strdup_printf (_("Properties: %s"), res->id ().c_str ());
	property_window->set_title (prop_buf);
	g_free (prop_buf);
    }

}

void FormWindow::recenter_canvas ()
{   
    float zoom_factor = canvas->get_pixels_per_unit ();
    int screen_width = res->get_manager ()->get_target ()->screen_width;
    int screen_height = res->get_manager ()->get_target ()->screen_height;
    
    canvas->set_scroll_region (0, 0, screen_width - 1, screen_height - 1);
    canvas->set_usize (int (screen_width * zoom_factor),
		       int (screen_height * zoom_factor));

    
    Gtk::ScrolledWindow *scrollwindow = Gtk::wrap (
	GTK_SCROLLED_WINDOW (glade_xml_get_widget (gui, "form_scrollwin")));
    
    scrollwindow->set_usize (screen_width + 40, screen_height + 40);
}

void FormWindow::force_canvas_update ()
{
    // Update form canvas item
    form_canvas_item->update ();

    // Update widget canvas items
    const std::set<Widget*>& widgets = res->get_widgets ();
    for (std::set<Widget*>::const_iterator i = widgets.begin ();
	 i != widgets.end (); i++)
	(*i)->get_canvas_item (form_canvas_item->get_widget_group ())->update ();
}

void FormWindow::colors_changed_cb ()
{
    force_canvas_update ();
}

void FormWindow::target_changed_cb ()
{
    force_canvas_update ();
    recenter_canvas ();
}

void FormWindow::zoom_entry_cb ()
{
    float zoom_factor = zoom_entry->get_value_as_float ();
    if (zoom_factor > GNOME_CANVAS_EPSILON)
    {
	canvas->set_pixels_per_unit (zoom_factor);
	recenter_canvas ();
    }
}

void FormWindow::widget_created_cb (Widget *widget)
{
    g_return_if_fail (widget != 0);

    using SigC::bind;
    using SigC::slot;
    
    widget->request_edit.connect (bind (
	slot (this, &FormWindow::show_property_editor),
	widget));

    widget->selected_last.connect (bind (
	slot (this, &FormWindow::widget_selected_last_cb),
	widget));
    
    Gtk::Tree *tree = top_treeitem->get_subtree ();
    if (!tree)
    {
	tree = new Gtk::Tree ();
	top_treeitem->set_subtree (*manage (tree));
	top_treeitem->expand ();
    }

    // Add tree item
    FormWindow_Helpers::WidgetTreeItem *tree_item = new FormWindow_Helpers::WidgetTreeItem (widget);
    tree_item->show ();
    tree_item->activate.connect (bind (slot (this, &FormWindow::select_widget), widget));
    tree_item->menu_remove.connect (bind (slot (this, &FormWindow::widget_menu_remove_cb), widget));
    tree_item->menu_edit.connect (bind (slot (this, &FormWindow::select_widget), widget));
    
    tree->add (*tree_item);
    
    // Add canvas item
    FormEditor::WidgetCanvasItem *canvas_item = widget->get_canvas_item (form_canvas_item->get_widget_group ());

    canvas_item->clicked.connect (bind (slot (this, &FormWindow::widget_clicked_cb), widget));
    canvas_item->released.connect (bind (slot (this, &FormWindow::widget_released_cb), widget));
    canvas_item->menu_remove.connect (bind (slot (this, &FormWindow::widget_menu_remove_cb), widget));
    canvas_item->menu_edit.connect (bind (slot (this, &FormWindow::select_widget), widget));
    
    canvas_item->drag_motion.connect (slot (this, &FormWindow::widget_drag_motion_cb));
    canvas_item->drag_end.connect (slot (this, &FormWindow::widget_drag_end_cb));
    
}

void FormWindow::widget_removed_cb (Widget *widget)
{
    // Remove from selection
    selection_remove (widget);

    // Remove property editor
    if (widget == edited_widget)
	show_form_property_editor ();
    
    // Remove tree item
    Gtk::Tree::ItemList &itemlist = top_treeitem->get_subtree ()->tree ();
    Gtk::Tree::ItemList::iterator i = itemlist.begin ();
    
    while (i != itemlist.end () && (*i)->get_user_data () != widget)
	i++;

    g_return_if_fail (i != itemlist.end ());

    itemlist.erase (i);
}

int FormWindow::canvas_overlay_cb (GdkEvent *event)
{
    if (event->type != GDK_BUTTON_PRESS)
	return false;

    if (event->button.button != 1)
	return false;

    add_widget (int (event->button.x - res->x),
		int (event->button.y - res->y));
    
    palette->palette_applied ();

    return true;
}

void FormWindow::palette_cb (Widgets::Type type)
{
    if (type == palette_selection)
	return;

    palette_selection = type;
    delete canvas_overlay;
    canvas_overlay = 0;
    
    // Change pointer
    Gdk_Window window = canvas->get_window ();
    if (type == Widgets::WIDGET_NONE)
    {
	window.set_cursor (0);

	// Un-block rubber band selection from idle handler -- this
	// way, it's done after the _after handler in the
	// SelectionWrapper is run
	Gtk::Main::idle.connect (SigC::retbind (SigC::bind (
	    SigC::slot (selection_canvas,
			&FormWindow_Helpers::CanvasSelectionWrapper::set_active),
	    true), 0));
	
	return;

    } else {

	Gdk_Cursor cursor (GDK_CROSSHAIR);
	window.set_cursor (cursor);
	cursor.destroy ();
    }
    
			   
    // Create overlay canvas item
    Gnome::CanvasPoints   points;

    int x1 = res->x;
    int y1 = res->y;
    int x2 = x1 + res->get_manager ()->get_target ()->screen_width;
    int y2 = y1 + res->get_manager ()->get_target ()->screen_height;
    
    points.push_back (Gnome::Art::Point (x1, y1));
    points.push_back (Gnome::Art::Point (x2, y1));
    points.push_back (Gnome::Art::Point (x2, y2));
    points.push_back (Gnome::Art::Point (x1, y2));
    points.push_back (Gnome::Art::Point (x1, y1));
    
    canvas_overlay = new Gnome::CanvasPolygon (*canvas->root (), points);
    canvas_overlay->event.connect (slot (this, &FormWindow::canvas_overlay_cb));
    canvas_overlay->raise_to_top ();

    selection_canvas->set_active (false);
}

void FormWindow::add_widget (int x, int y)
{
    g_return_if_fail (palette_selection != Widgets::WIDGET_NONE);
    
    Widget *widget = res->create_widget (palette_selection);

    widget->x = x - widget->get_width () / 2;
    widget->y = y - widget->get_height () / 2;
}

void FormWindow::form_clicked_cb ()
{
    select_form ();

    // Block selection rubber box
    selection_canvas->set_active (false);
}

void FormWindow::form_released_cb ()
{
    // Enable selection rubber box
    selection_canvas->set_active (true);
}

void FormWindow::widget_clicked_cb (Widget *widget)
{
    select_widget (widget);

    // Block selection rubber box
    selection_canvas->set_active (false);
}

void FormWindow::widget_released_cb (Widget *widget)
{
    // Enable selection rubber box
    selection_canvas->set_active (true);
}

void FormWindow::widget_menu_remove_cb (Widget *widget)
{
    g_return_if_fail (widget);
    
    if (selected_widgets.size () &&
	selected_widgets.find (widget) != selected_widgets.end ())
    {
	for (std::set<Widget*>::iterator i = selected_widgets.begin ();
	     i != selected_widgets.end ();)
	    res->remove_widget (*(i++));
    } else {
	res->remove_widget (widget);
    }
}

void FormWindow::form_drag_motion_cb (int dx, int dy)
{
    form_canvas_item->move (dx, dy);
}

void FormWindow::form_drag_end_cb (int dx, int dy)
{
    res->x += dx;
    res->y += dy;
}

void FormWindow::widget_drag_motion_cb (int dx, int dy)
{
    for (std::set<Widget*>::iterator i = selected_widgets.begin ();
	 i != selected_widgets.end (); i++)
    {
	FormEditor::CanvasItem *canvas_item = (*i)->get_canvas_item (form_canvas_item->get_widget_group ());
	canvas_item->move (dx, dy);
    }    
}

void FormWindow::widget_drag_end_cb (int dx, int dy)
{
    for (std::set<Widget*>::iterator i = selected_widgets.begin ();
	 i != selected_widgets.end (); i++)
    {
	(*i)->x += dx;
	(*i)->y += dy;
    }
}

void FormWindow::key_press_cb (GdkEventKey *e)
{
    // Del: delete current selection
    if (e->keyval == GDK_Delete ||
	e->keyval == GDK_KP_Delete)
    {
	for (std::set<Widget*>::iterator i = selected_widgets.begin ();
	     i != selected_widgets.end ();)
	    res->remove_widget (*(i++));
    }
}
