//$Id: mainwin.cc,v 1.33 2002/06/30 19:54:46 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"
#include <libgnome/libgnome.h>

#include "mainwin.h"

#include <gtk--/statusbar.h>
#include <gtk--/scrolledwindow.h>

#include <gnome--/dialog.h>
#include <gnome--/about.h>
#include <glade/glade.h>

#include <list>

#include "resource-manager.h"

#include "treeitem.h"
#include "top-treeitem.h"
#include "add-resource.h"
#include "preferences-win.h"
#include "preferences.h"
#include "session.h"

#include <fstream>
#include <sigc++/rettype.h>
#include <sigc++/retbind.h>

using namespace Guikachu;

GUI::MainWin::MainWin ():
    Gnome::App ("Guikachu", "Guikachu"),
    manager (0),
    top_item (0),
    filename (""),
    in_exit (false),
    recent_files_num (0)   
{
    using namespace SigC;
    
    /* File menu */
    Gnome::UI::Info m_file[] = {
	Gnome::MenuItems::New    (_("_New project"), _("Create a new project"),
				  slot (this, &MainWin::new_cb)),
    
	Gnome::MenuItems::Open   (slot (this, &MainWin::load_cb)),
	Gnome::MenuItems::Save   (slot (this, &MainWin::save_cb)),
	Gnome::MenuItems::SaveAs (slot (this, &MainWin::save_as_cb)),
	Gnome::UI::Item          (Gnome::UI::Icon (GNOME_STOCK_MENU_CONVERT),
				  _("Export _RCP..."),
				  slot (this, &MainWin::export_cb),
				  _("Create an RCP file for compilation")),
	
	Gnome::UI::Separator     (),
	Gnome::UI::Item          (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_ADD),
				  _("Add _Resource..."),
				  slot (this, &MainWin::add_cb),
				  _("Add new component to current project")),
	
	Gnome::UI::Separator     (),
	Gnome::MenuItems::Exit   (bind (rettype<void>(delete_event.slot()),
					(GdkEventAny*)0))
    };
    
    /* Settings menu */
    Gnome::UI::Info m_settings[] = {
	Gnome::MenuItems::Preferences (slot (this, &MainWin::preferences_cb))
    };
    
    /* Help menu */
    Gnome::UI::Info m_help[] = {
	Gnome::MenuItems::About (slot (&show_about))
    };

    /* Create the menu bar structure */
    Gnome::UI::SubTree menubar[] = {
	Gnome::Menus::File     (m_file),
	Gnome::Menus::Settings (m_settings),
	Gnome::Menus::Help     (m_help)
    };
    create_menus (menubar);

    /* Create status bar */
    set_statusbar (*manage (new Gtk::Statusbar ()));
    install_menu_hints ();

    // Create the toolbar
    Gnome::UI::Info toolbar [] = {
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_ADD),
			 _("Add"),
			 slot (this, &MainWin::add_cb),
			 _("Add new component to current project")),
	Gnome::UI::Separator (),
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_NEW),
			 _("New"),
			 slot (this, &MainWin::new_cb),
			 _("New project")),
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_OPEN),
			 _("Open"),
			 slot (this, &MainWin::load_cb),
			 _("Open project")),
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_SAVE),
			 _("Save"),
			 slot (this, &MainWin::save_cb),
			 _("Save project"))
    };
    create_toolbar (toolbar);

    
    // Listen for changes in recent file list
    Preferences::Interface::recent_files_changed.connect (
	slot (this, &MainWin::update_recent_files));
    update_recent_files ();
    
    top_tree = new Gtk::Tree ();
    
    // Scrollable viewport
    Gtk::ScrolledWindow *scrollwin = new Gtk::ScrolledWindow ();
    scrollwin->set_policy (GTK_POLICY_AUTOMATIC,
			   GTK_POLICY_AUTOMATIC);
    scrollwin->add_with_viewport (*manage (top_tree));
    scrollwin->show_all ();
    set_contents (*manage (scrollwin));
    set_usize (300, 250);

    /* Connect to window events */
    key_press_event.connect (retbind (slot (this, &MainWin::key_press_cb), 1));

    update_title ();
}

void GUI::MainWin::set_manager (ResourceManager *manager_)
{
    manager = manager_;

    // Top treeitem
    top_tree->tree ().clear ();
    top_item = new TopTreeItem (manager->get_application ());    
    top_item->show ();
    top_tree->tree ().push_back (*manage (top_item));

    // Create the tree for the resources
    main_tree = new Gtk::Tree ();
    top_item->set_subtree (*manage (main_tree));
    top_item->expand ();

    // Create subtrees
    subtrees.clear ();
    create_tree_for_type (Resources::RESOURCE_DIALOG,     _("Dialogs"));
    create_tree_for_type (Resources::RESOURCE_FORM,       _("Forms"));
    create_tree_for_type (Resources::RESOURCE_MENU,       _("Menus"));
    create_tree_for_type (Resources::RESOURCE_STRING,     _("Strings"));
    create_tree_for_type (Resources::RESOURCE_STRINGLIST, _("String lists"));
    create_tree_for_type (Resources::RESOURCE_BLOB,       _("Blobs"));

    /* Fill with existing resources */
    std::set<Resource*> resources = manager->get_resources ();
    for (std::set<Resource*>::iterator i = resources.begin ();
	 i != resources.end (); i++)
	resource_created_cb ((*i));

    /* Connect to Resource Manager signals */
    manager->resource_created.connect (slot (this, &MainWin::resource_created_cb));
    manager->resource_removed.connect (slot (this, &MainWin::resource_removed_cb));
}

void GUI::MainWin::update_recent_files ()
{
    // Clear old entries
    if (recent_files_num)
	remove_menu_range ("File/", 7, recent_files_num + 1);

    std::list<std::string> recent_files = Preferences::Interface::get_recent_files ();
    recent_files_num = recent_files.size ();

    if (recent_files_num)
    {
	std::list<Gnome::UI::Info> recent_items;
	recent_items.push_back (Gnome::UI::Separator ());

	std::list<std::string>::const_iterator i;
	int num;
	for (i = recent_files.begin (), num = 0;
	     i != recent_files.end (); i++, num++)
	{
	    std::string vis_filename = visible_filename (*i);

	    // Change _ to __ to avoid underscore accelerators
	    std::string escaped_filename;
	    for (std::string::iterator j = vis_filename.begin ();
		 j != vis_filename.end (); j++)
		if (*j == '_')
		    escaped_filename += "__";
		else
		    escaped_filename += *j;
	    
	    char *menu_label;
	    
	    if (num < 9)
		menu_label = g_strdup_printf ("_%d %s", num + 1, escaped_filename.c_str ());
	    else
		menu_label = g_strdup_printf ("%d %s", num + 1, escaped_filename.c_str ());
	    
	    recent_items.push_back (
		Gnome::UI::Item (menu_label,
				 SigC::bind ( SigC::slot (this, &MainWin::open), *i)));
	    
	    g_free (menu_label);
	}

	insert_menus ("File/Add _Resource...", recent_items);
	install_menu_hints ();
    }
}

void GUI::MainWin::resource_created_cb (Resource *resource)
{
    TreeItem *item = new GUI::TreeItem (resource);
    item->set_user_data (resource);
    item->show ();

    Gtk::TreeItem *subtree_item = subtrees.find (resource->get_type ())->second;
    g_assert (subtree_item);
    
    if (!subtree_item->get_subtree ())
    {
	subtree_item->set_subtree (*manage (new Gtk::Tree ()));
	subtree_item->expand ();
    }
    
    subtree_item->get_subtree ()->tree ().push_back (*manage (item));
    treeitems.insert (treeitem_map::value_type (resource, item));
}

void GUI::MainWin::resource_removed_cb (Resource *resource)
{
    treeitem_map::iterator treeitem_entry = treeitems.find (resource);
    g_assert (treeitem_entry != treeitems.end ());

    Gtk::TreeItem *subtree_item = subtrees.find (resource->get_type ())->second;
    g_assert (subtree_item);
    
    subtree_item->get_subtree ()->tree ().remove (treeitem_entry->second);

    treeitems.erase (treeitem_entry);
}

void GUI::MainWin::create_tree_for_type (Resources::Type    type,
					 const std::string &category_name)
{
    Gtk::TreeItem *subtree_item = new Gtk::TreeItem (category_name);
    std::pair<subtree_map::iterator, bool> success;
    
    success = subtrees.insert (subtree_map::value_type (type, subtree_item));
    
    if (!success.second)
    {
	delete subtree_item;

	return;
    }

    subtree_item->show ();
    main_tree->tree ().push_back (*manage (subtree_item));
}

void GUI::MainWin::add_cb ()
{
    AddResourceWin add_win (manager);

    add_win.run ();
}

void GUI::MainWin::preferences_cb ()
{
    static PreferencesWin pref_win;

    pref_win.run ();
}

void GUI::MainWin::update_title ()
{
    gchar *title_buf;

    if (filename != "")
	title_buf = g_strdup_printf (_("Guikachu: %s"), visible_filename (filename).c_str ());
    else
	title_buf = g_strdup_printf (_("Guikachu: %s"), _("<unnamed>"));
	
    set_title (title_buf);

    g_free (title_buf);
}

void GUI::MainWin::key_press_cb (GdkEventKey *e)
{
    // Del: delete current selection
    if (e->keyval == GDK_Delete ||
	e->keyval == GDK_KP_Delete)
    {
	std::list<Gtk::TreeItem*> sel_list (top_tree->selection ().begin (),
					    top_tree->selection ().end ());

	for (std::list<Gtk::TreeItem*>::iterator i = sel_list.begin ();
	     i != sel_list.end (); i++)
	{
	    // If the per-app resource is selected, reset it
	    if (dynamic_cast<GUI::TopTreeItem*> (*i))
	    {
		manager->get_application ()->reset ();
		continue;
	    }
	    
	    Resource *user_data = (Resource*)(*i)->get_user_data ();
	    Resource *res = dynamic_cast<Resource*> (user_data);

	    if (res)
		manager->remove_resource (res);
	}
    }
}

std::string GUI::visible_filename (const std::string &filename)
{
    // Cut off path
    std::string retval = g_basename (filename.c_str ());

    // If extension is .guikachu, cut that off as well
    static const char         *pattern = ".guikachu";
    static const unsigned int  pattern_len = strlen (pattern);
    
    if (retval.length () > pattern_len)
    {
	if (retval.substr (retval.length () - pattern_len) == pattern)
	    retval.erase (retval.length () - pattern_len);
    }

    return retval;
}

void GUI::show_about ()
{
    std::list<std::string> authors;
    authors.push_back ("Gerg rdi <cactus@cactus.rulez.org>");
    authors.push_back ("Jay Bloodworth <jbloodworth@sc.rr.com>");
    authors.push_back ("Christopher Keith Fairbairn <ckf13@student.canterbury.ac.nz>");
    
    Gnome::About *about = new Gnome::About ("Guikachu", VERSION,
					    "(C) 2001-2002 Gerg rdi",
					    authors,
					    _("PalmOS resource file editor"));
    about->run ();  
}

void GUI::show_error (const std::string &full_message,
		      const std::string &short_message)
{
    if (Guikachu::Main::initialized ())
    {
	Gnome::Dialog *dialog = Gnome::Dialogs::error (full_message);
	dialog->run ();
    } else {
	if (short_message != "")
	    std::cerr << _("Guikachu: Error: ") << short_message << std::endl;
    }
}

void GUI::show_warning (const std::string &full_message,
			const std::string &short_message)
{
    if (Guikachu::Main::initialized ())
    {
	Gnome::Dialog *dialog = Gnome::Dialogs::warning (full_message);
	dialog->run ();
    } else {
	if (short_message != "")
	    std::cerr << _("Guikachu: Warning: ") << short_message << std::endl;
    }
}
