//$Id: rcp-save.cc,v 1.4 2002/06/30 18:35:43 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "rcp-save.h"

#include <fstream>

#include "blob-res.h"
#include "dialog-res.h"
#include "menu-res.h"
#include "string-res.h"
#include "stringlist-res.h"
#include "rcp-save-form.h"

#include "config.h" // for VERSION

namespace Guikachu
{
    
namespace IO
{
    
namespace RCPSaver_funcs
{
    static void save_app      (Resources::Application *app, std::ostream &ostr);
    static void save_resource (Resource *resource,          std::ostream &ostr);
    
    static void save_res_blob       (Resources::Blob *res,       std::ostream &ostr);
    static void save_res_dialog     (Resources::Dialog *res,     std::ostream &ostr);
    static void save_res_menu       (Resources::Menu *res,       std::ostream &ostr);
    static void save_res_string     (Resources::String *res,     std::ostream &ostr);
    static void save_res_stringlist (Resources::StringList *res, std::ostream &ostr);
}

    
IOResult RCPSaver::save (ResourceManager *manager, const std::string &filename)
{
    std::ofstream fout (filename.c_str ());

    time_t curr_date = time (0);
    const char* date_str = ctime (&curr_date);
    
    fout << "/*" << std::endl;
    fout << " * This file was created by Guikachu " << VERSION
         << ". Do not edit it manually." << std::endl;
    fout << " * Creation date: " << date_str; // ctime ()'s return value ends in \n
    fout << " */" << std::endl << std::endl;;
    
    const std::set<Resource*> &resources = manager->get_resources ();
    
    // Export blobs before anything else, to allow #includes and whatnot
    for (std::set<Resource*>::const_iterator i = resources.begin ();
         i != resources.end (); i++)
    {
        if ((*i)->get_type () == Resources::RESOURCE_BLOB)
	    RCPSaver_funcs::save_res_blob (static_cast<Resources::Blob*> (*i), fout);
    }
    
    // Export per-app data
    RCPSaver_funcs::save_app (manager->get_application (), fout);
    
    // Export the rest of the resources
    for (std::set<Resource*>::const_iterator i = resources.begin ();
	 i != resources.end (); i++)
    {
        if ((*i)->get_type () != Resources::RESOURCE_BLOB)
	    RCPSaver_funcs::save_resource (*i, fout);
    }
    
    fout.close ();
    
    return RESULT_OK;
}

    
namespace RCPSaver_funcs
{
    
void save_app (Resources::Application *app, std::ostream &ostr)
{
    if (app->version != "")
	ostr << "VERSION ID 1 \"" << app->version << "\"" << std::endl;
    if (app->iconname != "")
	ostr << "APPLICATIONICONNAME ID 1 \"" << app->iconname << "\"" << std::endl;
    if (app->vendor != "")
	ostr << "APPLICATION ID 1 \"" << app->vendor << "\"" << std::endl;
    ostr << std::endl;
}

void save_resource (Resource *res, std::ostream &ostr)
{
    switch (res->get_type ())
    {
    case Resources::RESOURCE_BLOB:
	// We've already processed blobs before processing other resources
	g_assert_not_reached (); 
	break;
    case Resources::RESOURCE_DIALOG:
	save_res_dialog (static_cast<Resources::Dialog*> (res), ostr);
	break;
    case Resources::RESOURCE_MENU:
	save_res_menu (static_cast<Resources::Menu*> (res), ostr);
	break;
    case Resources::RESOURCE_STRING:
	save_res_string (static_cast<Resources::String*> (res), ostr);
	break;
    case Resources::RESOURCE_STRINGLIST:
	save_res_stringlist (static_cast<Resources::StringList*> (res), ostr);
	break;
    case Resources::RESOURCE_FORM:
	save_res_form (static_cast<Resources::Form*> (res), ostr);
	break;
    }
}


static void save_res_blob (Resources::Blob *res, std::ostream &ostr)
{
    // We use contents.get_val() instead of contents to avoid the
    // escaping of quote marks
    
    ostr << "/* Contents of blob '" << res->id << "' */" << std::endl;
    ostr << res->contents.get_val () << std::endl << std::endl;
}


static void save_res_dialog (Resources::Dialog *res, std::ostream &ostr)
{
    ostr << "ALERT ID " << res->id << std::endl;

    switch (res->dialog_type)
    {
    case Resources::Dialog::TYPE_INFORMATION:
	ostr << "INFORMATION" << std::endl;
	break;
    case Resources::Dialog::TYPE_CONFIRMATION:
	ostr << "CONFIRMATION" << std::endl;
	break;
    case Resources::Dialog::TYPE_WARNING:
	ostr << "WARNING" << std::endl;
	break;
    case Resources::Dialog::TYPE_ERROR:
	ostr << "ERROR" << std::endl;
	break;
    }
    
    ostr << "DEFAULTBUTTON " << res->default_button << std::endl;
    
    if (res->help_id != "")
	ostr << "  HELPID " << res->help_id << std::endl;

    ostr << "BEGIN" << std::endl;

    ostr << "  TITLE \"" << res->title << "\"" << std::endl
	 << "  MESSAGE \"" << res->text << "\"" << std::endl;

    const std::vector<std::string> &buttons = res->buttons;
    if (buttons.size ())
    {
	ostr << "  BUTTONS ";

	for (std::vector<std::string>::const_iterator i = buttons.begin ();
	     i !=  buttons.end (); i++)
	    ostr << "\"" << *i << "\" ";
	
	ostr << std::endl;
    }
    
    ostr << "END" << std::endl << std::endl;
}

static void save_res_menu (Resources::Menu *res, std::ostream &ostr)
{
    ostr << "MENU ID " << res->id << std::endl;

    ostr << "BEGIN" << std::endl;

    const Resources::Menu::MenuTree &submenus = res->get_submenus ();
    for (Resources::Menu::MenuTree::const_iterator i = submenus.begin ();
	 i != submenus.end (); i++)
    {
	ostr << "  PULLDOWN \"" << i->label << "\"" << std::endl;
	
	ostr << "  BEGIN" << std::endl;
	for (std::vector<Resources::Menu::MenuItem>::const_iterator j = i->items.begin ();
	     j != i->items.end (); j++)
	{
	    if (j->separator)
		ostr << "    MENUITEM SEPARATOR" << std::endl;
	    else
	    {
		ostr << "    MENUITEM \"" << j->label << "\" ID " << j->id;
		if (j->shortcut)
		    ostr << " \"" << j->shortcut << "\"";
		ostr << std::endl;
	    }
	}
	ostr << "  END" << std::endl;
    }
    
    ostr << "END" << std::endl << std::endl;
}

static void save_res_string (Resources::String *res, std::ostream &ostr)
{
    ostr << "STRING ID " << res->id << " \"" << res->text << "\"" << std::endl
	 << std::endl;
}

static void save_res_stringlist (Resources::StringList *res, std::ostream &ostr)
{
    ostr << "STRINGTABLE ID " << res->id
	 << " \"" << res->prefix << "\"" << std::endl;

    ostr << " ";
    const std::vector<std::string> &strings = res->strings;
    for (std::vector<std::string>::const_iterator i = strings.begin ();
	 i != strings.end (); i++)
	ostr << " \"" << *i << "\"";
    
    ostr << std::endl << std::endl;
}
    
} // namespace RCPSaver_funcs
} // namespace IO
} // namespace Guikachu
