//$Id: rcp-save-form.cc,v 1.3 2002/06/30 18:35:43 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "rcp-save-form.h"

#include "form-editor/label.h"
#include "form-editor/button.h"
#include "form-editor/pushbutton.h"
#include "form-editor/graffiti.h"
#include "form-editor/selector-trigger.h"
#include "form-editor/checkbox.h"
#include "form-editor/list.h"
#include "form-editor/scrollbar.h"
#include "form-editor/text-field.h"
#include "form-editor/table.h"
#include "form-editor/popup-trigger.h"
#include "form-editor/gadget.h"

namespace Guikachu
{
    
namespace IO
{
    
namespace RCPSaver_funcs
{

static void save_widget (Widget *widget, std::ostream &ostr);
    
static void save_widget_label      (Widgets::Label      *widget, std::ostream &ostr);
static void save_widget_button     (Widgets::Button     *widget, std::ostream &ostr);
static void save_widget_pushbutton (Widgets::PushButton *widget, std::ostream &ostr);
static void save_widget_graffiti   (Widgets::Graffiti   *widget, std::ostream &ostr);
static void save_widget_checkbox   (Widgets::Checkbox   *widget, std::ostream &ostr);
static void save_widget_list       (Widgets::List       *widget, std::ostream &ostr);
static void save_widget_scrollbar  (Widgets::ScrollBar  *widget, std::ostream &ostr);
static void save_widget_text_field (Widgets::TextField  *widget, std::ostream &ostr);
static void save_widget_table      (Widgets::Table      *widget, std::ostream &ostr);
static void save_widget_gadget     (Widgets::Gadget     *widget, std::ostream &ostr);

static void save_widget_selector_trigger (Widgets::SelectorTrigger *widget, std::ostream &node);
static void save_widget_popup_trigger    (Widgets::PopupTrigger    *widget, std::ostream &node);


static void save_element_pos (Widget *widget, std::ostream &ostr)
{
    ostr << " AT (" << widget->x << " " << widget->y << ")";
} 
    
template<class T>
static void save_element_pos_size (T *res, std::ostream &ostr)
{
    ostr << " AT (" << res->x     << " " << res->y      << " "
	            << res->width << " " << res->height << ")" << std::endl;
}

template<class T>
static void save_element_pos_autosize (T *res, std::ostream &ostr)
{
    gchar *width_str = 0, *height_str = 0;
    
    if (res->manual_width)
	width_str = g_strdup_printf ("%d", int (res->width));
    else
	width_str = g_strdup ("AUTO");
    
    if (res->manual_height)
	height_str = g_strdup_printf ("%d", int (res->height));
    else
	height_str = g_strdup ("AUTO");

    ostr << " AT (" << res->x << " " << res->y
	 << " " << width_str << " " << height_str << ")";

    g_free (width_str);
    g_free (height_str);
}

static void save_flag (bool value, std::ostream &ostr,
		       const std::string &true_str,
		       const std::string &false_str = "")
{
    if (value)
	ostr << true_str;
    else
	ostr << false_str;
}

void save_res_form (Resources::Form *res, std::ostream &ostr)
{
    ostr << "FORM ID " << res->id << std::endl;

    save_element_pos_size (res, ostr);
    ostr << "USABLE" << std::endl;

    save_flag (res->modal, ostr, "MODAL ");
    save_flag (res->frame, ostr, "FRAME ", "NOFRAME ");
    save_flag (res->savebehind, ostr, "SAVEBEHIND ", "NOSAVEBEHIND ");

    ostr << std::endl;
    
    if (res->help_id != "")
	ostr << "HELPID " << res->help_id << std::endl;

    if (res->menu_id != "")
	ostr << "MENUID " << res->menu_id << std::endl;

    if (res->def_button != "")
	ostr << "DEFAULTBTNID " << res->def_button << std::endl;

    const std::set<Widget*> &widgets = res->get_widgets ();
    if (widgets.size () || res->title != "")
    {
	ostr << "BEGIN" << std::endl;
	ostr << "  TITLE \"" << res->title << "\"" << std::endl;
	
	for (std::set<Widget*>::const_iterator i = widgets.begin ();
	     i != widgets.end (); i++)
	    save_widget (*i, ostr);
	    
	ostr << "END" << std::endl;
    }
    ostr << std::endl;
}

static void save_widget (Widget *widget, std::ostream &ostr)
{
    switch (widget->get_type ())
    {
    case Widgets::WIDGET_LABEL:
	save_widget_label (static_cast <Widgets::Label*> (widget), ostr);
	break;
    case Widgets::WIDGET_BUTTON:
	save_widget_button (static_cast <Widgets::Button*> (widget), ostr);
	break;
    case Widgets::WIDGET_PUSHBUTTON:
	save_widget_pushbutton (static_cast <Widgets::PushButton*> (widget), ostr);
	break;
    case Widgets::WIDGET_GRAFFITI:
	save_widget_graffiti (static_cast <Widgets::Graffiti*> (widget), ostr);
	break;
    case Widgets::WIDGET_SELECTOR_TRIGGER:
	save_widget_selector_trigger (static_cast <Widgets::SelectorTrigger*> (widget), ostr);
	break;
    case Widgets::WIDGET_CHECKBOX:
	save_widget_checkbox (static_cast <Widgets::Checkbox*> (widget), ostr);
	break;
    case Widgets::WIDGET_LIST:
	save_widget_list (static_cast <Widgets::List*> (widget), ostr);
	break;
    case Widgets::WIDGET_POPUP_TRIGGER:
	save_widget_popup_trigger (static_cast <Widgets::PopupTrigger*> (widget), ostr);
	break;
    case Widgets::WIDGET_SCROLLBAR:
	save_widget_scrollbar (static_cast <Widgets::ScrollBar*> (widget), ostr);
	break;
    case Widgets::WIDGET_TEXT_FIELD:
	save_widget_text_field (static_cast <Widgets::TextField*> (widget), ostr);
	break;
    case Widgets::WIDGET_TABLE:
	save_widget_table (static_cast <Widgets::Table*> (widget), ostr);
	break;
    case Widgets::WIDGET_GADGET:
	save_widget_gadget (static_cast <Widgets::Gadget*> (widget), ostr);
	break;
    case Widgets::WIDGET_NONE:
	g_assert_not_reached ();
	break;
    }    
}

static void save_widget_label (Widgets::Label *widget, std::ostream &ostr)
{
    ostr << "  LABEL \"" << widget->text << "\"" << " ID " << widget->id;

    save_element_pos (widget, ostr);
    
    ostr << " FONT " << widget->font;

    save_flag (widget->usable, ostr, " USABLE", " NONUSABLE");

    ostr << std::endl;
}

static void save_widget_button (Widgets::Button *widget, std::ostream &ostr)
{    
    std::string frame_str, anchor_str;
    std::string type_str;

    if (!widget->repeat)
	type_str = "BUTTON";
    else
	type_str = "REPEATBUTTON";
    
    switch (widget->frame)
    {
    case Widgets::Button::FRAME_NONE:
	frame_str = "NOFRAME";
	break;
    case Widgets::Button::FRAME_SIMPLE:
	frame_str = "FRAME";
	break;
    case Widgets::Button::FRAME_BOLD:
	frame_str = "BOLDFRAME";
	break;
    }

    if (widget->anchor_right)
	anchor_str = "RIGHTANCHOR";
    else
	anchor_str = "LEFTANCHOR";
    
    ostr << "  " << type_str << " \"" << widget->text << "\" ID " << widget->id;
    save_element_pos_autosize (widget, ostr);
    
    ostr << " FONT " << widget->font;
    ostr << " " << frame_str;
    ostr << " " << anchor_str;
    save_flag (widget->usable, ostr, " USABLE", " NONUSABLE");
    
    ostr << std::endl;
}

static void save_widget_pushbutton (Widgets::PushButton *widget, std::ostream &ostr)
{
    ostr << "  PUSHBUTTON \"" << widget->text << "\" ID " << widget->id;
    save_element_pos_autosize (widget, ostr);

    ostr << " FONT " << widget->font;
    ostr << " GROUP " << widget->group;
    save_flag (widget->usable, ostr, " USABLE", " NONUSABLE");
    
    ostr << std::endl;
}

static void save_widget_graffiti (Widgets::Graffiti *widget, std::ostream &ostr)
{
    ostr << "  GRAFFITISTATEINDICATOR";
    save_element_pos (widget, ostr);
    ostr << std::endl;
}

static void save_widget_checkbox (Widgets::Checkbox *widget, std::ostream &ostr)
{
    ostr << "  CHECKBOX \"" << widget->text << "\" ID " << widget->id;
    save_element_pos_autosize (widget, ostr);

    ostr << " FONT " << widget->font;
    ostr << " GROUP " << widget->group;
    save_flag (widget->usable, ostr, " USABLE", " NONUSABLE");
    save_flag (widget->anchor_right, ostr, " RIGHTANCHOR", " LEFTANCHOR");
    save_flag (widget->toggled, ostr, " CHECKED");
    
    ostr << std::endl;
}

static void save_widget_list (Widgets::List *widget, std::ostream &ostr)
{
    gchar *width_str;

    if (widget->manual_width)
	width_str = g_strdup_printf ("%d", int (widget->width));
    else
	width_str = g_strdup ("AUTO");
    
    ostr << "  LIST";
    const std::vector<std::string> &items = widget->items;
    for (std::vector<std::string>::const_iterator i = items.begin ();
	 i != items.end (); i++)
	ostr << " \"" << *i << "\"";
    
    ostr << " ID " << widget->id;
    ostr << " AT (" << widget->x << " " << widget->y << " " << width_str << " " << " AUTO)";
    ostr << " FONT " << widget->font;
    save_flag (widget->usable, ostr, " USABLE", " NONUSABLE");

    ostr << " VISIBLEITEMS " << widget->visible_items;
    
    ostr << std::endl;

    g_free (width_str);
}

static void save_widget_selector_trigger (Widgets::SelectorTrigger *widget, std::ostream &ostr)
{
    ostr << "  SELECTORTRIGGER \"" << widget->text << "\" ID " << widget->id;
    save_element_pos_autosize (widget, ostr);

    ostr << " FONT " << widget->font;
    save_flag (widget->usable, ostr, " USABLE", " NONUSABLE");
    save_flag (widget->anchor_right, ostr, " RIGHTANCHOR", " LEFTANCHOR");
    
    ostr << std::endl;
}

static void save_widget_popup_trigger (Widgets::PopupTrigger *widget, std::ostream &ostr)
{
    ostr << "  POPUPTRIGGER \"" << widget->text << "\" ID " << widget->id;
    save_element_pos_autosize (widget, ostr);

    ostr << " FONT " << widget->font;
    save_flag (widget->usable, ostr, " USABLE", " NONUSABLE");
    save_flag (widget->anchor_right, ostr, " RIGHTANCHOR", " LEFTANCHOR");
    
    ostr << std::endl;

    if (widget->list_id != "")
	ostr << "  POPUPLIST ID " << widget->id << " " << widget->list_id << std::endl;
}

static void save_widget_scrollbar (Widgets::ScrollBar *widget, std::ostream &ostr)
{
    gchar *width_str, *height_str;

    width_str = g_strdup_printf ("%d", widget->get_width ());
    height_str = g_strdup_printf ("%d", int (widget->height));

    ostr << "  SCROLLBAR ID " << widget->id;
    ostr << " AT (" << widget->x << " " << widget->y
	 << " " << width_str << " " << height_str << ")";

    ostr << " VALUE "    << widget->value
	 << " MIN "      << widget->min_value
	 << " MAX "      << widget->max_value
	 << " PAGESIZE " << widget->page_size;
    ostr << std::endl;
    
    g_free (height_str);
    g_free (width_str);
}

static void save_widget_text_field (Widgets::TextField *widget, std::ostream &ostr)
{
    gchar *width_str = 0, *height_str = 0;

    width_str = g_strdup_printf ("%d", widget->get_width ());
    
    if (widget->manual_height)
	height_str = g_strdup_printf ("%d", int (widget->height));
    else
	height_str = g_strdup ("AUTO");
    
    ostr << "  FIELD ID " << widget->id;
    ostr << " AT (" << widget->x << " " << widget->y << " "
	 << width_str << " " << height_str << ")";

    g_free (height_str);
    g_free (width_str);

    ostr << " FONT " << widget->font;
    save_flag (widget->usable, ostr, " USABLE", " NONUSABLE");
    save_flag (widget->editable, ostr, " EDITABLE", " NONEDITABLE");
    save_flag (widget->multi_line, ostr, " MULTIPLELINES", " SINGLELINE");
    save_flag (widget->justify_right, ostr, " RIGHTALING", " LEFTALIGN");
    save_flag (widget->underline, ostr, " UNDERLINED");
    save_flag (widget->dynamic_size, ostr, " DYNAMICSIZE");
    save_flag (widget->auto_shift, ostr, " AUTOSHIFT");
    save_flag (widget->numeric, ostr, " NUMERIC");
    save_flag (widget->has_scrollbar, ostr, " HASSCROLLBAR");

    if (widget->max_length)
	ostr << " MAXCHARS " << widget->max_length;
    else
	ostr << " MAXCHARS 32767";

    ostr << std::endl;
}

static void save_widget_table (Widgets::Table *widget, std::ostream &ostr)
{   
    ostr << "  TABLE ID " << widget->id;
    ostr << " AT (" << widget->x << " " << widget->y << " AUTO AUTO)";
    ostr << " ROWS "   << widget->num_rows;

    const std::vector<int>& columns = widget->column_width;
    ostr << " COLUMNS " << widget->num_columns;
    ostr << " COLUMNWIDTHS ";
    for (int i = 0; i < widget->num_columns; i++)
	ostr << columns[i] << " ";
	
    ostr << std::endl;
}

static void save_widget_gadget (Widgets::Gadget *widget, std::ostream &ostr)
{
    ostr << "  GADGET ID " << widget->id;
    save_element_pos_size (widget, ostr);
    save_flag (widget->usable, ostr, " USABLE", " NONUSABLE");
	
    ostr << std::endl;
}
    
} // namespace RCPSaver_funcs
} // namespace IO
} // namespace Guikachu
