//$Id: guikachu-save.cc,v 1.2 2002/06/30 18:35:43 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "guikachu-save.h"

#include "storage.h"
#include "storage-node.h"

#include "resource-util.h"

#include "blob-res.h"
#include "dialog-res.h"
#include "menu-res.h"
#include "string-res.h"
#include "stringlist-res.h"
#include "guikachu-save-form.h"

namespace Guikachu
{
    
namespace IO
{
    
namespace GuikachuSaver_funcs
{
    static void save_app      (Resources::Application *app, StorageNode &root_node);
    static void save_target   (Target *target,              StorageNode &root_node);
    static void save_resource (Resource *resource,          StorageNode &root_node);
    
    static void save_res_blob       (Resources::Blob *res,       StorageNode &node);
    static void save_res_dialog     (Resources::Dialog *res,     StorageNode &node);
    static void save_res_menu       (Resources::Menu *res,       StorageNode &node);
    static void save_res_string     (Resources::String *res,     StorageNode &node);
    static void save_res_stringlist (Resources::StringList *res, StorageNode &node);
}

    
IOResult GuikachuSaver::save (ResourceManager *manager, const std::string &filename)
{
    Storage     storage;
    StorageNode root_node = storage.create_root ("guikachu");

    // Save per-application data
    GuikachuSaver_funcs::save_app (manager->get_application (), root_node);

    // Save target machine data
    GuikachuSaver_funcs::save_target (manager->get_target (), root_node);
    
    // Save resources
    std::set<Resource*> resources = manager->get_resources ();
    for (std::set<Resource*>::const_iterator i = resources.begin ();
	 i != resources.end (); i++)
	GuikachuSaver_funcs::save_resource (*i, root_node);

    storage.save (filename);

    return RESULT_OK;
}

    
namespace GuikachuSaver_funcs
{
    
void save_app (Resources::Application *app, StorageNode &root_node)
{
    StorageNode node = root_node.add_node ("application");
    
    if (app->iconname != "")
        node.add_node ("iconname", app->iconname);
    if (app->version != "")
        node.add_node ("version",  app->version);
    if (app->vendor != "")
        node.add_node ("vendor",   app->vendor);
}

void save_target (Target *target, StorageNode &root_node)
{
    StorageNode node = root_node.add_node ("target");
    
    if (target->get_stock_id () != "")
    {
        node.set_prop ("stock_id", target->get_stock_id ());
        return;
    }
    
    StorageNode screen_node = node.add_node ("screen");
    screen_node.set_prop ("width",  target->screen_width);
    screen_node.set_prop ("height", target->screen_height);
}

void save_resource (Resource *res, StorageNode &root_node)
{
    std::string curr_node_name = Resources::type_id_from_type (res->get_type ());
    
    StorageNode node = root_node.add_node (curr_node_name);
    node.set_prop ("id", res->id);
	
    switch (res->get_type ())
    {
    case Resources::RESOURCE_BLOB:
	save_res_blob (static_cast<Resources::Blob*> (res), node);
	break;
    case Resources::RESOURCE_DIALOG:
	save_res_dialog (static_cast<Resources::Dialog*> (res), node);
	break;
    case Resources::RESOURCE_MENU:
	save_res_menu (static_cast<Resources::Menu*> (res), node);
	break;
    case Resources::RESOURCE_STRING:
	save_res_string (static_cast<Resources::String*> (res), node);
	break;
    case Resources::RESOURCE_STRINGLIST:
	save_res_stringlist (static_cast<Resources::StringList*> (res), node);
	break;
    case Resources::RESOURCE_FORM:
	save_res_form (static_cast<Resources::Form*> (res), node);
	break;
    }
}

static void save_res_blob (Resources::Blob *res, StorageNode &node)
{
    node.set_content (res->contents);
}

static void save_res_dialog (Resources::Dialog *res, StorageNode &node)
{
    std::string type_str;

    /* Dialog type */
    switch (res->dialog_type)
    {
    case Resources::Dialog::TYPE_INFORMATION:
        type_str = "information";
        break;
    case Resources::Dialog::TYPE_CONFIRMATION:
        type_str = "confirmation";
        break;
    case Resources::Dialog::TYPE_WARNING:
        type_str = "warning";
        break;
    case Resources::Dialog::TYPE_ERROR:
        type_str = "error";
    }

    StorageNode my_node = node.add_node ("type");
    my_node.set_prop ("value", type_str);

    /* Title */
    node.add_node ("title", res->title);
    /* Text */
    node.add_node ("text", res->text);

    /* Help ID */
    if (res->help_id != "")
    {
        my_node = node.add_node ("help");
        my_node.set_prop ("id", res->help_id);
    }

    /* Buttons */
    const std::vector<std::string> &buttons = res->buttons;
    if (buttons.size ())
    {
        my_node = node.add_node ("buttons");
        my_node.set_prop ("default", res->default_button);
        for (std::vector <std::string>::const_iterator i = buttons.begin ();
             i != buttons.end (); i++)
        {
            StorageNode button_node = my_node.add_node ("button");
            button_node.set_prop ("label", *i);
        }
    }

}

static void save_res_menu (Resources::Menu *res, StorageNode &node)
{
    const Resources::Menu::MenuTree& submenus = res->get_submenus ();
    
    for (Resources::Menu::MenuTree::const_iterator i = submenus.begin ();
         i != submenus.end (); i++)
    {
        StorageNode submenu_node = node.add_node ("submenu");
        submenu_node.set_prop ("label", i->label);

        for (std::vector<Resources::Menu::MenuItem>::const_iterator j = i->items.begin ();
             j != i->items.end (); j++)
        {
            if (j->separator)
            {
                submenu_node.add_node ("separator");
            } else {
                StorageNode item_node = submenu_node.add_node ("menuitem");
                item_node.set_prop ("id",    j->id);
                item_node.set_prop ("label", j->label);

                if (j->shortcut)
                    item_node.set_prop ("shortcut", j->shortcut);
            }
        }
    }
}

static void save_res_string (Resources::String *res, StorageNode &node)
{
    node.add_node ("text", res->text);
}

static void save_res_stringlist (Resources::StringList *res, StorageNode &node)
{
    node.add_node ("prefix", res->prefix);
    
    const std::vector<std::string> &strings = res->strings;
    for (std::vector<std::string>::const_iterator i = strings.begin ();
         i != strings.end (); i++)
        node.add_node ("text", *i);
}
    
} // namespace GuikachuSaver_funcs
} // namespace IO
} // namespace Guikachu
