//$Id: guikachu-save-form.cc,v 1.2 2002/06/30 18:35:43 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "guikachu-save-form.h"

#include "form-editor/widget-util.h"

#include "form-editor/label.h"
#include "form-editor/button.h"
#include "form-editor/pushbutton.h"
#include "form-editor/graffiti.h"
#include "form-editor/selector-trigger.h"
#include "form-editor/checkbox.h"
#include "form-editor/list.h"
#include "form-editor/scrollbar.h"
#include "form-editor/text-field.h"
#include "form-editor/table.h"
#include "form-editor/popup-trigger.h"
#include "form-editor/gadget.h"

namespace Guikachu
{
    
namespace IO
{
    
namespace GuikachuSaver_funcs
{

static void save_widget (Widget *widget, StorageNode &node);
    
static void save_widget_label      (Widgets::Label      *widget, StorageNode &node);
static void save_widget_button     (Widgets::Button     *widget, StorageNode &node);
static void save_widget_pushbutton (Widgets::PushButton *widget, StorageNode &node);
static void save_widget_graffiti   (Widgets::Graffiti   *widget, StorageNode &node);
static void save_widget_checkbox   (Widgets::Checkbox   *widget, StorageNode &node);
static void save_widget_list       (Widgets::List       *widget, StorageNode &node);
static void save_widget_scrollbar  (Widgets::ScrollBar  *widget, StorageNode &node);
static void save_widget_text_field (Widgets::TextField  *widget, StorageNode &node);
static void save_widget_table      (Widgets::Table      *widget, StorageNode &node);
static void save_widget_gadget     (Widgets::Gadget     *widget, StorageNode &node);

static void save_widget_selector_trigger (Widgets::SelectorTrigger *widget,
					  StorageNode              &node);
static void save_widget_popup_trigger    (Widgets::PopupTrigger    *widget,
					  StorageNode              &node);


template<class T>
static StorageNode create_pos_node (T *res, StorageNode &root_node)
{
    StorageNode node = root_node.add_node ("pos");
    
    node.set_prop ("x", res->x);
    node.set_prop ("y", res->y);

    return node;
}

template<class T>
static void save_element_pos (T *res, StorageNode &root_node)
{
    create_pos_node (res, root_node);
}

template<class T>
static void save_element_pos_size (T *res, StorageNode &root_node)
{
    StorageNode node = create_pos_node (res, root_node);

    node.set_prop ("width",  res->width);
    node.set_prop ("height", res->height);
}

template<class T>
static void save_element_pos_autosize (T *res, StorageNode &root_node)
{
    StorageNode node = create_pos_node (res, root_node);
    
    if (res->manual_width)
	node.set_prop ("width", res->width);
    else
	node.set_prop ("width", "auto");

    if (res->manual_height)
	node.set_prop ("height", res->height);
    else
	node.set_prop ("height", "auto");
}

static void save_element_text (Widgets::Textual *res, StorageNode &root_node)
{
    StorageNode node = root_node.add_node ("text", res->text);
    node.set_prop ("font", res->font);
}

void save_res_form (Resources::Form *res, StorageNode &node)
{
    /* Title */
    node.add_node ("title", res->title);
    
    /* Position */
    save_element_pos_size (res, node);
    
    /* Help ID */
    if (res->help_id != "")
	node.add_node ("help").set_prop ("id", res->help_id);

    /* Menu ID */
    if (res->menu_id != "")
	node.add_node ("menu").set_prop ("id", res->menu_id);

    if (res->modal)
	node.add_node ("modal");
    if (res->frame)
	node.add_node ("frame");
    if (res->savebehind)
	node.add_node ("savebehind");

    /* Widgets */
    const std::set<Widget*> &widgets = res->get_widgets ();
    if (widgets.size ())
    {
	StorageNode widgetlist_node = node.add_node ("widgets");

	if (res->def_button != "")
	    widgetlist_node.set_prop ("default", res->def_button);
	
	for (std::set<Widget*>::const_iterator i = widgets.begin ();
	     i != widgets.end (); i++)
	    save_widget (*i, widgetlist_node);
    }
}

static void save_widget (Widget *widget, StorageNode &node)
{
    std::string curr_node_name = Widgets::type_id_from_type (widget->get_type ());
    
    StorageNode widget_node = node.add_node (curr_node_name);
    widget_node.set_prop ("id", widget->id);

    switch (widget->get_type ())
    {
    case Widgets::WIDGET_LABEL:
	save_widget_label (static_cast <Widgets::Label*> (widget), widget_node);
	break;
    case Widgets::WIDGET_BUTTON:
	save_widget_button (static_cast <Widgets::Button*> (widget), widget_node);
	break;
    case Widgets::WIDGET_PUSHBUTTON:
	save_widget_pushbutton (static_cast <Widgets::PushButton*> (widget), widget_node);
	break;
    case Widgets::WIDGET_GRAFFITI:
	save_widget_graffiti (static_cast <Widgets::Graffiti*> (widget), widget_node);
	break;
    case Widgets::WIDGET_SELECTOR_TRIGGER:
	save_widget_selector_trigger (static_cast <Widgets::SelectorTrigger*> (widget), widget_node);
	break;
    case Widgets::WIDGET_CHECKBOX:
	save_widget_checkbox (static_cast <Widgets::Checkbox*> (widget), widget_node);
	break;
    case Widgets::WIDGET_LIST:
	save_widget_list (static_cast <Widgets::List*> (widget), widget_node);
	break;
    case Widgets::WIDGET_POPUP_TRIGGER:
	save_widget_popup_trigger (static_cast <Widgets::PopupTrigger*> (widget), widget_node);
	break;
    case Widgets::WIDGET_SCROLLBAR:
	save_widget_scrollbar (static_cast <Widgets::ScrollBar*> (widget), widget_node);
	break;
    case Widgets::WIDGET_TEXT_FIELD:
	save_widget_text_field (static_cast <Widgets::TextField*> (widget), widget_node);
	break;
    case Widgets::WIDGET_TABLE:
	save_widget_table (static_cast <Widgets::Table*> (widget), widget_node);
	break;
    case Widgets::WIDGET_GADGET:
	save_widget_gadget (static_cast <Widgets::Gadget*> (widget), widget_node);
	break;
    case Widgets::WIDGET_NONE:
	g_assert_not_reached ();
	break;
    }    
}

static void save_widget_label (Widgets::Label *widget, StorageNode &node)
{
    /* Position */
    save_element_pos (widget, node);

    /* Usable */
    if (widget->usable)
	node.add_node ("usable");
    
    /* Text and font */
    save_element_text (widget, node);
}

static void save_widget_button (Widgets::Button *widget, StorageNode &node)
{    
    /* Position and size */
    save_element_pos_autosize (widget, node);

    /* Frame */
    if (widget->frame != Widgets::Button::FRAME_NONE)
    {
	StorageNode frame_node = node.add_node ("frame");
	if (widget->frame == Widgets::Button::FRAME_BOLD)
	    frame_node.set_prop ("bold", "bold");
    }
    
    /* Usable */
    if (widget->usable)
	node.add_node ("usable");

    /* Repeating */
    if (widget->repeat)
	node.add_node ("repeat");

    /* Anchor_right */
    if (widget->anchor_right)
	node.add_node ("anchor_right");
    
    /* Label and font */
    save_element_text (widget, node);
}

static void save_widget_pushbutton (Widgets::PushButton *widget, StorageNode &node)
{
    save_element_pos_autosize (widget, node);
    
    /* Usable */
    if (widget->usable)
	node.add_node ("usable");

    /* Label and font */
    save_element_text (widget, node);
    
    /* Group ID */
    node.add_node ("group").set_prop ("id", widget->group);
}

static void save_widget_graffiti (Widgets::Graffiti *widget, StorageNode &node)
{
    save_element_pos (widget, node);
}

static void save_widget_checkbox (Widgets::Checkbox *widget, StorageNode &node)
{
    save_element_pos_autosize (widget, node);
    
    /* Usable */
    if (widget->usable)
	node.add_node ("usable");

    /* Group ID */
    node.add_node ("group").set_prop ("id", widget->group);
    
    /* Toggled */
    if (widget->toggled)
	node.add_node ("toggled");
    
    /* Anchor_right */
    if (widget->anchor_right)
	node.add_node ("anchor_right");
    
    /* Label and font */
    save_element_text (widget, node);
}

static void save_widget_list (Widgets::List *widget, StorageNode &node)
{
    /* Position */
    StorageNode pos_node = create_pos_node (widget, node);

    /* Size */
    if (widget->manual_width)
	pos_node.set_prop ("width", widget->width);
    else
	pos_node.set_prop ("width", "auto");

    /* Usable */
    if (widget->usable)
	node.add_node ("usable");

    /* List items */
    StorageNode item_node = node.add_node ("items");
    item_node.set_prop ("font", widget->font);
    item_node.set_prop ("visible", widget->visible_items);

    const std::vector<std::string> &items = widget->items ();
    for (std::vector<std::string>::const_iterator i = items.begin ();
	 i != items.end (); i++)
	item_node.add_node ("item", *i);
}

static void save_widget_selector_trigger (Widgets::SelectorTrigger *widget,
					  StorageNode              &node)
{
    save_element_pos_autosize (widget, node);
    
    /* Usable */
    if (widget->usable)
	node.add_node ("usable");

    /* Anchor_right */
    if (widget->anchor_right)
	node.add_node ("anchor_right");
    
    /* Label and font */
    save_element_text (widget, node);
}

static void save_widget_popup_trigger (Widgets::PopupTrigger *widget,
				       StorageNode           &node)
{
    save_element_pos_autosize (widget, node);

    /* Usable */
    if (widget->usable)
	node.add_node ("usable");

    /* Anchor_right */
    if (widget->anchor_right)
	node.add_node ("anchor_right");
    
    /* Label and font */
    save_element_text (widget, node);

    if (widget->list_id != "")
	node.add_node ("popuplist").set_prop ("id", widget->list_id);
}

static void save_widget_scrollbar (Widgets::ScrollBar *widget, StorageNode &node)
{
    /* Position */
    StorageNode pos_node = create_pos_node (widget, node);

    /* Size */
    if (widget->manual_width)
	pos_node.set_prop ("width", widget->width);
    else
	pos_node.set_prop ("width", "auto");

    pos_node.set_prop ("height", widget->height);
    
    /* Usable */
    if (widget->usable)
	node.add_node ("usable");

    /* Values */
    StorageNode val_node = node.add_node ("values");
    val_node.set_prop ("min",       widget->min_value);
    val_node.set_prop ("max",       widget->max_value);
    val_node.set_prop ("page_size", widget->page_size);
    val_node.set_prop ("value",     widget->value);
}

static void save_widget_text_field (Widgets::TextField *widget, StorageNode &node)
{
    /* Position */
    StorageNode pos_node = create_pos_node (widget, node);

    /* Size */
    pos_node.set_prop ("width", widget->width);

    if (widget->manual_height)
	pos_node.set_prop ("height", widget->height);
    else
	pos_node.set_prop ("height", "auto");
    
    /* Flags */
    if (widget->usable)
	node.add_node ("usable");
    if (widget->editable)
	node.add_node ("editable");
    if (widget->multi_line)
	node.add_node ("multi_line");
    if (widget->auto_shift)
	node.add_node ("auto_shift");
    if (widget->numeric)
	node.add_node ("numeric");

    if (widget->underline)
	node.add_node ("underline");
    if (widget->dynamic_size)
	node.add_node ("dynamic_size");
    if (widget->justify_right)
	node.add_node ("justify_right");
    if (widget->has_scrollbar)
	node.add_node ("has_scrollbar");

    /* Font and length */
    StorageNode text_node = node.add_node ("text");
    text_node.set_prop ("font", widget->font);
    text_node.set_prop ("max_length", widget->max_length);
}

static void save_widget_table (Widgets::Table *widget, StorageNode &node)
{
    save_element_pos (widget, node);

    StorageNode data_node = node.add_node ("table_data");

    /* Number of rows and columns */
    data_node.set_prop ("rows", widget->num_rows);

    const std::vector<int> &columns = widget->column_width;
    for (int i = 0; i < widget->num_columns; i++)
	data_node.add_node ("column").set_prop ("width", columns[i]);
}

static void save_widget_gadget (Widgets::Gadget *widget, StorageNode &node)
{
    save_element_pos_size (widget, node);

    /* Usable */
    if (widget->usable)
	node.add_node ("usable");
}
    
} // namespace GuikachuSaver_funcs
} // namespace IO
} // namespace Guikachu
