//$Id: guikachu-load.cc,v 1.3 2002/06/30 18:35:43 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "guikachu-load.h"

#include "config.h"
#include <libgnome/libgnome.h>

/* Having to include these for just open()/close() and *time() sure sucks */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "storage.h"
#include "storage-node.h"
#include "storage-processor.h"
#include "storage-node-handlers.h"

#include "mainwin.h" // FIXME: calling the GUI directly from an IO
                     // handler is Not Good

#include "blob-res.h"
#include "dialog-res.h"
#include "menu-res.h"
#include "string-res.h"
#include "stringlist-res.h"
#include "guikachu-load-form.h"

using namespace Guikachu;

namespace Guikachu
{
    
namespace IO
{
    
namespace GuikachuLoader_funcs
{   
    static void load_app      (Resources::Application *app, const StorageNode &node);
    static void load_target   (Target *target,              const StorageNode &node);
    static void load_resource (ResourceManager *manager,    StorageNode &node);
    
    static void load_res_blob       (Resources::Blob *res,       const StorageNode &node);
    static void load_res_dialog     (Resources::Dialog *res,     const StorageNode &node);
    static void load_res_menu       (Resources::Menu *res,       const StorageNode &node);
    static void load_res_string     (Resources::String *res,     const StorageNode &node);
    static void load_res_stringlist (Resources::StringList *res, const StorageNode &node);
}


void GuikachuLoader::encoding_fixed_cb ()
{
    encoding_fixed = true;
}


IOResult GuikachuLoader::load (ResourceManager *manager, const std::string &filename)
{
    Storage storage;

    int fd = open (filename.c_str (), O_RDONLY);
    if (fd == -1)
        return RESULT_NOTFOUND;
    close (fd);
        
    
    if (!storage.load (filename))
        return RESULT_INVALID;    
    
    StorageNode root_node = storage.get_root ();
    if (root_node.name () != "guikachu")
        return RESULT_INVALID;

    // Reset context
    encoding_fixed = false;

    // Set up callbacks
    storage.encoding_notify.connect (SigC::slot (this, &GuikachuLoader::encoding_fixed_cb));
    
    // Walk the tree
    for (StorageNode curr_node = root_node.children ();
         curr_node; curr_node++)
    {
        if (curr_node.name () == "application")
	    GuikachuLoader_funcs::load_app (manager->get_application (), curr_node);
        
        else if (curr_node.name () == "target")
	    GuikachuLoader_funcs::load_target (manager->get_target (), curr_node);
        
        else
            GuikachuLoader_funcs::load_resource (manager, curr_node);
    }

    // Show warning if 8-bit input was encountered
    if (encoding_fixed)
    {
        encoding_warning ();
        manager->set_dirty ();
    }
    
    return RESULT_OK;
}

void GuikachuLoader::encoding_warning ()
{
    GUI::show_warning (
	_("Previous releases of Guikachu always stored\n"
	  "strings in the local encoding, while PalmOS\n"
	  "uses the Latin-1 character  set. The encoding\n"
	  "of old Guikachu files can not be determined,\n"
	  "so all non-ASCII characters have been\n"
	  "automatically converted to escape sequences.\n"
	  "\n"
	  "Please review the results of the automatic\n"
	  "conversion"),
	_("Input file contains non-ASCII characters"));
}




namespace GuikachuLoader_funcs
{
   
static void load_app (Resources::Application *app, const StorageNode &node)
{
    app->iconname = "";
    app->version =  "";
    app->vendor =   "";

    StorageProcessor processor;
    processor.add_handler ("iconname", new StorageNodeHandlers::StringContents (app->iconname));
    processor.add_handler ("version",  new StorageNodeHandlers::StringContents (app->version));
    processor.add_handler ("vendor",   new StorageNodeHandlers::StringContents (app->vendor));

    processor.run (node);
}

static void load_target (Target *target, const StorageNode &node)
{
    // Is this a stock target?
    std::string stock_id = node.get_prop_string ("stock_id");
    if (stock_id != "")
    {
        target->load_stock (stock_id);
        return;
    }
    
    for (StorageNode curr_node = node.children (); curr_node; curr_node++)
    {
        if (curr_node.name () == "screen")
        {
            target->screen_width  = curr_node.get_prop_int ("width");
            target->screen_height = curr_node.get_prop_int ("height");
        } else {
            g_warning ("Error parsing custom target definition: "
                       "unexpected element `%s'",
                       curr_node.name ().c_str ());
        }
    }
}

static void load_resource (ResourceManager *manager, StorageNode &node)
{
    std::string node_name = node.name ();
    std::string id = node.get_prop_string ("id");

    if (id == "")
    {
        g_warning ("`%s': Missing ID field", node_name.c_str ());
        return;
    }

    
    if (node_name == "blob")
    {
	load_res_blob (static_cast<Resources::Blob*> (
	    manager->create_resource (Resources::RESOURCE_BLOB, id)), node);
    }
    else if (node_name == "dialog")
    {
	load_res_dialog (static_cast<Resources::Dialog*> (
	    manager->create_resource (Resources::RESOURCE_DIALOG, id)), node);
    }
    else if (node_name == "form")
    {
	load_res_form (static_cast<Resources::Form*> (
	    manager->create_resource (Resources::RESOURCE_FORM, id)), node);
    }
    else if (node_name == "menu")
    {
	load_res_menu (static_cast<Resources::Menu*> (
	    manager->create_resource (Resources::RESOURCE_MENU, id)), node);
    }
    else if (node_name == "string")
    {
	load_res_string (static_cast<Resources::String*> (
	    manager->create_resource (Resources::RESOURCE_STRING, id)), node);
    }
    else if (node_name == "stringlist")
    {
	load_res_stringlist (static_cast<Resources::StringList*> (
	    manager->create_resource (Resources::RESOURCE_STRINGLIST, id)), node);
    }
}

static void load_res_blob (Resources::Blob *res, const StorageNode &node)
{
    res->contents = node.get_content ();
}

static void load_res_dialog (Resources::Dialog *res, const StorageNode &node)
{
    // Create a bunch of 'lambda functors'
    class DialogTypeHandler: public StorageNodeHandlers::HandlerBase
    {
	Property<Resources::Dialog::DialogType> &target;
    public:
	explicit DialogTypeHandler (Property<Resources::Dialog::DialogType> &target_) :
	    target (target_)
	    {};
	
	void operator() (const StorageNode &node)
	    {
		const std::string &xml_type = node.get_prop_string ("value");
		
		if (xml_type == "information")
		    target = Resources::Dialog::TYPE_INFORMATION;
		else if (xml_type == "confirmation")
		    target = Resources::Dialog::TYPE_CONFIRMATION;
		else if (xml_type == "warning")
		    target = Resources::Dialog::TYPE_WARNING;
		else if (xml_type == "error")
		    target = Resources::Dialog::TYPE_ERROR;
	    };
    };

    class ButtonsHandler: public StorageNodeHandlers::StringListProp
    {
	Resources::Dialog *res;
    public:
	explicit ButtonsHandler (Resources::Dialog *res_):
	    StorageNodeHandlers::StringListProp ("button", "label", res_->buttons),
	    res (res_)
	    {};

	void operator() (const StorageNode &node)
	    {
		/* Set default button */
		res->default_button = node.get_prop_int ("default");
		
		/* Iterate buttons */
		StorageNodeHandlers::StringListProp::operator() (node);
	    };
    };
    
    StorageProcessor processor;
    processor.add_handler ("text",    new StorageNodeHandlers::StringContents (res->text));
    processor.add_handler ("title",   new StorageNodeHandlers::StringContents (res->title));
    processor.add_handler ("help",    new StorageNodeHandlers::StringProp ("id", res->help_id));
    processor.add_handler ("type",    new DialogTypeHandler (res->dialog_type));
    processor.add_handler ("buttons", new ButtonsHandler (res));

    processor.run (node);
}

static void load_res_menu (Resources::Menu *res, const StorageNode &node)
{
    Resources::Menu::MenuTree submenus;
    
    for (StorageNode curr_node = node.children (); curr_node; ++curr_node)
    {
        if (curr_node.name () == "submenu")
        {
            Resources::Menu::Submenu submenu;
            std::string label = curr_node.get_prop_string ("label");
            
            if (label == "")
                break;
            
            submenu.label = label;
            
            for (StorageNode sub_node = curr_node.children ();
                 sub_node; sub_node++)
            {
                Resources::Menu::MenuItem menu_item;
                if (sub_node.name () == "menuitem")
                {
                    std::string id = sub_node.get_prop_string ("id");
                    std::string label = sub_node.get_prop_string ("label");
                    char        shortcut = sub_node.get_prop_char ("shortcut");
		    
                    if (id == "" || label == "")
                        break;

                    menu_item.separator = false;

                    menu_item.id = id;
                    menu_item.label = label;                
                    menu_item.shortcut = shortcut;

                    submenu.items.push_back (menu_item);
                }
                else if (sub_node.name () == "separator")
                {
                    menu_item.separator = true;

                    submenu.items.push_back (menu_item);
                }
                else
                    g_warning ("Error parsing `menu' resource: "
                               "unexpected element `%s/%s'",
                               curr_node.name ().c_str (),
                               sub_node.name ().c_str ());
            }
	    
            submenus.push_back (submenu);
	}
	else
	    g_warning ("Error parsing `menu' resource: "
		       "unexpected element `%s'", curr_node.name ().c_str ());
    }

    res->set_submenus (submenus);
}

static void load_res_string (Resources::String *res, const StorageNode &node)
{
    StorageProcessor processor;
    processor.add_handler ("text", new StorageNodeHandlers::StringContents (res->text));

    processor.run (node);
}

static void load_res_stringlist (Resources::StringList *res, const StorageNode &node)
{
    std::vector<std::string> strings_tmp;
    
    for (StorageNode curr_node = node.children (); curr_node; ++curr_node)
    {
        if (curr_node.name () == "text")
            strings_tmp.push_back (curr_node.get_content ());
        else if (curr_node.name () == "prefix")
            res->prefix = curr_node.get_content ();
        else
            g_warning ("Error parsing `stringlist' resource: "
                       "unexpected element `%s'", curr_node.name ().c_str ());
    }

    res->strings = strings_tmp;
}

    
} // namespace GuikachuLoader_funcs
} // namespace IO
} // namespace Guikachu
