//$Id: guikachu-load-form.cc,v 1.2 2002/06/30 18:35:43 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "guikachu-load-form.h"

#include "storage-node.h"
#include "storage-processor.h"
#include "storage-node-handlers.h"

#include "form-editor/widget.h"
#include "form-editor/widget-util.h"

#include "form-editor/label.h"
#include "form-editor/button.h"
#include "form-editor/pushbutton.h"
#include "form-editor/graffiti.h"
#include "form-editor/selector-trigger.h"
#include "form-editor/checkbox.h"
#include "form-editor/list.h"
#include "form-editor/scrollbar.h"
#include "form-editor/text-field.h"
#include "form-editor/table.h"
#include "form-editor/popup-trigger.h"
#include "form-editor/gadget.h"

namespace Guikachu
{
    
namespace IO
{
    
namespace GuikachuLoader_funcs
{

static void load_widget (Resources::Form *parent_form, StorageNode &node);

static void load_widget_label      (Widgets::Label      *widget, StorageNode &node);
static void load_widget_button     (Widgets::Button     *widget, StorageNode &node);
static void load_widget_pushbutton (Widgets::PushButton *widget, StorageNode &node);
static void load_widget_graffiti   (Widgets::Graffiti   *widget, StorageNode &node);
static void load_widget_checkbox   (Widgets::Checkbox   *widget, StorageNode &node);
static void load_widget_list       (Widgets::List       *widget, StorageNode &node);
static void load_widget_scrollbar  (Widgets::ScrollBar  *widget, StorageNode &node);
static void load_widget_text_field (Widgets::TextField  *widget, StorageNode &node);
static void load_widget_table      (Widgets::Table      *widget, StorageNode &node);
static void load_widget_gadget     (Widgets::Gadget     *widget, StorageNode &node);

static void load_widget_selector_trigger (Widgets::SelectorTrigger *widget,
					  StorageNode              &node);
static void load_widget_popup_trigger    (Widgets::PopupTrigger    *widget,
					  StorageNode              &node);

// Useful for small ("lambda-like") node handlers
#define HANDLER(name, datatype, dataname, impl)                   \
    class name: public Guikachu::StorageNodeHandlers::HandlerBase \
    {                                                             \
        datatype dataname;                                        \
    public:                                                       \
        explicit name (datatype data): dataname (data)            \
            {};                                                   \
                                                                  \
        void operator() (const StorageNode &node)                 \
            impl;                                                 \
    };


// Code of XML node handlers shared between widgets
template<class T>
class PositionHandler: public StorageNodeHandlers::HandlerBase
{
    T *element;
public:
    PositionHandler (T *element_): element (element_)
    {}

    void operator() (const StorageNode &node)
    {
	element->x = node.get_prop_int ("x");
	element->y = node.get_prop_int ("y");
    }
};

typedef PositionHandler<Widget> WidgetPositionHandler;

class SizePosHandler: public StorageNodeHandlers::HandlerBase
{
protected:
    Widgets::Resizeable *element;
    
public:
    SizePosHandler (Widgets::Resizeable *element_): element (element_)
    {}

protected:
    virtual void load_width (const StorageNode &node)
    {
	std::string width_str = node.get_prop_string ("width");
	
	element->width = atoi (width_str.c_str ());
	
	Widgets::AutoWidth *autowidth =
	    dynamic_cast<Widgets::AutoWidth*> (element);
	
	if (autowidth)
	    if (width_str == "auto")
		autowidth->manual_width = false;
	    else
		autowidth->manual_width = true;
    }

    virtual void load_height (const StorageNode &node)
    {
	std::string height_str = node.get_prop_string ("height");
	
	element->height = atoi (height_str.c_str ());
	
	Widgets::AutoHeight *autoheight =
	    dynamic_cast<Widgets::AutoHeight*> (element);
	
	if (autoheight)
	    if (height_str == "auto")
		autoheight->manual_height = false;
	    else
		autoheight->manual_height = true;
    }
    
public:
    void operator() (const StorageNode &node)
    {
	element->set_x (node.get_prop_int ("x"));
	element->set_y (node.get_prop_int ("y"));
	
	load_width (node);
	load_height (node);
    }
};

class TextHandler: public StorageNodeHandlers::HandlerBase
{
    Widgets::Textual *element;
public:
    TextHandler (Widgets::Textual *element_): element (element_)
    {}

    void operator() (const StorageNode &node)
    {
	element->font = node.get_prop_int ("font");
	element->text = node.get_content ();
    }
};
    
template<class T>
static void load_element_position (T *res, StorageNode &node)
{
    res->x = node.get_prop_int ("x");
    res->y = node.get_prop_int ("y");
}

template<class T>
static void load_element_size (T *res, StorageNode &node)
{
    res->width  = node.get_prop_int ("width");
    res->height = node.get_prop_int ("height");
}

void load_res_form (Resources::Form *res, StorageNode &node)
{
    // Defaults for binary tags
    res->frame = false;
    res->modal = false;
    res->savebehind = false;
    
    for (StorageNode curr_node = node.children ();
         curr_node; curr_node++)
    {
        if (curr_node.name () == "title")
            res->title = curr_node.get_content ();
        else if (curr_node.name () == "help")
            res->help_id = curr_node.get_prop_string ("id");
        else if (curr_node.name () == "menu")
            res->menu_id = curr_node.get_prop_string ("id");
        else if (curr_node.name () == "modal")
            res->modal = true;
        else if (curr_node.name () == "frame")
            res->frame = true;
        else if (curr_node.name () == "savebehind")
            res->savebehind = true;
	else if (curr_node.name () == "pos")
        {
	    load_element_position (res, curr_node);
	    load_element_size (res, curr_node);
        }
        else if (curr_node.name () == "widgets")
        {
            res->def_button = curr_node.get_prop_string ("default");
            
            for (StorageNode widget_node = curr_node.children (); widget_node; widget_node++)
                load_widget (res, widget_node);
        }
        else
            g_warning ("Error parsing `form' resource: "
                       "unexpected element `%s'",
                       curr_node.name ().c_str ());
    }
}

static void load_widget (Resources::Form *parent_form, StorageNode &node)
{
    std::string node_name = node.name ();
    std::string id = node.get_prop_string ("id");

    if (id == "")
    {
        g_warning ("`%s': Missing ID field", node_name.c_str ());
        return;
    }
    
    Widgets::Type type = Widgets::type_from_type_id (node.name ());
    
    if (type == Widgets::WIDGET_NONE)
    {
	g_warning ("Unknown widget type `%s'", node.name ().c_str ());
	return;
    }
    
    Widget *widget = parent_form->create_widget (type, id);
    
    switch (type)
    {
    case Widgets::WIDGET_LABEL:
	load_widget_label (static_cast <Widgets::Label*> (widget), node);
	break;
    case Widgets::WIDGET_BUTTON:
	load_widget_button (static_cast <Widgets::Button*> (widget), node);
	break;
    case Widgets::WIDGET_PUSHBUTTON:
	load_widget_pushbutton (static_cast <Widgets::PushButton*> (widget), node);
	break;
    case Widgets::WIDGET_GRAFFITI:
	load_widget_graffiti (static_cast <Widgets::Graffiti*> (widget), node);
	break;
    case Widgets::WIDGET_SELECTOR_TRIGGER:
	load_widget_selector_trigger (static_cast <Widgets::SelectorTrigger*> (widget), node);
	break;
    case Widgets::WIDGET_CHECKBOX:
	load_widget_checkbox (static_cast <Widgets::Checkbox*> (widget), node);
	break;
    case Widgets::WIDGET_LIST:
	load_widget_list (static_cast <Widgets::List*> (widget), node);
	break;
    case Widgets::WIDGET_POPUP_TRIGGER:
	load_widget_popup_trigger (static_cast <Widgets::PopupTrigger*> (widget), node);
	break;
    case Widgets::WIDGET_SCROLLBAR:
	load_widget_scrollbar (static_cast <Widgets::ScrollBar*> (widget), node);
	break;
    case Widgets::WIDGET_TEXT_FIELD:
	load_widget_text_field (static_cast <Widgets::TextField*> (widget), node);
	break;
    case Widgets::WIDGET_TABLE:
	load_widget_table (static_cast <Widgets::Table*> (widget), node);
	break;
    case Widgets::WIDGET_GADGET:
	load_widget_gadget (static_cast <Widgets::Gadget*> (widget), node);
	break;
    case Widgets::WIDGET_NONE:
	g_assert_not_reached ();
	break;
    }
}

static void load_widget_label (Widgets::Label *widget, StorageNode &node)
{
    StorageProcessor processor;
    processor.add_handler ("pos",    new WidgetPositionHandler (widget));
    processor.add_handler ("text",   new TextHandler (widget));
    processor.add_handler ("usable", new StorageNodeHandlers::Flag (widget->usable));

    processor.run (node);
}

static void load_widget_button (Widgets::Button *widget, StorageNode &node)
{
    /* Default values */
    widget->frame = Widgets::Button::FRAME_NONE;
    
    // Locally used node handlers
    HANDLER(FrameHandler, Property<Widgets::Button::FrameType>&, target,
    {
	if (node.get_prop_string ("bold") != "")
	    target = Widgets::Button::FRAME_BOLD;
	else
	    target = Widgets::Button::FRAME_SIMPLE;
    })
	    
    StorageProcessor processor;
    processor.add_handler ("pos",          new SizePosHandler (widget));
    processor.add_handler ("text",         new TextHandler (widget));
    processor.add_handler ("frame",        new FrameHandler (widget->frame));
    processor.add_handler ("usable",       new StorageNodeHandlers::Flag (widget->usable));
    processor.add_handler ("repeat",       new StorageNodeHandlers::Flag (widget->repeat));
    processor.add_handler ("anchor_right", new StorageNodeHandlers::Flag (widget->anchor_right));

    processor.run (node);
}

static void load_widget_pushbutton (Widgets::PushButton *widget, StorageNode &node)
{
    StorageProcessor processor;
    processor.add_handler ("pos",    new SizePosHandler (widget));
    processor.add_handler ("text",   new TextHandler (widget));
    processor.add_handler ("usable", new StorageNodeHandlers::Flag (widget->usable));
    processor.add_handler ("group",  new StorageNodeHandlers::IntProp ("id", widget->group));
    
    processor.run (node);
}

static void load_widget_graffiti (Widgets::Graffiti *widget, StorageNode &node)
{
    StorageProcessor processor;
    processor.add_handler ("pos", new WidgetPositionHandler (widget));

    processor.run (node);
}

static void load_widget_selector_trigger (Widgets::SelectorTrigger *widget,
					  StorageNode              &node)
{
    StorageProcessor processor;
    processor.add_handler ("pos",          new SizePosHandler (widget));
    processor.add_handler ("text",         new TextHandler (widget));
    processor.add_handler ("usable",       new StorageNodeHandlers::Flag (widget->usable));
    processor.add_handler ("anchor_right", new StorageNodeHandlers::Flag (widget->anchor_right));

    processor.run (node);
}

static void load_widget_checkbox (Widgets::Checkbox *widget, StorageNode &node)
{
    StorageProcessor processor;
    processor.add_handler ("pos",          new SizePosHandler (widget));
    processor.add_handler ("text",         new TextHandler (widget));
    processor.add_handler ("usable",       new StorageNodeHandlers::Flag (widget->usable));
    processor.add_handler ("anchor_right", new StorageNodeHandlers::Flag (widget->anchor_right));
    processor.add_handler ("group",        new StorageNodeHandlers::IntProp ("id", widget->group));
    processor.add_handler ("toggled",      new StorageNodeHandlers::Flag (widget->toggled));
    
    processor.run (node);
}

static void load_widget_list (Widgets::List *widget, StorageNode &node)
{
    class ItemsHandler: public StorageNodeHandlers::StringListContents
    {
	Widgets::List *widget;
    public:
	ItemsHandler (Widgets::List *widget_):
	    StorageNodeHandlers::StringListContents ("item", widget_->items),
	    widget (widget_)
	{}

	void operator() (const StorageNode &node)
	{
	    widget->font = node.get_prop_int ("font");
	    widget->visible_items = node.get_prop_int ("visible");
	    StorageNodeHandlers::StringListContents::operator() (node);
	}
    };

    StorageProcessor processor;
    processor.add_handler ("pos",    new SizePosHandler (widget));
    processor.add_handler ("usable", new StorageNodeHandlers::Flag (widget->usable));
    processor.add_handler ("items",  new ItemsHandler (widget));
    
    processor.run (node);
}

static void load_widget_scrollbar (Widgets::ScrollBar *widget, StorageNode &node)
{
    HANDLER(ValuesHandler, Widgets::ScrollBar *, widget,
    {
	widget->min_value = node.get_prop_int ("min");
	widget->max_value = node.get_prop_int ("max");
	widget->page_size = node.get_prop_int ("page_size");
	widget->value     = node.get_prop_int ("value");
    })
    
    StorageProcessor processor;
    processor.add_handler ("pos",    new SizePosHandler (widget));
    processor.add_handler ("usable", new StorageNodeHandlers::Flag (widget->usable));
    processor.add_handler ("values", new ValuesHandler (widget));

    processor.run (node);
}

static void load_widget_text_field (Widgets::TextField *widget, StorageNode &node)
{
    HANDLER(TextHandler, Widgets::TextField *, widget,
    {
	widget->font = node.get_prop_int ("font");
	widget->max_length = node.get_prop_int ("max_length");
    })
	
    /*
     * Evil hack: some files created by Guikachu 1.0 contain 'auto'
     * when width was 30
     */
    class PosHandler: public SizePosHandler
    {
    public:
	PosHandler (Widgets::TextField *widget):
	    SizePosHandler (widget)
	{}
	
    protected:
	void load_width (const StorageNode &node)
	{
	    std::string width_str = node.get_prop_string ("width");
	    
	    if (width_str == "auto")
		element->width = 30;
	    else
		element->width = atoi (width_str.c_str ());
	}
    };
    
    StorageProcessor processor;
    processor.add_handler ("pos",           new PosHandler (widget));
    processor.add_handler ("text",          new TextHandler (widget));
    processor.add_handler ("usable",        new StorageNodeHandlers::Flag (widget->usable));
    processor.add_handler ("editable",      new StorageNodeHandlers::Flag (widget->editable));
    processor.add_handler ("multi_line",    new StorageNodeHandlers::Flag (widget->multi_line));
    processor.add_handler ("auto_shift",    new StorageNodeHandlers::Flag (widget->auto_shift));
    processor.add_handler ("numeric",       new StorageNodeHandlers::Flag (widget->numeric));
    processor.add_handler ("underline",     new StorageNodeHandlers::Flag (widget->underline));
    processor.add_handler ("dynamic_size",  new StorageNodeHandlers::Flag (widget->dynamic_size));
    processor.add_handler ("justify_right", new StorageNodeHandlers::Flag (widget->justify_right));
    processor.add_handler ("has_scrollbar", new StorageNodeHandlers::Flag (widget->has_scrollbar));
    
    processor.run (node);
}

static void load_widget_table (Widgets::Table *widget, StorageNode &node)
{
    class TableDataHandler: public StorageNodeHandlers::HandlerBase
    {
	Widgets::Table *widget;
	std::vector<int> columns_tmp;
	
    public:
	TableDataHandler (Widgets::Table *widget_):
	    widget (widget_)
	{
	    // Reset to empty default values
	    widget->column_width = std::vector<int>(0);
	    widget->num_columns  = 0;
	}

	~TableDataHandler ()
	{
	    widget->column_width = columns_tmp;
	}

	void operator() (const StorageNode &node)
	{
	    class TableColumnHandler: public StorageNodeHandlers::HandlerBase
	    {
		std::vector<int> &columns;
	    public:
		TableColumnHandler (std::vector<int> &columns_) :
		    columns (columns_)
		{}
		
		void operator() (const StorageNode &node)
		{
		    columns.push_back (node.get_prop_int ("width"));
		}
	    };
	    
	    widget->num_rows = node.get_prop_int ("rows");
	    StorageProcessor columns_processor;
	    columns_processor.add_handler ("column", new TableColumnHandler (columns_tmp));
	};
    };

    StorageProcessor processor;
    processor.add_handler ("pos",        new WidgetPositionHandler (widget));
    processor.add_handler ("table_data", new TableDataHandler (widget));
    processor.run (node);
}

static void load_widget_popup_trigger (Widgets::PopupTrigger *widget,
				       StorageNode           &node)
{
    StorageProcessor processor;
    processor.add_handler ("pos",          new SizePosHandler (widget));
    processor.add_handler ("text",         new TextHandler (widget));
    processor.add_handler ("usable",       new StorageNodeHandlers::Flag (widget->usable));
    processor.add_handler ("anchor_right", new StorageNodeHandlers::Flag (widget->anchor_right));
    processor.add_handler ("popuplist",    new StorageNodeHandlers::StringProp ("id", widget->list_id));

    processor.run (node);
}
    
static void load_widget_gadget (Widgets::Gadget *widget, StorageNode &node)
{
    StorageProcessor processor;
    processor.add_handler ("pos",    new SizePosHandler (widget));
    processor.add_handler ("usable", new StorageNodeHandlers::Flag (widget->usable));

    processor.run (node);
}

} // namespace GuikachuLoader_funcs
} // namespace IO
} // namespace Guikachu
