//$Id: preferences.cc,v 1.7.2.2 2002/05/29 16:05:17 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "preferences.h"

#include <gconf/gconf-client.h>
#include <algorithm>

namespace Guikachu
{
    namespace Preferences
    {
	namespace Interface
	{
	    static void gconf_recent_files_cb (GConfClient  *client,
					       unsigned int  conn_id,
					       GConfEntry   *entry,
					       gpointer      user_data);
	}
	
	namespace FormEditor
	{
	    static void gconf_color_cb (GConfClient  *client,
					unsigned int  cnxn_id,
					GConfEntry   *entry,
					gpointer      user_data);	    
	}

	GConfClient *gconf_client;

	static std::string get_gconf_string (const std::string &key);
    }
}

#define GCONF_ROOT        "/apps/guikachu"
#define GCONF_COLORS_ROOT "/FormEditor/Colors"

using namespace Guikachu;

void Preferences::load ()
{
    // Set up GConf client
    gconf_client = gconf_client_get_default ();
    gconf_client_add_dir (gconf_client, "/apps/guikachu",
			  GCONF_CLIENT_PRELOAD_RECURSIVE, 0);
    
    // Set up GConf callbacks
    // User interface
    gconf_client_notify_add (gconf_client, GCONF_ROOT "/Interface",
			     Interface::gconf_recent_files_cb,
			     0, 0, 0);
    
    // Colors
    gconf_client_notify_add (gconf_client, GCONF_ROOT GCONF_COLORS_ROOT,
			     FormEditor::gconf_color_cb,
			     0, 0, 0);
}

void Preferences::save ()
{
    gtk_object_unref (GTK_OBJECT (gconf_client));
}

std::string Preferences::get_gconf_string (const std::string &key)
{
    std::string real_key = GCONF_ROOT + key;
    char *value = gconf_client_get_string (gconf_client, real_key.c_str (), 0);

    if (value)
	return value;
    else
	return "";
}



SigC::Signal0<void> Preferences::Interface::recent_files_changed;

static void Preferences::Interface::gconf_recent_files_cb (GConfClient  *client,
							   unsigned int  conn_id,
							   GConfEntry   *entry,
							   gpointer      user_data)
{
    recent_files_changed.emit ();
}

std::list<std::string> Preferences::Interface::get_recent_files ()
{
    GSList *gconf_list = gconf_client_get_list (gconf_client,
						GCONF_ROOT "/Interface/recent_files",
						GCONF_VALUE_STRING, 0);

    std::list<std::string> retval;
    
    for (GSList *i = gconf_list; i; i = g_slist_next (i))
    {
	retval.push_back ((char*)i->data);
    }

    return retval;
}

void Preferences::Interface::add_recent_file (const std::string &filename)
{
    std::list<std::string> recent_list = get_recent_files ();

    // Remove latest addition from list
    std::list<std::string>::iterator old_pos =
	std::find (recent_list.begin (), recent_list.end (), filename);
    if (old_pos != recent_list.end ())
	recent_list.erase (old_pos);

    // Prepend latest addition to list
    recent_list.push_front (filename);

    // Truncate list
    int recent_list_size = gconf_client_get_int (
	gconf_client, GCONF_ROOT "/Interface/recent_files_size", 0);
    int pos;
    std::list<std::string>::iterator truncate_begin;

    for (pos = 0, truncate_begin = recent_list.begin ();
	 pos < recent_list_size && truncate_begin != recent_list.end ();
	 pos++, truncate_begin++)
	;

    if (truncate_begin != recent_list.end ())
	recent_list.erase (truncate_begin, recent_list.end ());

    // Save new list in GConf
    GSList *c_recent_list = 0;
    GSList *c_i = c_recent_list;
    for (std::list<std::string>::const_iterator i = recent_list.begin ();
	 i != recent_list.end (); i++)
	if (!c_recent_list)
	    c_recent_list = c_i = g_slist_append(c_i, (void*)(i->c_str ()));
	else
	    c_i = g_slist_append(c_i, (void*)(i->c_str ()));

    gconf_client_set_list (gconf_client, GCONF_ROOT "/Interface/recent_files",
			   GCONF_VALUE_STRING, c_recent_list, 0);
    
    g_slist_free (c_recent_list);
}



SigC::Signal0<void> Preferences::FormEditor::colors_changed;

void Preferences::FormEditor::set_colors (const std::string &color_fg,
					  const std::string &color_disabled,
					  const std::string &color_bg,
					  const std::string &color_selection)
{
#define SET_COLOR(key,value) gconf_client_set_string (gconf_client, GCONF_ROOT "/FormEditor/Colors/" key, value.c_str (), 0)

    SET_COLOR("foreground", color_fg);
    SET_COLOR("foreground_disabled", color_disabled);
    SET_COLOR("background", color_bg);
    SET_COLOR("selection", color_selection);

#undef SET_COLOR
    
}

std::string Preferences::FormEditor::get_color_fg ()
{
    return get_gconf_string (GCONF_COLORS_ROOT "/foreground");
}

std::string Preferences::FormEditor::get_color_disabled ()
{
    return get_gconf_string (GCONF_COLORS_ROOT "/foreground_disabled");
}

std::string Preferences::FormEditor::get_color_bg ()
{
    return get_gconf_string (GCONF_COLORS_ROOT "/background");
}

std::string Preferences::FormEditor::get_color_selection ()
{
    return get_gconf_string (GCONF_COLORS_ROOT "/selection");
}

static void Preferences::FormEditor::gconf_color_cb (GConfClient  *client,
						     unsigned int  cnxn_id,
						     GConfEntry   *entry,
						     gpointer      user_data)
{
    colors_changed.emit ();
}

float Preferences::FormEditor::get_default_zoom ()
{
    return gconf_client_get_float (gconf_client, GCONF_ROOT "/FormEditor/default_zoom", 0);
}

void Preferences::FormEditor::set_default_zoom (float default_zoom)
{
    gconf_client_set_float (gconf_client,
			    GCONF_ROOT "/FormEditor/default_zoom",
			    default_zoom, 0);
}
