//$Id: mainwin-io.cc,v 1.19 2002/02/10 14:11:47 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"
#include <libgnome/libgnome.h>

#include "mainwin.h"

#include "resource-manager.h"
#include "session.h"
#include <gnome--/main.h>
#include <gnome--/messagebox.h>

using namespace Guikachu;
using std::string;

static string time_ago_string (time_t ago)
{
    string  ret_val;
    gchar  *buf = 0;
    
    int minutes = ago / 60;
    int hours = minutes / 60;
    int days = hours / 24;

    if (days > 1)
	buf = g_strdup_printf (_("If you don't save, your work from "
				 "the last %d days will be lost."),
			       days);
    else if (hours > 1)
	buf = g_strdup_printf (_("If you don't save, your work from "
				 "the last %d hours will be lost."),
			       hours);
    else if (minutes > 1)
	buf = g_strdup_printf (_("If you don't save, your work from "
				 "the last %d minutes will be lost."),
			       minutes);
    else
	buf = g_strdup_printf (_("If you don't save, your work from "
				 "the last minute will be lost."));

    ret_val = buf;
    g_free (buf);
    return ret_val;
}

bool GUI::MainWin::check_save ()
{
    if (!ResourceManager::is_dirty ())
    {
	if (in_exit)
	    Gnome::Main::quit ();
	return true;
    }
    
    int   response;
    int   time_since_save = time (0) - ResourceManager::get_modification_time ();
    char *filename_message;

    if (filename != "")
	filename_message = g_strdup_printf (
	    _("Save changes to document \"%s\"?"),
	    g_basename (filename.c_str ()));
    else
	filename_message = g_strdup (
	    _("Save changes to the current document?"));

    std::string message = filename_message;
    message += "\n\n" + time_ago_string (time_since_save);

    g_free (filename_message);

    Gnome::Dialog *dialog = new Gnome::MessageBox (
	message, GNOME_MESSAGE_BOX_WARNING, std::vector<string>());

    static const int save_btn = 2;
    static const int discard_btn = 0;
    static const int cancel_btn = 1;
    
    dialog->append_button (_("Don't Save"), GNOME_STOCK_PIXMAP_TRASH);
    dialog->append_button (GNOME_STOCK_BUTTON_CANCEL);
    dialog->append_button (_("Save"), GNOME_STOCK_PIXMAP_SAVE);

    dialog->set_modal (true);
    dialog->set_default (save_btn);
    gnome_dialog_grab_focus (dialog->gtkobj (), save_btn);
    
    response = dialog->run_and_close ();

    if (response == cancel_btn || response == -1) // Cancel
	return false;

    if (response == discard_btn)
	if (in_exit)
	    Gnome::Main::quit ();
    
    if (response == save_btn)
    {
	save_cb ();
	if (ResourceManager::is_dirty ())
	    return check_save ();
    }
	    
    return true;
}

void GUI::MainWin::new_cb ()
{
    // Check if file has changed
    if (!check_save ())
	return;

    ResourceManager::reset ();
    filename = "";
    Guikachu::Main::instance ()->set_filename (filename);

    update_title ();
}

void GUI::MainWin::set_filename (const string& filename_)
{
    filename = filename_;
    update_title ();
}

void GUI::MainWin::filesel_close_cb (Gtk::FileSelection *fs)
{
    delete fs;
    Gtk::Main::quit ();
}

void GUI::MainWin::save_cb ()
{
    if (filename == "")
	save_as_cb ();
    else
    {
	ResourceManager::save (filename);
	if (in_exit)
	    Gnome::Main::quit ();
    }
}

void GUI::MainWin::save_as_cb ()
{
    Gtk::FileSelection *fs = new Gtk::FileSelection ();

    fs->show_fileop_buttons ();
    fs->set_title (_("Save Guikachu file"));

    fs->get_ok_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_save_cb), fs));
    fs->get_cancel_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_close_cb), fs));

    fs->set_transient_for (*this);
    fs->set_modal (true);
    fs->show ();
    Gtk::Main::run ();
}

void GUI::MainWin::load_cb ()
{
    // Check if file has changed
    if (!check_save ())
	return;
    
    // Set up a file dialog and show it    
    Gtk::FileSelection *fs = new Gtk::FileSelection ();

    fs->hide_fileop_buttons ();
    fs->set_title (_("Select Guikachu file to open"));

    fs->get_ok_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_open_cb), fs));
    fs->get_cancel_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_close_cb), fs));

    fs->set_transient_for (*this);
    fs->set_modal (true);
    fs->show ();
    Gtk::Main::run ();
}

void GUI::MainWin::export_cb ()
{
    Gtk::FileSelection *fs = new Gtk::FileSelection ();

    fs->show_fileop_buttons ();
    fs->set_title (_("Export to RCP"));
    
    fs->get_ok_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_export_cb), fs));
    fs->get_cancel_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_close_cb), fs));

    fs->set_transient_for (*this);
    fs->set_modal (true);
    fs->show ();
    Gtk::Main::run ();
}


/*********************************
 * File I/O dialog callbacks
 *********************************/
void GUI::MainWin::filesel_save_cb (Gtk::FileSelection *fs)
{
    filename = fs->get_filename ();
    filesel_close_cb (fs);
    Guikachu::Main::instance ()->set_filename (filename);

    update_title ();
    save_cb ();

    if (in_exit)
	Gnome::Main::quit ();
}

void GUI::MainWin::filesel_open_cb (Gtk::FileSelection *fs)
{
    string new_filename = fs->get_filename ();
    filesel_close_cb (fs);

    ResourceManager::IOResult result = ResourceManager::load (new_filename);

    gchar *buffer;
    switch (result)
    {
    case ResourceManager::IO_OK:
	filename = new_filename;
	Guikachu::Main::instance ()->set_filename (filename);
	update_title ();
	break;
    case ResourceManager::IO_NOTFOUND:
	buffer = g_strdup_printf (_("Unable to open `%s'"),
				  g_basename (new_filename.c_str ()));
	GUI::show_error (buffer);
	g_free (buffer);
	break;
    case ResourceManager::IO_INVALID:
	buffer = g_strdup_printf (_("Unable to open `%s': "
				    "Not a valid Guikachu document"),
				  g_basename (new_filename.c_str ()));
	GUI::show_error (buffer);
	g_free (buffer);
	break;
    }
}

void GUI::MainWin::filesel_export_cb (Gtk::FileSelection *fs)
{
    string filename = fs->get_filename ();
    filesel_close_cb (fs);
    
    ResourceManager::export_rcp (filename);
}

int GUI::MainWin::delete_event_impl (GdkEventAny *e)
{
    in_exit = true;

    if (!check_save ())
	in_exit = false;

    return true;
}
