//$Id: resource-manager-io.cc,v 1.29 2002/05/05 18:49:53 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "resource-manager.h"
#include "resource-util.h"

#include "config.h"
#include <libgnome/libgnome.h>

/* Having to include these for just open()/close() and *time() sure sucks */
#include <fstream>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "mainwin.h"

#include "target.h"
#include "storage.h"

using namespace Guikachu;

void ResourceManager::reset ()
{
    resource_map_t resource_map_copy (resource_map);
    
    for (resource_map_t::iterator i = resource_map.begin ();
	 i != resource_map.end ();)
	remove_resource ((i++)->second);
    
    if (!app)
    {
	app = new Resources::Application ();
	app->changed.connect (SigC::slot(&ResourceManager::set_dirty));
    }
    else
	app->reset ();

    if (!target)
    {
	target = new Target ();
	target->changed.connect (SigC::slot(&ResourceManager::set_dirty));
    }
    target->load_stock (Target::get_default_stock_id ());

    clear_dirty ();
}

ResourceManager::IOResult ResourceManager::load (const std::string &filename)
{
    Storage storage;

    int fd = open (filename.c_str (), O_RDONLY);
    if (fd == -1)
	return IO_NOTFOUND;
    close (fd);
	
    
    if (!storage.load (filename))
	return IO_INVALID;

    // Reset flags
    encoding_fixed = false;
    
    StorageNode root_node = storage.get_root ();
    if (root_node.name () != "guikachu")
        return IO_INVALID;

    reset ();

    // Walk the tree
    for (StorageNode curr_node = root_node.children ();
         curr_node; curr_node++)
    {
	if (curr_node.name () == "application")
	    app->load (curr_node);
	
	else if (curr_node.name () == "target")
	    target->load (curr_node);
	
	else
	    load_resource (curr_node);
    }

    clear_dirty ();

    if (encoding_fixed)
    {
	encoding_warning ();
	set_dirty ();
    }
    
    return IO_OK;
}

Resource* ResourceManager::load_resource (StorageNode &node)
{
    Resources::Type res_type;

    std::string node_name = node.name ();
    
    if (node_name == "dialog")
	res_type = Resources::RESOURCE_DIALOG;
    else if (node_name == "form")
	res_type = Resources::RESOURCE_FORM;
    else if (node_name == "menu")
	res_type = Resources::RESOURCE_MENU;
    else if (node_name == "string")
	res_type = Resources::RESOURCE_STRING;
    else if (node_name == "stringlist")
	res_type = Resources::RESOURCE_STRINGLIST;
    else if (node_name == "blob")
	res_type = Resources::RESOURCE_BLOB;
    else
    {
	g_warning ("Unknown resource type `%s'", node_name.c_str ());
	return 0;
    }

    std::string id = node.get_prop<std::string> ("id");

    if (id == "")
    {
	g_warning ("`%s': Missing ID field", node_name.c_str ());
	return 0;
    }
    
    Resource *resource = create_resource (res_type, id);
    resource->load (node);
    
    return resource;
}

ResourceManager::IOResult ResourceManager::save (const std::string &filename)
{
    Storage     storage;
    StorageNode root = storage.create_root ("guikachu");
    StorageNode node;

    // Save per-application data
    node = root.add_node ("application");
    app->save (node);

    // Save target machine data
    node = root.add_node ("target");
    target->save (node);
    
    // Save resources
    for (std::map<std::string, Resource*>::iterator i = resource_map.begin ();
	 i != resource_map.end ();
	 i++)
    {
	Resource *resource = i->second;
	std::string curr_node_name = Resources::type_id_from_type (resource->get_type ());

	node = root.add_node (curr_node_name);
	node.set_prop ("id", resource->id);
	resource->save (node);
    }

    storage.save (filename);

    clear_dirty ();

    return IO_OK;
}

void ResourceManager::export_rcp (const std::string &filename)
{
    std::ofstream fout (filename.c_str ());

    time_t curr_date = time (0);
    const char* date_str = ctime (&curr_date);
    
    fout << "/*" << std::endl;
    fout << " * This file was created by Guikachu " << VERSION
	 << ". Do not edit it manually." << std::endl;
    fout << " * Creation date: " << date_str; // ctime ()'s return value ends in \n
    fout << " */" << std::endl << std::endl;;

    // Export blobs before anything else, to allow #includes and whatnot
    for (std::map<std::string, Resource*>::iterator i = resource_map.begin ();
	 i != resource_map.end (); i++)
    {
	if (i->second->get_type () == Resources::RESOURCE_BLOB)
	    fout << i->second;
    }

    // Export per-app data
    fout << app;

    // Export the rest of the resources
    for (std::map<std::string, Resource*>::iterator i = resource_map.begin ();
	 i != resource_map.end (); i++)
    {
	if (i->second->get_type () != Resources::RESOURCE_BLOB)
	    fout << i->second;
    }
    
    fout.close ();
}

void ResourceManager::encoding_fixed_cb ()
{
    encoding_fixed = true;
}

void ResourceManager::encoding_warning ()
{
    GUI::show_warning (_("Previous releases of Guikachu always stored\n"
			 "strings in the local encoding, while PalmOS\n"
			 "uses the Latin-1 character  set. The encoding\n"
			 "of old Guikachu files can not be determined,\n"
			 "so all non-ASCII characters have been\n"
			 "automatically converted to escape sequences.\n"
			 "\n"
			 "Please review the results of the automatic\n"
			 "conversion after the file had been loaded."));
}

