//$Id: target.cc,v 1.5 2002/01/08 23:27:07 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "target.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include "storage.h"
#include <glib.h> // For g_warning

#include <sys/types.h>
#include <dirent.h>

#include <sigc++/signal_system.h>

using namespace Guikachu;

// Cache of available stock targets
namespace
{
    struct StockTargetData
    {
	std::string description;
	int screen_width;
	int screen_height;

	StockTargetData (const std::string &description_,
			 int                screen_width_,
			 int                screen_height_):
	    description (description_),
	    screen_width (screen_width_),
	    screen_height (screen_height_)
	    {}
    };

    typedef std::map<std::string, StockTargetData> StockTargetCache;
    static StockTargetCache stock_target_cache;
    static const StockTargetData& get_stock_data (const std::string &stock_id);
    static StockTargetData load_stock_data (const std::string &stock_id);
}


Target::Target ():
    screen_width (changed),
    screen_height (changed)
{
    load_stock (get_default_stock_id ());

    changed.connect (SigC::slot (this, &Target::changed_cb));
}

void Target::changed_cb ()
{
    // Invalidate stock ID
    stock_id = "";
}

std::string Target::get_default_stock_id ()
{
    return "palm";
}

void Target::load_stock (const std::string &stock_id_)
{
    g_return_if_fail (stock_id_ != "");
    
    if (stock_id_ == stock_id)
	return;
    
    const StockTargetData &stock_data = get_stock_data (stock_id_);

    screen_width  = stock_data.screen_width;
    screen_height = stock_data.screen_height;

    stock_id = stock_id_;
    changed ();
}

std::string Target::get_stock_id () const
{
    return stock_id;
}

std::map<std::string, std::string> Target::stock_targets ()
{
    std::map<std::string, std::string> ret_val;

    DIR    *target_def_dir = opendir (GUIKACHU_TARGETDIR);
    g_return_val_if_fail (target_def_dir != 0, ret_val);

    dirent       *target_def_dirent;
    std::string   filename;
    std::string   id;

    while ((target_def_dirent = readdir (target_def_dir)) != 0)
    {
	filename = target_def_dirent->d_name;
	
	// Get file extension...
	std::string::size_type i = filename.rfind (".guikachu-target");
	if (i == (filename.length () - strlen (".guikachu-target")))
	{
	    // ... then chop it off
	    id = std::string (filename, 0, i);

	    // Load the target definition into cache, and return its
	    // ID and description
	    ret_val[id] = get_stock_data (id).description;
	}
    }
    
    closedir (target_def_dir);
    
    return ret_val;
}

namespace {
const StockTargetData& get_stock_data (const std::string &stock_id)
{
    // Look up cache first
    StockTargetCache::const_iterator cache_hit = stock_target_cache.find (stock_id);
    if (cache_hit != stock_target_cache.end ())
	return cache_hit->second;

    // If not found in cache, load...
    const StockTargetData &entry = load_stock_data (stock_id);

    // ... and store
    std::pair<StockTargetCache::const_iterator, bool> cache_store;
    cache_store = stock_target_cache.insert (std::make_pair (stock_id, entry));
    cache_hit = cache_store.first;
    
    return cache_hit->second;
}

StockTargetData load_stock_data (const std::string &stock_id)
{
    std::string description = stock_id;
    int screen_width = 160;
    int screen_height = 160;
    
    std::string filename = GUIKACHU_TARGETDIR;
    filename += "/" + stock_id + ".guikachu-target";

    Storage target_def;
    target_def.load (filename);
    StorageNode root_node = target_def.get_root ();
    if (root_node.name () != "guikachu-target")
    {
	g_warning ("`%s' is not a valid Guikachu target definition",
		   filename.c_str ());
	return StockTargetData (description, screen_width, screen_height);
    }

    // Load translated entries first
    description = root_node.get_translated_child ("description").get_content ();
    
    for (StorageNode curr_node = root_node.children ();
	 curr_node; curr_node++)
    {
	if (curr_node.name () == "description")
	    // Already loaded -- mentioned here only to surpass waning
	    ;
	else if (curr_node.name () == "screen")
	{
	    screen_width  = curr_node.get_prop<int> ("width");
	    screen_height = curr_node.get_prop<int> ("height");
	} else {
	    g_warning ("Error parsing target definition: "
		       "unexpected element `%s'",
		       curr_node.name ().c_str ());
	}
    }

    return StockTargetData (description, screen_width, screen_height);
}

} // Anonymous namespace


void Target::save (StorageNode &node) const
{
    if (stock_id != "")
    {
	node.set_prop ("stock_id", stock_id);
	return;
    }
    
    StorageNode screen_node = node.add_node ("screen");
    screen_node.set_prop ("width",  screen_width);
    screen_node.set_prop ("height", screen_height);
}

bool Target::load (StorageNode &node)
{
    // Is this a stock target?
    std::string stock_id = node.get_prop<std::string> ("stock_id");
    if (stock_id != "")
    {
	load_stock (stock_id);
	return true;
    }
    
    for (StorageNode curr_node = node.children (); curr_node; curr_node++)
    {
	if (curr_node.name () == "screen")
	{
	    screen_width  = curr_node.get_prop<int> ("width");
	    screen_height = curr_node.get_prop<int> ("height");
	} else {
	    g_warning ("Error parsing custom target definition: "
		       "unexpected element `%s'",
		       curr_node.name ().c_str ());
	}
    }

    return true;
}
