//$Id: button.cc,v 1.31 2002/01/28 18:21:58 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "button.h"
#include "button-canvas.h"

#include "form-editor.h"

using Guikachu::Widgets::Button;
using namespace Guikachu::GUI::FormEditor;

Button::Button (Resources::Form   *owner,
		const std::string &id):
    Widget (owner, id),
    Resizeable (changed),
    AutoWidth (changed),
    AutoHeight (changed),
    Textual (changed, id),
    
    editor (0),
    frame (changed, BUTTON_FRAME_SIMPLE),
    usable (changed, true),
    repeat (changed, false),
    anchor_right (changed, false)
{
}

Button::~Button ()
{
    if (editor)
	delete editor;
}

Gtk::Widget *Button::create_editor ()
{
    if (!editor)
	editor = new ButtonProperties (this);

    return editor->get_editor ();
}

WidgetCanvasItem* Button::create_canvas_item (Gnome::CanvasGroup *parent_group)
{
    return new ButtonCanvasItem (this, parent_group);
}


bool Button::load (StorageNode &node)
{
    /* Non-NULL default values */
    usable = false;
    frame = BUTTON_FRAME_NONE;
    
    for (StorageNode curr_node  = node.children ();
	 curr_node; curr_node++)
    {
	if (curr_node.name () == "pos")
	{
	    x = curr_node.get_prop<int> ("x");
	    y = curr_node.get_prop<int> ("y");

	    std::string width_str = curr_node.get_prop<std::string> ("width");
	    if (width_str == "auto")
	    {
		manual_width = false;
		width = 0;
	    } else {
		manual_width = true;
		width = atoi (width_str.c_str ());
	    }

	    std::string height_str = curr_node.get_prop<std::string> ("height");
	    if (height_str == "auto")
	    {
		manual_height = false;
		height = 0;
	    } else {
		manual_height = true;
		height = atoi (height_str.c_str ());
	    }
	}
	else if (curr_node.name () == "text")
	{
	    text = curr_node.get_content ();
	    font = curr_node.get_prop <int> ("font");
	}
	else if (curr_node.name () == "frame")
	{
	    if (curr_node.get_prop<std::string> ("bold") != "")
		frame = BUTTON_FRAME_BOLD;
	    else
		frame = BUTTON_FRAME_SIMPLE;
	}
	else if (curr_node.name () == "usable")
	    usable = true;
	else if (curr_node.name () == "repeat")
	    repeat = true;
	else if (curr_node.name () == "anchor_right")
	    anchor_right = true;
	else if (curr_node.name () == "usable")
	    usable = true;
	else
	    g_warning ("Error parsing `button' widget: "
		       "unexpected element `%s'",
		       curr_node.name ().c_str ());
    }

    changed ();

    return true;
}

void Button::save (StorageNode &node) const
{
    StorageNode my_node;
    
    /* Position */
    my_node = node.add_node ("pos");
    my_node.set_prop ("x", x);
    my_node.set_prop ("y", y);

    /* Size */
    if (manual_width)
	my_node.set_prop ("width", width);
    else
	my_node.set_prop ("width", "auto");

    if (manual_height)
	my_node.set_prop ("height", height);
    else
	my_node.set_prop ("height", "auto");
    
    /* Frame */
    if (frame != BUTTON_FRAME_NONE)
    {
	my_node = node.add_node ("frame");
	if (frame == BUTTON_FRAME_BOLD)
	    my_node.set_prop ("bold", "bold");
    }
    
    /* Usable */
    if (usable)
	node.add_node ("usable");

    /* Repeating */
    if (repeat)
	node.add_node ("repeat");

    /* Anchor_right */
    if (anchor_right)
	node.add_node ("anchor_right");
    
    /* Label and font */
    my_node = node.add_node ("text", text);
    my_node.set_prop ("font", font);
}

void Button::save_rcp (std::ostream &ostr) const
{
    std::string frame_str, anchor_str;
    gchar *width_str, *height_str;
    std::string type_str;

    if (!repeat)
	type_str = "BUTTON";
    else
	type_str = "REPEATBUTTON";
    
    if (manual_width)
	width_str = g_strdup_printf ("%d", int (width));
    else
	width_str = g_strdup ("AUTO");
    
    if (manual_height)
	height_str = g_strdup_printf ("%d", int (height));
    else
	height_str = g_strdup ("AUTO");
    
    switch (frame)
    {
    case BUTTON_FRAME_NONE:
	frame_str = "NOFRAME";
	break;
    case BUTTON_FRAME_SIMPLE:
	frame_str = "FRAME";
	break;
    case BUTTON_FRAME_BOLD:
	frame_str = "BOLDFRAME";
	break;
    }

    if (anchor_right)
	anchor_str = "RIGHTANCHOR";
    else
	anchor_str = "LEFTANCHOR";
    
    ostr << "  " << type_str << " \"" << text << "\" ID " << id;
    ostr << " AT (" << x << " " << y << " " << width_str << " " << height_str << ")";
    ostr << " FONT " << font;
    ostr << " " << frame_str;
    ostr << " " << anchor_str;
    if (usable)
	ostr << " USABLE";
    else
	ostr << " NONUSABLE";
    
    ostr << std::endl;

    g_free (height_str);
    g_free (width_str);
}

int Button::get_auto_width () const
{
    int text_width = get_string_width (font, text);
    return 6 + text_width + 6;
}

int Button::get_auto_height () const
{
    int text_height = get_font_height (font);

    if (frame == BUTTON_FRAME_SIMPLE)
	text_height += 2;
    if (frame == BUTTON_FRAME_BOLD)
	text_height += 1;
	
    return text_height;
}
