//$Id: resource-manager.cc,v 1.29 2002/01/10 17:06:51 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "resource-manager.h"

#include "mainwin.h"
#include "resource-util.h"

#include "dialog-res.h"
#include "form-res.h"
#include "menu-res.h"
#include "string-res.h"
#include "stringlist-res.h"
#include "blob-res.h"

#include <fstream>
#include <gnome-xml/parser.h>

using namespace Guikachu;

#define ASCII_TOUPPER(c)  (((c) >= 'a' && (c) <= 'z') ? (c) - 32 : (c))

int NoCase::compare (const std::string &x, const std::string &y) const
{
    // We can't use strncmp() as that would be locale-dependant
    std::string::const_iterator p = x.begin ();
    std::string::const_iterator q = y.begin ();

    while (p != x.end () && q != y.end () &&
	   ASCII_TOUPPER (*p) == ASCII_TOUPPER (*q))
    {
	p++;
	q++;
    }

    // Return value:  0 if x == y
    //               +1 if x > y
    //               -1 if x < y
    if (p == x.end () && q == y.end ()) return 0;
    if (p == x.end ()) return -1;
    if (q == y.end ()) return 1;

    return ASCII_TOUPPER (*p) < ASCII_TOUPPER (*q) ? -1 : 1;
}

bool NoCase::operator () (const std::string &x, const std::string &y) const
{
    // Return value: true  if x <  y
    //               false if x >= y
    return (compare (x, y) < 0);
}

ResourceManager::resource_map_t  ResourceManager::resource_map;
ResourceManager::id_set_t        ResourceManager::id_set;
Resources::Application          *ResourceManager::app = 0;
Target                          *ResourceManager::target = 0;
bool                             ResourceManager::dirty = false;
time_t                           ResourceManager::modification_time = 0;

SigC::Signal1<void, Resource*> ResourceManager::resource_created;
SigC::Signal1<void, Resource*> ResourceManager::resource_removed;

std::string ResourceManager::create_id (const std::string &prefix)
{
    unsigned int num = 0;
    std::string ret_val;
    gchar *curr_name;
    const char *prefix_str = prefix.c_str ();
    
    for (num = 1; num != 0; num++)
    {
	curr_name = g_strdup_printf ("%s%d", prefix_str, num);

	if (id_set.find (curr_name) == id_set.end ())
	{
	    ret_val = curr_name;
	    g_free (curr_name);
	    break;
	} else {
	    g_free (curr_name);
	}
    }

    if (!num)
	return "";

    return ret_val;
}

bool ResourceManager::register_id (const std::string &id)
{
    if (id_set.find (id) != id_set.end ())
    {
	char *error_msg = g_strdup_printf (_("Non-unique identifier `%s'"), id.c_str ());
	GUI::show_error (error_msg);
	g_free (error_msg);
	return false;
    }

    id_set.insert (id);
    return true;
}

void ResourceManager::unregister_id (const std::string &id)
{
    id_set.erase (id);
}

std::string ResourceManager::validate_id (std::string id)
{
    std::string allowed_characters =
	"abcdefghijklmnopqrstuvwxyz_ABCDEFGHIJKLMNOPQRSTUVWXYZ_0123456789";
    
    for (std::string::iterator i = id.begin ();
	 i != id.end (); i++)
	if (allowed_characters.find (*i) == std::string::npos)
	    *i = '_';

    return id;
}

Resource* ResourceManager::create_resource (Resources::Type type,
					    std::string     id = "")
{
    id = validate_id (id);
    
    if (id == "")
	id = create_id (Resources::prefix_from_type (type));
    
    if (!register_id (id))
	return 0;

    std::string real_id = id;
    for (std::string::iterator i = real_id.begin ();
	 i != real_id.end (); i++)
    {
	if (*i == ' ')
	    *i = '_';
    }
    
    Resource *res;

    switch (type)
    {
    case Resources::RESOURCE_DIALOG:
	res = new Resources::Dialog (real_id);
	break;
    case Resources::RESOURCE_STRING:
	res = new Resources::String (real_id);
	break;
    case Resources::RESOURCE_STRINGLIST:
	res = new Resources::StringList (real_id);
	break;
    case Resources::RESOURCE_BLOB:
	res = new Resources::Blob (real_id);
	break;
    case Resources::RESOURCE_MENU:
	res = new Resources::Menu (real_id);
	break;
    case Resources::RESOURCE_FORM:
	Resources::Form *form = new Resources::Form (real_id);
	form->widget_changed.connect (SigC::slot (&ResourceManager::set_dirty));
	res = form;
	break;
    }
    
    resource_map.insert (std::make_pair (real_id, res));

    res->changed.connect (SigC::slot (&ResourceManager::set_dirty));

    set_dirty ();
    
    resource_created.emit (res);
    
    return res;
}

bool ResourceManager::rename_resource (const std::string &old_id,
				       const std::string &new_id)
{
    // No-ops: old_id == new_id
    if (old_id == new_id)
	return true;

    // No-ops: new_id and old_id differ only in case
    if (NoCase ().compare (old_id, new_id) == 0)
	return true;

    // Error: empty id
    if (new_id == "")
	return false;

    // Error: old_id does not exist
    if (id_set.find (old_id) == id_set.end ())
	return false;

    if (id_set.find (new_id) != id_set.end ())
    {
	char *error_msg = g_strdup_printf (_("Non-unique identifier `%s'"), new_id.c_str ());
	GUI::show_error (error_msg);
	g_free (error_msg);

	return false;
    }
    
    id_set.erase (old_id);
    id_set.insert (new_id);

    resource_map_t::iterator map_i = resource_map.find (old_id);
    
    if (map_i != resource_map.end ())
    {
	Resource *res = map_i->second;
	resource_map.erase (map_i);
	resource_map.insert (std::make_pair (new_id, res));
    }

    return true;
}

void ResourceManager::remove_resource (Resource* resource)
{
    resource_removed.emit (resource);

    unregister_id (resource->id);
    resource_map.erase (resource->id ());
    
    delete resource;

    set_dirty ();
}

std::set<Resource*> ResourceManager::get_resources ()
{
    std::set<Resource*> ret;

    for (resource_map_t::const_iterator i = resource_map.begin ();
	 i != resource_map.end (); i++)
    {
	ret.insert (i->second);
    }

    return ret;
}

Resource* ResourceManager::get_resource (const std::string &id)
{
    resource_map_t::const_iterator found = resource_map.find (id);

    if (found == resource_map.end ())
	return 0;
    else
	return found->second;
}

void ResourceManager::set_dirty ()
{
    if (!modification_time)
	modification_time = time (0);
    dirty = true;
}

void ResourceManager::clear_dirty ()
{
    modification_time = 0;
    dirty = false;
}

bool ResourceManager::is_dirty ()
{
    return dirty;
}

time_t ResourceManager::get_modification_time ()
{
    return modification_time;
}
