//$Id: font.cc,v 1.9 2002/01/01 21:23:40 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "font.h"

#include <strstream>
#include <fstream>
#include <stdio.h> /* For sscanf */
#include <list>

using namespace Guikachu::GUI::FormEditor;

Font::Font (const std::string &filename):
    height (0)
{
    parse (filename);
}

Font::Font ():
    height (0)
{
}

void Font::parse (const std::string &filename)
{
    std::ifstream stream ((std::string (GUIKACHU_FONTDIR) + "/" + filename).c_str ());

    parse (stream);
}

void Font::parse (std::istream &stream)
{
    /* FIXME: Error handling, robustness issues regarding garbage
     * input */
    
    std::string curr_line;
    bool in_header = true;
    
    while (in_header)
    {
	getline (stream, curr_line);

	if (curr_line == "")
	    continue;
	
	std::strstream curr_line_stream;
	std::string    param_name;
	int            param_val;
	
	curr_line_stream << curr_line;
	curr_line_stream >> param_name;
	curr_line_stream >> param_val;

	if (param_name == "ascent")
	    ascent = param_val;

	else if (param_name == "descent")
	    descent = param_val;
	
	else if (param_name == "GLYPH")
	{
	    in_header = false;
	    continue;
	}
    }

    while (!stream.eof ())
    {
	std::strstream curr_line_stream;
	std::string    param_name;
	
	curr_line_stream << curr_line;
	curr_line_stream >> param_name;

	if (param_name == "GLYPH")
	{
	    std::string glyph_index;
	    char        glyph_char;
	    
	    curr_line_stream >> glyph_index;

	    /* Un-escape char index */
            if (glyph_index[0] == '\'')
            {
                sscanf (glyph_index.c_str (), "'%c'", &glyph_char);
            } else {
                glyph_char = atoi (glyph_index.c_str ());
            }

	    /* Parse current glyph */
	    Glyph curr_glyph (stream, height);

	    glyphs.insert (std::pair <char, Glyph> (glyph_char, curr_glyph));
	}
	
	getline (stream, curr_line);
    }
}

static void free_pixdata_delete (guchar *pixels, gpointer data)
{
    delete pixels;
}


GdkPixbuf* Font::render (const std::string &text,
			 const Gdk_Color   &color) const
{
    if (text == "")
	return 0;
    
    std::list <Glyph> typeset;
    int width = 0;
    
    for (unsigned int i = 0; i < text.length (); i++)
    {
	std::hash_map<char, Glyph>::const_iterator curr_glyph_i = glyphs.find (text[i]);
	
	if (curr_glyph_i != glyphs.end ())
        {
            const Glyph& curr_glyph = curr_glyph_i->second;
            typeset.push_back (curr_glyph);

	    width += curr_glyph.width;
        }
    }

    guchar *pixbuf_data = new guchar[width * height * 4];
    int x_position = 0;

    for (std::list<Glyph>::const_iterator i = typeset.begin ();
         i != typeset.end (); i++)
    {
        const Glyph& glyph = *i;

        unsigned int curr_width = glyph.width;
        bool **curr_bitmap = glyph.bitmap;
        
        for (unsigned int y = 0; y < height; y++)
        {
            for (unsigned int x = 0; x < curr_width; x++)
            {
                if (curr_bitmap[y][x])
                {
                    pixbuf_data[((y * width) + x_position + x) * 4 + 0] = color.red >> 8;
                    pixbuf_data[((y * width) + x_position + x) * 4 + 1] = color.green >> 8;
                    pixbuf_data[((y * width) + x_position + x) * 4 + 2] = color.blue >> 8;
                    pixbuf_data[((y * width) + x_position + x) * 4 + 3] = 255;
                } else {
                    pixbuf_data[((y * width) + x_position + x) * 4 + 3] = 0;
                }
            }
	}

	x_position += curr_width;

    }
    
    return gdk_pixbuf_new_from_data (pixbuf_data,
				     GDK_COLORSPACE_RGB,
				     true,
				     8,
				     width, height,
				     width * 4,
				     free_pixdata_delete, 0);
}

unsigned int Font::string_width (const std::string &text) const
{
    if (text == "")
	return 0;
    
    int width = 0;
    
    for (unsigned int i = 0; i < text.length (); i++)
    {
	std::hash_map<char, Glyph>::const_iterator curr_glyph_i = glyphs.find (text[i]);
	
	if (curr_glyph_i != glyphs.end ())
	    width += curr_glyph_i->second.width;
    }

    return width;
}
