//$Id: form.cc,v 1.33 2001/12/21 17:20:18 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "form.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include <gnome--/canvas-group.h>
#include <gnome--/canvas-polygon.h>
#include <gnome--/canvas-rect.h>
#include <gdk-pixbuf/gnome-canvas-pixbuf.h>
#include <sigc++/retbind.h>
#include <gnome--/app-helper.h>
#include <libgnomeui/gnome-stock.h>

#include <list>
#include <math.h>

#include "form-editor.h"

using namespace Guikachu::GUI::FormEditor;

#define GUIKACHU_WINDOW_TITLE_HEIGHT 13

Form::Form (Resources::Form *res_,
	    GUI::FormWindow *win_):
    res (res_),
    win (win_),
    editor (res),
    group (0),
    background_item (0),
    frame_group (0),
    title_group (0),
    widget_group (0)
{
    drag_context.dragging = false;
    drag_context.drag_occured = false;
    
    using namespace SigC;

    group = new Gnome::CanvasGroup (*(win->get_canvas ()->root ()), 0, 0);
    group->event.connect (slot (this, &Form::canvas_event_cb));

    widget_group = new Gnome::CanvasGroup (*group, 0, 0);

    // Popup menu
    std::list <Gnome::UI::Info> popup_menu_list;
    popup_menu_list.push_back (
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_MENU_PROP),
			 _("_Edit"), clicked.slot ()));
    Gnome::UI::fill (popup_menu, popup_menu_list, *(popup_menu.get_accel_group ()));

    res->changed.connect (slot (this, &Form::update));
    update ();
}

Form::~Form ()
{
}

Gnome::CanvasGroup * Form::get_canvas () const
{
    return widget_group;
}

void Form::update ()
{
    draw_background ();
    draw_frame ();
    draw_title ();

    group->set_x (0);
    group->set_y (0);
    group->move (res->x, res->y);

    widget_group->raise_to_top ();
}

Gtk::Widget* Form::get_editor ()
{
    return editor.get_editor ();
}

void Form::draw_background ()
{
    Gnome::CanvasGroup *root = win->get_canvas ()->root ();

    using namespace Gnome::CanvasHelpers;
    delete background_item;
    background_item = new Gnome::CanvasRect (*root,
					     0, 0,
					     GUIKACHU_SCREEN_WIDTH - 1,
					     GUIKACHU_SCREEN_HEIGHT - 1);
    *background_item
	<< width_pixels (0)
	<< fill_color (get_background_color ());
    background_item->lower_to_bottom ();
}

void Form::draw_frame ()
{
    delete frame_group;
    frame_group = new Gnome::CanvasGroup (*group, 0, 0);
    
    Gnome::CanvasRect *frame_rect =
	new Gnome::CanvasRect (*frame_group,
			       -1, -1,
			       res->width, res->height);
    frame_rect->set_width_units (1);
    frame_rect->set_fill_color ("");
    
    if (res->modal)
	draw_frame_modal ();
    else
	if (res->frame)
	    draw_frame_nonmodal ();
}

void Form::draw_frame_nonmodal ()
{
    Gnome::CanvasRect *frame_rect =
	new Gnome::CanvasRect (*frame_group,
			       -1, -1,
			       res->width, res->height);
    
    frame_rect->set_width_units (1);
    frame_rect->set_outline_color (get_foreground_color (true));
    frame_rect->set_fill_color ("");
}

static void free_pixdata_delete (guchar *pixels, gpointer closure)
{
    delete pixels;
}

static GdkPixbuf *draw_solid (int width, int height)
{
    static Gdk_Color color (get_foreground_color (true));
    static guchar red   = color.red >> 8;
    static guchar green = color.green >> 8;
    static guchar blue  = color.blue >> 8;
    
    int num_pixels = width * height;
    guchar *pixbuf_data = new guchar[width * height * 4];
    
    for (int offset = 0; offset < num_pixels * 4; offset += 4)
    {
	pixbuf_data[offset + 0] = red;
	pixbuf_data[offset + 1] = green;
	pixbuf_data[offset + 2] = blue;
	pixbuf_data[offset + 3] = 255;	
    }

    return gdk_pixbuf_new_from_data (pixbuf_data,
				     GDK_COLORSPACE_RGB,
				     true,
				     8,
				     width, height,
				     width * 4,
				     free_pixdata_delete, 0);
    
}

void Form::draw_frame_modal ()
{
    int width = res->width - 1;
    int height = res->height - 1;

    GdkPixbuf *pixbuf;

    pixbuf = draw_solid (width + 1, 2);
    FormEditor::draw_pixbuf (*frame_group, pixbuf, 0, -2);
    FormEditor::draw_pixbuf (*frame_group, pixbuf, 0, height + 1);
    gdk_pixbuf_unref (pixbuf);

    pixbuf = draw_solid (2, height + 1);
    FormEditor::draw_pixbuf (*frame_group, pixbuf, -2, 0);
    FormEditor::draw_pixbuf (*frame_group, pixbuf, width + 1, 0);
    gdk_pixbuf_unref (pixbuf);

    pixbuf = draw_solid (2, 2);
    FormEditor::draw_pixbuf (*frame_group, pixbuf, -1, -1);
    FormEditor::draw_pixbuf (*frame_group, pixbuf, width, -1);
    FormEditor::draw_pixbuf (*frame_group, pixbuf, width, height);
    FormEditor::draw_pixbuf (*frame_group, pixbuf, -1, height);
    gdk_pixbuf_unref (pixbuf);
}

void Form::draw_title ()
{
    delete title_group;
    title_group = new Gnome::CanvasGroup (*group, 0, 0);

    if (res->modal)
	draw_title_modal ();
    else
	draw_title_nonmodal ();
}

void Form::draw_title_nonmodal ()
{
    Gnome::CanvasPoints points;
    int title_width;
    const Font &font = get_title_font ();

    title_width = get_string_width (1, res->title); /* Title is font#1 (FIXME) */
    title_width = std::min<int> (title_width + 4, res->width);

    int max_width = res->width;
    int max_height = std::min<int> (res->height, GUIKACHU_WINDOW_TITLE_HEIGHT);
    
    // NW corner
    points.push_back (Gnome::Art::Point (0, 1));
    points.push_back (Gnome::Art::Point (1, 0));

    // NE corner
    points.push_back (Gnome::Art::Point (title_width,     0));
    points.push_back (Gnome::Art::Point (title_width + 1, 1));

    // SE corner
    points.push_back (Gnome::Art::Point ((title_width + 1), (GUIKACHU_WINDOW_TITLE_HEIGHT + 1)));
    // SW corner
    points.push_back (Gnome::Art::Point (0, (GUIKACHU_WINDOW_TITLE_HEIGHT + 1)));
    // Back to NW
    points.push_back (Gnome::Art::Point (0, 1));

    {
	using namespace Gnome::CanvasHelpers;

	*(new Gnome::CanvasPolygon (*title_group, points))
	    << width_units (1)
	    << fill_color (get_foreground_color (true));

	render_text (*title_group, res->title, font,
		     get_background_color (),
		     3, 2,
		     GTK_ANCHOR_NW,
		     max_width, max_height);
	
	*(new Gnome::CanvasRect (*title_group,
				 0, GUIKACHU_WINDOW_TITLE_HEIGHT,
				 res->width - 1, GUIKACHU_WINDOW_TITLE_HEIGHT + 1))
	    << width_units (1)
	    << fill_color (get_foreground_color (true));
    }
}

void Form::draw_title_modal ()
{
    Gnome::CanvasPoints points;
    const Font &font = get_title_font ();

    int title_center = res->width / 2;
    int title_height = std::min<int> (res->height, 11);

    int max_width = res->width;
    int max_height = std::min<int> (res->height, 11);

    {
	using namespace Gnome::CanvasHelpers;
    
	*(new Gnome::CanvasRect (*title_group,
				 0, 0,
				 res->width, title_height))
	    << width_units (1)
	    << fill_color (get_foreground_color (true));

	render_text (*title_group, res->title, font,
		     get_background_color (),
		     title_center, 0,
		     GTK_ANCHOR_N,
		     max_width, max_height);
    }
}

void Form::selected_cb (bool selected)
{
}

int Form::canvas_event_cb (GdkEvent *e)
{
    // Single click
    if (e->type == GDK_BUTTON_PRESS &&
	e->button.button == 1)
	clicked ();

    // Context menu
    if ((e->type == GDK_BUTTON_PRESS) &&
	(e->button.button == 3))
	popup_menu.popup (e->button.button, e->button.time);
    
    // Begin drag
    if (e->type == GDK_BUTTON_PRESS &&
	e->button.button == 1 &&
	!drag_context.dragging)
	drag_begin_impl (e);

    // Dragging
    if (e->type == GDK_MOTION_NOTIFY &&
	drag_context.dragging)
	drag_motion_impl (e);

    // End drag
    if (e->type == GDK_BUTTON_RELEASE &&
	(e->button.button == 1) &&
	drag_context.dragging)
	drag_end_impl (e);

    return true;
}

void Form::drag_begin_impl (GdkEvent *e)
{
    if ((e->type == GDK_BUTTON_PRESS) &&
	(e->button.button == 1))
    {
	drag_context.dragging = true;
	drag_context.drag_occured = false;

	drag_context.last_x = e->button.x;
	drag_context.last_y = e->button.y;

	drag_context.offset_x = drag_context.last_x - res->x;
	drag_context.offset_y = drag_context.last_y - res->y;

	drag_context.delta_remainder_x = 0;
	drag_context.delta_remainder_y = 0;	
	
	Gdk_Cursor cursor (GDK_FLEUR);
	group->grab (GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
		     cursor, e->button.time);
	cursor.destroy ();
    }
}

void Form::drag_motion_impl (GdkEvent *e)
{
    drag_context.drag_occured = true;

    double curr_x = e->motion.x;
    double curr_y = e->motion.y;
    
    // East bound
    if ((curr_x - drag_context.offset_x) > (GUIKACHU_SCREEN_WIDTH - 1))
	curr_x = (GUIKACHU_SCREEN_WIDTH - 1) + drag_context.offset_x;
    
    // South bound
    if ((curr_y - drag_context.offset_y) > (GUIKACHU_SCREEN_HEIGHT - 1))
	curr_y = (GUIKACHU_SCREEN_HEIGHT - 1) + drag_context.offset_y;
    
    // West bound
    if ((curr_x - drag_context.offset_x) < 0)
	curr_x = drag_context.offset_x;
    
    // North bound
    if ((curr_y - drag_context.offset_y) < 0)
	curr_y = drag_context.offset_y;

    // Remainders need to be collected
    double dx = (curr_x - drag_context.last_x) + drag_context.delta_remainder_x;
    double dy = (curr_y - drag_context.last_y) + drag_context.delta_remainder_y;

    drag_context.delta_remainder_x = fmod (dx, 1.0);
    drag_context.delta_remainder_y = fmod (dy, 1.0);
    
    drag_context.last_x = curr_x;
    drag_context.last_y = curr_y;

    int dxi = (int)(dx / 1);
    int dyi = (int)(dy / 1);
    group->move (dxi, dyi);
}

void Form::drag_end_impl (GdkEvent *e)
{
    group->ungrab (e->button.time);
    drag_context.dragging = false;
    
    if (drag_context.drag_occured)
    {
	int dx = (int)(drag_context.last_x - drag_context.offset_x - drag_context.delta_remainder_x);
	int dy = (int)(drag_context.last_y - drag_context.offset_y - drag_context.delta_remainder_y);

	res->x = dx;
	res->y = dy;
    }
}
