//$Id: resource-manager-io.cc,v 1.20 2001/12/15 12:21:36 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "resource-manager.h"

#include "config.h"
#include <libgnome/libgnome.h>

/* Having to include these for just open()/close() and *time() sure sucks */
#include <fstream>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "storage.h"

using namespace Guikachu;

void ResourceManager::reset ()
{
    resource_map_t resource_map_copy (resource_map);
    
    for (resource_map_t::iterator i = resource_map.begin ();
	 i != resource_map.end ();)
	remove_resource ((i++)->second);
    
    if (!app)
	app = new Resources::Application ();
    else
	app->reset ();

    clear_dirty ();
}

ResourceManager::IOResult ResourceManager::load (const std::string &filename)
{
    Storage storage;

    int fd = open (filename.c_str (), O_RDONLY);
    if (fd == -1)
	return IO_NOTFOUND;
    close (fd);
	
    
    if (!storage.load (filename))
	return IO_INVALID;
	
    StorageNode root_node = storage.get_root ();
    if (root_node.name () != "guikachu")
        return IO_INVALID;

    /*
     * Walk the tree
     */
    reset ();

    for (StorageNode curr_node = root_node.children ();
         curr_node; curr_node++)
    {
	if (curr_node.name () == "application")
	    app->load (curr_node);
	else
	    load_resource (curr_node);
    }

    clear_dirty ();

    return IO_OK;
}

Resources::Resource* ResourceManager::load_resource (StorageNode &node)
{
    Resources::Resource *resource = 0;

    std::string node_name = node.name ();
    std::string id = node.get_prop<std::string> ("id");

    if (id == "")
    {
	g_warning ("`%s': Missing ID field", node_name.c_str ());
	return NULL;
    }

    if (node_name == "dialog")
	resource = create_resource (Resources::RESOURCE_DIALOG, id);
    else if (node_name == "form")
	resource = create_resource (Resources::RESOURCE_FORM, id);
    else if (node_name == "menu")
	resource = create_resource (Resources::RESOURCE_MENU, id);
    else if (node_name == "string")
	resource = create_resource (Resources::RESOURCE_STRING, id);
    else if (node_name == "stringlist")
	resource = create_resource (Resources::RESOURCE_STRINGLIST, id);
    else if (node_name == "blob")
	resource = create_resource (Resources::RESOURCE_BLOB, id);
    else
	g_warning ("Unknown resource type `%s'", node_name.c_str ());

    if (resource)
	resource->load (node);
    
    return resource;
}

ResourceManager::IOResult ResourceManager::save (const std::string &filename)
{
    Storage     storage;
    StorageNode root = storage.create_root ("guikachu");
    StorageNode node;

    // Save per-application data
    node = root.add_node ("application");
    app->save (node);
    
    // Save resources
    for (std::map<std::string, Resources::Resource*>::iterator i = resource_map.begin ();
	 i != resource_map.end ();
	 i++)
    {
	Resources::Resource *resource = i->second;
	std::string curr_node_name;

	switch (resource->get_type ())
	{
	case Resources::RESOURCE_STRING:
	    curr_node_name = "string";
	    break;
	case Resources::RESOURCE_STRINGLIST:
	    curr_node_name = "stringlist";
	    break;
	case Resources::RESOURCE_BLOB:
	    curr_node_name = "blob";
	    break;
	case Resources::RESOURCE_DIALOG:
	    curr_node_name = "dialog";
	    break;
	case Resources::RESOURCE_MENU:
	    curr_node_name = "menu";
	    break;
	case Resources::RESOURCE_FORM:
	    curr_node_name = "form";
	    break;
	}

	node = root.add_node (curr_node_name);
	node.set_prop ("id", resource->id);
	resource->save (node);
    }

    storage.save (filename);

    clear_dirty ();

    return IO_OK;
}

void ResourceManager::export_rcp (const std::string &filename)
{
    std::ofstream fout (filename.c_str ());

    time_t curr_date = time (0);
    const char* date_str = ctime (&curr_date);
    
    fout << "/*" << std::endl;
    fout << " * This file was created by Guikachu " << VERSION
	 << ". Do not edit it manually." << std::endl;
    fout << " * Creation date: " << date_str; // ctime ()'s return value ends in \n
    fout << " */" << std::endl << std::endl;;
    
    fout << app;
    
    for (std::map<std::string, Resources::Resource*>::iterator i = resource_map.begin ();
	 i != resource_map.end ();
	 i++)
    {
	fout << i->second;
    }
    
    fout.close ();
}
