//$Id: table.cc,v 1.7 2001/11/04 11:42:14 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "table.h"
#include "table-canvas.h"

#include "form-editor.h"

using Guikachu::Widgets::Table;
using namespace Guikachu::GUI::FormEditor;

Table::Table (Resources::Form *owner,
	      const string& id):
    Widget       (owner, id),
    Resizeable   (changed),
    editor       (0),
    num_rows     (changed, 1),
    num_columns  (changed, 1),
    column_width (changed, vector<int>(1, 1))
{
    changed.connect (SigC::slot (this, &Table::update));
}

Table::~Table ()
{
    if (editor)
	delete editor;
}

Gtk::Widget *Table::create_editor ()
{
    if (!editor)
	editor = new TableProperties (this);
    
    return editor->get_editor ();
}

CanvasItem* Table::create_canvas_item (GUI::FormEditor::Form *form)
{
    return new TableCanvasItem (this, form);
}

bool Table::load (StorageNode &node)
{
    column_width = vector<int>(0);
    num_columns  = 0;
    
    for (StorageNode curr_node  = node.children ();
	 curr_node; curr_node++)
    {
	if (curr_node.name () == "pos")
	{
	    x = curr_node.get_prop<int> ("x");
	    y = curr_node.get_prop<int> ("y");

	    string width_str = curr_node.get_prop<string> ("width");
	    if (width_str == "auto")
	    {
		manual_width = false;
		width = 0;
	    } else {
		manual_width = true;
		width = atoi (width_str.c_str ());
	    }

	    string height_str = curr_node.get_prop<string> ("height");
	    if (height_str == "auto")
	    {
		manual_height = false;
		height = 0;
	    } else {
		manual_height = true;
		height = atoi (height_str.c_str ());
	    }
	}
	else if (curr_node.name () == "table_data")
	{
	    num_rows = curr_node.get_prop<int> ("rows");
	    
	    /* iterate over column fields */
	    vector<int> columns;
	    StorageNode column_node;
	    for (column_node = curr_node.children ();
		 column_node; column_node++)
	    {
		if (column_node.name () == "column")
		    columns.push_back (column_node.get_prop<int> ("width"));
		else
		    g_warning ("Error parsing `table' resource: "
			       "unexpected element `%s/%s'",
			       curr_node.name ().c_str (),
			       column_node.name ().c_str ());
	    }
	    column_width = columns;
	    num_columns = columns.size ();
	}
	else
	    g_warning ("Error parsing `table' widget: "
		       "unexpected element `%s'",
		       curr_node.name ().c_str ());
    }

    changed ();

    return true;
}

void Table::save (StorageNode &node) const
{
    StorageNode my_node;

    /* Position */
    my_node = node.add_node ("pos");
    my_node.set_prop ("x", x);
    my_node.set_prop ("y", y);

    /* Size */
    if (manual_width)
	my_node.set_prop ("width", width);
    else
	my_node.set_prop ("width", "auto");

    if (manual_height)
	my_node.set_prop ("height", height);
    else
	my_node.set_prop ("height", "auto");

    my_node = node.add_node ("table_data");

    /* Number of rows and columns */
    my_node.set_prop ("rows", num_rows);

    vector<int> columns = column_width;
    for (int i = 0; i < num_columns; i++)
	my_node.add_node ("column").set_prop ("width", columns[i]);
}

void Table::save_rcp (ostream &ostr) const
{
    gchar *width_str, *height_str;

    if (manual_width)
	width_str = g_strdup_printf ("%d", width ());
    else
	width_str = g_strdup ("AUTO");
    
    if (manual_height)
	height_str = g_strdup_printf ("%d", height ());
    else
	height_str = g_strdup ("AUTO");

    ostr << "  TABLE ID " << id;
    ostr << " AT (" << x << " " << y << " " << width_str << " " << height_str << ")";
    ostr << " ROWS "   << num_rows;

    vector<int> columns = column_width;
    ostr << " COLUMNS " << num_columns;
    ostr << " COLUMNWIDTHS ";
    for (int i = 0; i < num_columns; i++)
	ostr << columns[i] << " ";
	
    ostr << endl;

    g_free (height_str);
    g_free (width_str);
}

int Table::get_auto_width ()
{
    int ret_val = 0;
    
    const vector<int> &columns = column_width;
    for (unsigned int i = 0; i < columns.size (); i++)
	ret_val += columns[i];
    
    return ret_val;
}

int Table::get_auto_height ()
{
    return 100;
}

void Table::update ()
{
    /* Make sure column_width contains at least num_columns elements */
    
    const vector<int> &columns = column_width;
    
    int num_columns_in_vec = columns.size ();
    int num_columns_target = num_columns;

    if (num_columns_in_vec < num_columns_target)
    {
	vector<int> columns_copy = columns;
	
	for (int i = num_columns_in_vec; i < num_columns_target; i++)
	    columns_copy.push_back (1);
	
	column_width = columns_copy;
    }
}
