//$Id: canvasitem.cc,v 1.16 2001/10/27 13:29:26 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "canvasitem.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include <gnome--/canvas-rect.h>
#include <gnome--/app-helper.h>
#include <libgnomeui/gnome-stock.h>

#include <list>
#include <math.h>

#include "form-editor.h"

using namespace Guikachu::GUI::FormEditor;

CanvasItem::CanvasItem (Widgets::Widget  *widget_,
			FormEditor::Form *form_):
    item (0),
    widget (widget_),
    bounding_box (0),
    form (form_),
    selected (false)
{
    drag_context.dragging = false;
    drag_context.drag_occured = false;
    
    // Popup menu
    list <Gnome::UI::Info> popup_menu_list;
    popup_menu_list.push_back (
	Gnome::UI::Item (Gnome::UI::Icon (GNOME_STOCK_PIXMAP_TRASH),
			 _("Remove"), menu_remove.slot ()));
    
    Gnome::UI::fill (popup_menu, popup_menu_list, *(popup_menu.get_accel_group ()));
    
    widget->changed.connect  (SigC::slot (this, &CanvasItem::update));
    widget->selected.connect (SigC::slot (this, &CanvasItem::widget_selected_cb));
}

CanvasItem::~CanvasItem ()
{
    if (item)
	delete item;

    if (bounding_box)
	delete bounding_box;
}

void CanvasItem::set_canvas_item (Gnome::CanvasGroup *item_)
{
    item = item_;
    bounding_box = 0;
    
    item->event.connect (SigC::slot (this, &CanvasItem::canvas_event_cb));
    widget_selected_cb (selected);
}

int CanvasItem::canvas_event_cb (GdkEvent *e)
{
    // Single click
    if ((e->type == GDK_BUTTON_PRESS) &&
	(e->button.button == 1))
	clicked ();

    // Context menu
    if ((e->type == GDK_BUTTON_PRESS) &&
	(e->button.button == 3))
	popup_menu.popup (e->button.button, e->button.time);
    
    // Begin drag
    if (e->type == GDK_BUTTON_PRESS &&
	e->button.button == 1 &&
	!drag_context.dragging)
	drag_begin_impl (e);
    
    // Dragging
    if (e->type == GDK_MOTION_NOTIFY &&
	drag_context.dragging)
	drag_motion_impl (e);

    // Ending drag
    if (e->type == GDK_BUTTON_RELEASE &&
	(e->button.button == 1) &&
	drag_context.dragging)
	drag_end_impl (e);
    
    return TRUE;
}

void CanvasItem::drag_begin_impl (GdkEvent *e)
{
    drag_context.dragging = true;
    drag_context.drag_occured = false;

    drag_context.last_x = drag_context.start_x = e->button.x;
    drag_context.last_y = drag_context.start_y = e->button.y;
    
    drag_context.offset_x = drag_context.last_x - widget->x;
    drag_context.offset_y = drag_context.last_y - widget->y;

    drag_context.delta_remainder_x = 0;
    drag_context.delta_remainder_y = 0;

    Gdk_Cursor cursor (GDK_FLEUR);
    item->grab (GDK_POINTER_MOTION_MASK | GDK_BUTTON_MOTION_MASK |
		GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK,
		cursor, e->button.time);
    cursor.destroy ();

    drag_begin.emit ();
}

void CanvasItem::drag_motion_impl (GdkEvent *e)
{
    drag_context.drag_occured = true;
    
    double curr_x = e->motion.x;
    double curr_y = e->motion.y;
    
    // East bound
    if ((curr_x - drag_context.offset_x) > (GUIKACHU_SCREEN_WIDTH - 1))
	    curr_x = (GUIKACHU_SCREEN_WIDTH - 1) + drag_context.offset_x;
    
    // South bound
    if ((curr_y - drag_context.offset_y) > (GUIKACHU_SCREEN_HEIGHT - 1))
	curr_y = (GUIKACHU_SCREEN_HEIGHT - 1) + drag_context.offset_y;
    
    // West bound
    if ((curr_x - drag_context.offset_x) < 0)
	curr_x = drag_context.offset_x;
    
    // North bound
    if ((curr_y - drag_context.offset_y) < 0)
	curr_y = drag_context.offset_y;

    // Remainders need to be collected
    double dx = (curr_x - drag_context.last_x) + drag_context.delta_remainder_x;
    double dy = (curr_y - drag_context.last_y) + drag_context.delta_remainder_y;

    drag_context.delta_remainder_x = fmod (dx, 1.0);
    drag_context.delta_remainder_y = fmod (dy, 1.0);
    
    drag_context.last_x = curr_x;
    drag_context.last_y = curr_y;     

    drag_motion.emit (dx / 1, dy / 1);
}

void CanvasItem::drag_end_impl (GdkEvent *e)
{
    item->ungrab (e->button.time);
    drag_context.dragging = false;
    
    if (drag_context.drag_occured)
    {
	int dx = (int)(drag_context.last_x - drag_context.start_x - drag_context.delta_remainder_x);
	int dy = (int)(drag_context.last_y - drag_context.start_y - drag_context.delta_remainder_y);

	drag_end.emit (dx, dy);
    }
}

void CanvasItem::move (int dx, int dy)
{
    item->move (dx, dy);
}

void CanvasItem::widget_selected_cb (bool selected_)
{
    selected = selected_;
    
    if (!selected)
    {
	if (bounding_box)
	{
	    delete bounding_box;
	    bounding_box = 0;
	}
	return;
    }

    delete bounding_box;
    bounding_box = create_bounding_box ();
    item->raise_to_top ();
}

Gnome::CanvasItem * CanvasItem::create_bounding_box ()
{
    int x1, y1, x2, y2;
    
    get_bounds (x1, y1, x2, y2);

    using namespace Gnome::CanvasHelpers;
    Gnome::CanvasItem *bounding_box = new Gnome::CanvasRect (*item, x1, y1, x2, y2);
    *bounding_box
	<< width_units (2)
	<< outline_color (GUIKACHU_SELECTED);

    return bounding_box;
}

void CanvasItem::get_bounds (int &x1, int &y1,
			     int &x2, int &y2)
{
    double x1_, y1_, x2_, y2_;
    
    item->get_bounds (x1_, y1_, x2_, y2_);

    x1 = (int)x1_ - 2;
    y1 = (int)y1_ - 2;
    x2 = (int)x2_ + 2;
    y2 = (int)y2_ + 2;
}
