//$Id: resource-manager-io.cc,v 1.9 2001/04/27 21:32:11 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "resource-manager.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include <fstream>
#include <time.h>
#include <gnome-xml/parser.h>

#include "mainwin.h"

using namespace Guikachu;

void ResourceManager::reset ()
{
    map<string_case, Resources::Resource*>::iterator i;
    
    for (i = resource_map.begin ();
	 i != resource_map.end ();
	 i++)
    {
	remove_resource (i->second);
    }

    if (!app)
	app = new Resources::Application ();
    else
	app->reset ();
    
    dirty = false;
}

void ResourceManager::load (const string& filename)
{
    reset ();
    
    xmlDocPtr doc;
    xmlNodePtr root_node, curr_node;

    doc = xmlParseFile (filename.c_str ());

    if (!doc)
    {
	gchar *buffer = g_strdup_printf (_("Unable to open `%s'"),
					 filename.c_str ());
	GUI::show_error (buffer);
	g_free (buffer);

	return;
    }
	
    root_node = xmlDocGetRootElement (doc);
    
    if (strcmp ((char*)root_node->name, "guikachu"))
    {
	gchar *buffer = g_strdup_printf (_("Unable to open `%s': "
					   "Not a valid Guikachu document"),
					 filename.c_str ());
	GUI::show_error (buffer);
	g_free (buffer);

        xmlFreeDoc(doc);
        return;
    }

    /*
     * Walk the tree
     */
    for (curr_node = root_node->xmlChildrenNode;
         curr_node != NULL;
         curr_node = curr_node->next)
    {
	if (!strcmp ((char*)curr_node->name, "application"))
	    app->load (doc, curr_node);
	else
	    load_resource (doc, curr_node);
    }
    
    dirty = false;
}

Resources::Resource* ResourceManager::load_resource (xmlDocPtr doc,
						     xmlNodePtr node)
{
    Resources::Resource *resource = 0;
    
    char *node_name = (char*)node->name;
    char *id = (char*)xmlGetProp (node, (xmlChar*)"id");

    if (!id)
    {
	g_warning ("`%s': Missing ID field", node_name);
	return NULL;
    }

    if (!strcmp (node_name, "dialog"))
	resource = create_resource (Resources::RESOURCE_DIALOG, id);
    else if (!strcmp (node_name, "form"))
	resource = create_resource (Resources::RESOURCE_FORM, id);
    else if (!strcmp (node_name, "menu"))
	resource = create_resource (Resources::RESOURCE_MENU, id);
    else if (!strcmp (node_name, "string"))
	resource = create_resource (Resources::RESOURCE_STRING, id);
    else
	g_warning ("Unknown resource type `%s'", node_name);

    if (resource)
	resource->load (doc, node);
    
    return resource;
}

void ResourceManager::save (const string& filename)
{
    xmlDocPtr doc = xmlNewDoc ((xmlChar*)"1.0");
    xmlNodePtr root = xmlNewNode (NULL, (xmlChar*)"guikachu");
    xmlNodePtr node = 0;
    
    xmlDocSetRootElement (doc, root);

    // Save per-application data
    node = xmlNewChild (root, NULL, (xmlChar*) "application", NULL);
    app->save (node);
    
    // Save resources
    for (map<string_case, Resources::Resource*>::iterator i = resource_map.begin ();
	 i != resource_map.end ();
	 i++)
    {
	xmlChar *curr_node_name = 0;

	switch (i->second->get_type ())
	{
	case Resources::RESOURCE_STRING:
	    curr_node_name = (xmlChar*) "string";
	    break;
	case Resources::RESOURCE_DIALOG:
	    curr_node_name = (xmlChar*) "dialog";
	    break;
	case Resources::RESOURCE_MENU:
	    curr_node_name = (xmlChar*) "menu";
	    break;
	case Resources::RESOURCE_FORM:
	    curr_node_name = (xmlChar*) "form";
	    break;
	}
	
	node = xmlNewChild (root, NULL, curr_node_name, NULL);
	xmlSetProp (node, (xmlChar*)"id", (xmlChar*)i->second->get_id ().c_str ());
	i->second->save (node);
    }
    
    xmlSaveFile (filename.c_str(), doc);

    dirty = false;
}

void ResourceManager::export_rcp (const string& filename)
{
    ofstream fout (filename.c_str ());

    time_t curr_date = time (0);
    const char* date_str = ctime (&curr_date);
    
    fout << "/*" << endl;
    fout << " * This file was created by Guikachu. Do not edit it  manually" << endl;
    fout << " * Creation date: " << date_str; // ctime ()'s return value ends in \n
    fout << " */" << endl << endl;;
    
    fout << app;
    
    for (map<string_case, Resources::Resource*>::iterator i = resource_map.begin ();
	 i != resource_map.end ();
	 i++)
    {
	fout << i->second;
    }
    
    fout.close ();
}
