//$Id: button.cc,v 1.10 2001/04/22 13:03:11 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "button.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include "form-editor.h"

#include <gnome--.h>
#include <gnome-xml/xmlmemory.h>

using Guikachu::Widgets::Button;
using namespace Guikachu::GUI::FormEditor;

namespace Guikachu
{
    namespace GUI
    {
	namespace FormEditor
	{
	    class ButtonCanvasItem: public CanvasItem
	    {
		Widgets::Button    *widget;
		FormEditor::Form   *form;
		Gnome::CanvasGroup *group;
	    public:
		ButtonCanvasItem (Widgets::Button  *widget,
				  FormEditor::Form *form);
	    private:
		void update ();
	    };
	}
    }
}

Button::Button (Resources::Form *owner_,
		const string& id_):
    text (id_),
    font (0),
    frame (BUTTON_FRAME_SIMPLE),
    usable (true),
    repeat (false),
    anchor_right (false),
    width (1),
    manual_width (false),
    height (1),
    manual_height (false)
{
    id = id_;
    owner = owner_;

    x = 5;
    y = 20;

    editor = new ButtonProperties (this);
}

Button::~Button ()
{
    delete editor;
}

Gtk::Widget *Button::get_editor ()
{
    return editor->get_editor ();
}

bool Button::load (xmlDocPtr doc,
		   xmlNodePtr node)
{
    xmlNodePtr curr_node;

    bool xml_pos_set = false;
    bool xml_text_set = false;
    
    gchar *xml_val;

    /* Non-NULL default values */
    usable = false;
    
#define XML_HAS_NAME(node, val) (!strcmp ((char*) node->name, val))
    
#define XML_APPLY(attr)       \
	if (xml_val)          \
	{                     \
	    attr = xml_val;   \
	    xmlFree (xml_val);\
	}

#define XML_GET_CONTENT(node, attr) \
	xml_val = (char*)xmlNodeListGetString (doc, node->xmlChildrenNode, 1); \
        XML_APPLY(attr);
    
#define XML_GET_NUM_PROP(node, prop, attr)			\
	xml_val = (char*)xmlGetProp (node, (xmlChar*) prop);	\
	if (xml_val)						\
	{							\
	    attr = atoi(xml_val);				\
	    xmlFree (xml_val);					\
	}

    for (curr_node  = node->xmlChildrenNode;
	 curr_node != NULL;
	 curr_node  = curr_node->next)
    {
	if (XML_HAS_NAME (curr_node, "pos") && !xml_pos_set)
	{
	    XML_GET_NUM_PROP (curr_node, "x", x);
	    XML_GET_NUM_PROP (curr_node, "y", y);

	    xml_val = (char*)xmlGetProp (curr_node, (xmlChar*) "width");
	    if (!xml_val)
	    {
		manual_width = false;
	    } else {
		if (!strcasecmp (xml_val, "auto"))
		{
		    manual_width = false;
		} else {
		    manual_width = true;
		    width = atoi (xml_val);
		}
		xmlFree (xml_val);
	    }
	    
	    xml_val = (char*)xmlGetProp (curr_node, (xmlChar*) "height");
	    if (!xml_val)
	    {
		manual_height = false;
	    } else {
		if (!strcasecmp (xml_val, "auto"))
		{
		    manual_height = false;
		} else {
		    manual_height = true;
		    height = atoi (xml_val);
		}
		xmlFree (xml_val);
	    }
	    
	    xml_pos_set = true;
	}
	else if (XML_HAS_NAME (curr_node, "text") && !xml_text_set)
	{
	    XML_GET_CONTENT (curr_node, text);
	    XML_GET_NUM_PROP (curr_node, "font", font);
		
	    xml_text_set = true;
	}
	else if (XML_HAS_NAME (curr_node, "usable"))
	{
	    usable = true;
	}
	else if (XML_HAS_NAME (curr_node, "repeat"))
	{
	    repeat = true;
	}
	else if (XML_HAS_NAME (curr_node, "anchor_right"))
	{
	    anchor_right = true;
	}
	else
	    g_warning ("Error parsing `button' widget: "
		       "unexpected element `%s'", curr_node->name);
    }

    changed ();

    return true;
}

void Button::save (xmlNodePtr node) const
{
    xmlNodePtr my_node;
    gchar *buffer;

    /* Position */
    my_node = xmlNewChild (node, NULL, (xmlChar*) "pos", NULL);
    buffer = g_strdup_printf ("%d", x);
    xmlSetProp (my_node, (xmlChar*) "x", (xmlChar*) buffer);
    g_free (buffer);
    buffer = g_strdup_printf ("%d", y);
    xmlSetProp (my_node, (xmlChar*) "y", (xmlChar*) buffer);
    g_free (buffer);

    /* Frame */
    if (manual_width)
	buffer = g_strdup_printf ("%d", width);
    else
	buffer = g_strdup ("auto");
    xmlSetProp (my_node, (xmlChar*) "width", (xmlChar*) buffer);
    g_free (buffer);

    if (manual_height)
	buffer = g_strdup_printf ("%d", height);
    else
	buffer = g_strdup ("auto");
    xmlSetProp (my_node, (xmlChar*) "height", (xmlChar*) buffer);
    g_free (buffer);

    /* Usable */
    if (usable)
	my_node = xmlNewChild (node, NULL, (xmlChar*) "usable", NULL);

    /* Repeating */
    if (repeat)
	my_node = xmlNewChild (node, NULL, (xmlChar*) "repeat", NULL);

    /* Anchor_right */
    if (anchor_right)
	my_node = xmlNewChild (node, NULL, (xmlChar*) "anchor_right", NULL);
    
    /* Label and font */
    my_node = xmlNewChild (node, NULL, (xmlChar*) "text",
			   (xmlChar*) text.c_str ());
    buffer = g_strdup_printf ("%d", font);
    xmlSetProp (my_node, (xmlChar*) "font", (xmlChar*) buffer);
    g_free (buffer);
}

void Button::save_rcp (ostream &ostr) const
{
    string frame_str, anchor_str;
    gchar *width_str, *height_str;
    string type_str;

    if (!repeat)
	type_str = "BUTTON";
    else
	type_str = "REPEATBUTTON";
    
    if (manual_width)
	width_str = g_strdup_printf ("%d", width);
    else
	width_str = g_strdup ("AUTO");
    
    if (manual_height)
	height_str = g_strdup_printf ("%d", height);
    else
	height_str = g_strdup ("AUTO");
    
    switch (frame)
    {
    case BUTTON_FRAME_NONE:
	frame_str = "NOFRAME";
	break;
    case BUTTON_FRAME_SIMPLE:
	frame_str = "FRAME";
	break;
    case BUTTON_FRAME_BOLD:
	frame_str = "BOLDFRAME";
	break;
    }

    if (anchor_right)
	anchor_str = "RIGHTANCHOR";
    else
	anchor_str = "LEFTANCHOR";
    
    ostr << "  " << type_str << " \"" << text << "\" ID " << id;
    ostr << " AT (" << x << " " << y << " " << width_str << " " << height_str << ")";
    ostr << " FONT " << font;
    ostr << " " << frame_str;
    ostr << " " << anchor_str;
    if (usable)
	ostr << " USABLE";
    
    ostr << endl;

    g_free (height_str);
    g_free (width_str);
}

ButtonCanvasItem::ButtonCanvasItem (Widgets::Button  *widget_,
				    FormEditor::Form *form_):
    CanvasItem (widget_),
    widget (widget_),
    form (form_),
    group (0)
{
    using namespace SigC;
    
    widget->changed.connect (slot (this, &ButtonCanvasItem::update));
    update ();
}

void ButtonCanvasItem::update ()
{
    if (group)
	delete group;

    group = new Gnome::CanvasGroup (*(form->get_canvas ()), 0, 0);
    set_canvas_item (group);
    
    GdkFont *font = get_font (widget->get_font ());
    
    int width = widget->get_width ();
    int height = widget->get_height ();
    int frame_width = 0;
    
    int text_height, text_width;
    int descent, ascent;
    int dummy;

    gdk_string_extents (font, widget->get_text ().c_str (),
			&dummy, &dummy,
			&text_width, &ascent, &descent);
    text_height = ascent + descent;
    
    if (!widget->get_manual_width ())
	width = 2 + text_width + 2;
    if (!widget->get_manual_height ())
	height = 4 + text_height + 4;

    switch (widget->get_frame ())
    {
    case Widgets::BUTTON_FRAME_NONE:
	frame_width = 0;
	break;
    case Widgets::BUTTON_FRAME_SIMPLE:
	frame_width = 1;
	break;
    case Widgets::BUTTON_FRAME_BOLD:
	frame_width = 2;
	break;
    }
    
    int x1 = widget->get_x ();
    int x2 = x1 + width;
    int y1 = widget->get_y ();
    int y2 = y1 + height;

    Gnome::CanvasPoints frame_points;
    using namespace Gnome::Art;

    frame_points.push_back (Point (x1,     y1 + 2));
    frame_points.push_back (Point (x1 + 2, y1    ));

    frame_points.push_back (Point (x2 - 2, y1    ));
    frame_points.push_back (Point (x2,     y1 + 2));
    
    frame_points.push_back (Point (x2,     y2 - 2));
    frame_points.push_back (Point (x2 - 2, y2));
    
    frame_points.push_back (Point (x1 + 2, y2));
    frame_points.push_back (Point (x1,     y2 - 2));

    frame_points.push_back (Point (x1,     y1 + 2));

    if (widget->get_frame () != Widgets::BUTTON_FRAME_NONE)
    {
	Gnome::CanvasPolygon *button_frame = new Gnome::CanvasPolygon (
	    *group,
	    "points", frame_points.gtkobj (),
	    "width_units", (double) frame_width,
	    0);
	if (widget->get_usable ())
	    button_frame->set_outline_color (GUIKACHU_FGC);
	else
	    button_frame->set_outline_color (GUIKACHU_FGC_DISABLED);
    }

    {
	int text_x = x1 + width / 2;
	int text_y = y1 + height / 2;
	
	Gnome::CanvasText *button_text = new Gnome::CanvasText (
	    *group,
	    "text", widget->get_text ().c_str (),
	    "font_gdk", font,
	    "x", (double) text_x,
	    "y", (double) text_y,
	    "clip_width", (double) width,
	    "clip_height", (double) height,
	    "clip", TRUE,
	    "anchor", GTK_ANCHOR_CENTER,
	    "fill_color", GUIKACHU_FGC,
	    0);
	
	if (widget->get_usable ())
	    button_text->set_fill_color (GUIKACHU_FGC);
	else
	    button_text->set_fill_color (GUIKACHU_FGC_DISABLED);
    }
    
    gdk_font_unref (font);
}

CanvasItem* Button::get_canvas_item (GUI::FormEditor::Form *form)
{
    return new ButtonCanvasItem (this, form);
}
