//$Id: dialog-win.cc,v 1.5 2001/03/04 14:07:48 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "dialog-win.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include <gtk--/menu.h>
#include <gnome--/dialog.h>
#include <sigc++/retbind.h>

using namespace Guikachu;

GUI::DialogWindow::DialogWindow (Guikachu::Resources::Dialog *res_):
    res (res_),
    help_res (0),
    update_block (true)
{
    gui = glade_xml_new (GLADEDIR "/guikachu.glade", 0);

    using namespace Gtk;
    
    window = wrap (GTK_WINDOW (glade_xml_get_widget (gui, "res_dialog")));
    window->delete_event.connect (slot (this, &DialogWindow::delete_event_impl));


    // Widgets for page 1
    id_entry = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "dialog_name")));
    title_entry = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "dialog_title")));
    text_entry = wrap (GTK_TEXT (glade_xml_get_widget (gui, "dialog_text")));
    type_combo = wrap (GTK_OPTION_MENU (glade_xml_get_widget (gui, "dialog_type")));
    help_combo = wrap (GTK_COMBO (glade_xml_get_widget (gui, "dialog_help")));
    btnHelpEdit = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "dialog_help_edit")));
    
    text_entry->set_word_wrap (true);

    // Set up callbacks
    id_entry->activate.connect (bind (
	slot (this, &DialogWindow::id_entry_cb), (GdkEventFocus*) 0));
    id_entry->focus_out_event.connect (retbind (
	slot (this, &DialogWindow::id_entry_cb), 0));

    title_entry->activate.connect (bind (
	slot (this, &DialogWindow::title_entry_cb), (GdkEventFocus*) 0));
    title_entry->focus_out_event.connect (retbind (
	slot (this, &DialogWindow::title_entry_cb), 0));

    text_entry->changed.connect (slot (this, &DialogWindow::text_entry_cb));

    btnHelpEdit->clicked.connect (slot (this, &DialogWindow::help_edit_cb));
    
    // Dialog type dropdown
    Menu *type_menu = new Menu ();

#define DIALOG_TYPE(s,t) \
	    type_menu->items ().push_back ( \
		Menu_Helpers::MenuElem ( \
		    s, \
		    bind (slot (this, &DialogWindow::type_menu_cb), \
			  Resources::t)));

    DIALOG_TYPE (_("Information"),  DIALOG_INFORMATION);
    DIALOG_TYPE (_("Confirmation"), DIALOG_CONFIRMATION);
    DIALOG_TYPE (_("Warning"),      DIALOG_WARNING);
    DIALOG_TYPE (_("Error"),        DIALOG_ERROR);

#undef DIALOG_TYPE
    
    type_menu->show_all ();
    type_combo->set_menu (*manage (type_menu));

    // Fill the Help ID combo list
    set<Resources::Resource*> resource_set = ResourceManager::get_resources ();
    vector<string> help_list;
    help_list.push_back ("");
    for (set<Resources::Resource*>::iterator i = resource_set.begin ();
	 i != resource_set.end (); i++)
    {
	if ((*i)->get_type () == Resources::RESOURCE_STRING)
	{
	    help_list.push_back ((*i)->get_id ());
	    (*i)->changed.connect (bind (
		slot (this, &DialogWindow::help_res_changed_cb),
		(*i)));
	}
    }
    help_combo->set_popdown_strings (help_list);

    ResourceManager::resource_created.connect (slot (this, &DialogWindow::resource_created_cb));
    ResourceManager::resource_removed.connect (slot (this, &DialogWindow::resource_removed_cb));
    help_combo->get_entry ()->changed.connect (slot (this, &DialogWindow::help_combo_cb));

    // Widgets for page 2
    button_list = wrap (GTK_LIST (glade_xml_get_widget (gui, "dialog_buttonlist")));
    button_list->selection_changed.connect (slot (this, &DialogWindow::button_sel_change_cb));

    btnAdd    =  wrap (GTK_BUTTON (glade_xml_get_widget (gui, "dialog_btn_add")));
    btnUp     =  wrap (GTK_BUTTON (glade_xml_get_widget (gui, "dialog_btn_up")));
    btnDown   =  wrap (GTK_BUTTON (glade_xml_get_widget (gui, "dialog_btn_down")));
    btnRemove =  wrap (GTK_BUTTON (glade_xml_get_widget (gui, "dialog_btn_remove")));
    btnDefault = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "dialog_btn_default")));

    btnAdd->clicked.connect     (slot (this, &DialogWindow::button_add_cb));
    btnUp->clicked.connect      (slot (this, &DialogWindow::button_up_cb));
    btnDown->clicked.connect    (slot (this, &DialogWindow::button_down_cb));
    btnRemove->clicked.connect  (slot (this, &DialogWindow::button_remove_cb));
    btnDefault->clicked.connect (slot (this, &DialogWindow::button_default_cb));
    
    res->changed.connect (slot (this, &DialogWindow::update));
    update ();
}

GUI::DialogWindow::~DialogWindow ()
{
    window->destroy ();
    gtk_object_unref (GTK_OBJECT (gui));
}

void GUI::DialogWindow::show ()
{
    window->show ();
    window->get_window ().raise ();
}

int GUI::DialogWindow::delete_event_impl (GdkEventAny *e)
{
    window->hide ();
    return true;
}

void GUI::DialogWindow::resource_created_cb (Resources::Resource *new_res)
{
    if (new_res->get_type () != Resources::RESOURCE_STRING)
	return;

    update_block = true;

    Gtk::ListItem *item = new Gtk::ListItem (new_res->get_id ());
    item->set_user_data (new_res);
    item->show ();
    help_combo->get_list ()->add (*manage (item));
    help_combo->get_entry ()->set_text (res->get_help_id ());

    if (new_res->get_id () == res->get_help_id ())
	    help_res = new_res;

    new_res->changed.connect (bind (
	slot (this, &DialogWindow::help_res_changed_cb),
	new_res));
    
    update_block = false;
}

void GUI::DialogWindow::resource_removed_cb (Resources::Resource *help_res)
{
    if (help_res->get_type () != Resources::RESOURCE_STRING)
	return;

    update_block = true;

    set<Resources::Resource*> resource_set = ResourceManager::get_resources ();
    vector<string> help_list;
    for (set<Resources::Resource*>::iterator i = resource_set.begin ();
	 i != resource_set.end (); i++)
    {
	if ((*i)->get_type () == Resources::RESOURCE_STRING &&
	    (*i) != help_res)
	    help_list.push_back ((*i)->get_id ());
    }
    help_combo->set_popdown_strings (help_list);
    
    update_block = false;

    if (help_res->get_id () == res->get_help_id ())
	res->set_help_id ("");
}

void GUI::DialogWindow::help_res_changed_cb (Resources::Resource *changed_res)
{
    if (changed_res == help_res)
	res->set_help_id (help_res->get_id ());

    update_block = true;
    using namespace Gtk::List_Helpers;

    ItemList& itemlist = help_combo->get_list ()->items ();
    ItemList::iterator i = itemlist.begin ();
    while (i != itemlist.end () && (*i)->get_user_data () != changed_res)
	i++;
    g_return_if_fail (i != itemlist.end ());

    (*i)->remove ();
    (*i)->add_label (changed_res->get_id ());
    
    update_block = false;
}

void GUI::DialogWindow::update ()
{
    gchar *title_buf = g_strdup_printf (_("Dialog: %s"), res->get_id ().c_str ());
    window->set_title (title_buf);
    g_free (title_buf);

    update_block = true;
    
    id_entry->set_text (res->get_id ());
    title_entry->set_text (res->get_title ());

    if (text_entry->get_length())
	text_entry->delete_text (0, text_entry->get_length ());
    text_entry->insert (res->get_text ());

    type_combo->set_history (res->get_dialog_type ());
    help_combo->get_entry ()->set_text (res->get_help_id ());
    help_res = ResourceManager::get_resource (res->get_help_id ());

    if (help_res)
	btnHelpEdit->set_sensitive (true);
    else
	btnHelpEdit->set_sensitive (false);

    
    vector<string> buttons = res->get_buttons ();
    vector<string>::iterator i;
    unsigned int j;
    unsigned int default_button = res->get_default_button ();
    
    button_list->items().clear ();
    for (i = buttons.begin (), j = 0;
	 i != buttons.end (); i++, j++)
    {
	string label;

	if (j != default_button)
	    label = *i;
	else
	    label = "*" + *i + "*";
	
	Gtk::ListItem *item = new Gtk::ListItem (label);
	item->show ();
	item->set_user_data (GUINT_TO_POINTER (j));
	
	button_list->items().push_back (*manage (item));
    }
    
    update_block = false;
}

void GUI::DialogWindow::id_entry_cb (GdkEventFocus *e)
{
    res->set_id (id_entry->get_text ());
}

void GUI::DialogWindow::title_entry_cb (GdkEventFocus *e)
{
    res->set_title (title_entry->get_text ());
}

void GUI::DialogWindow::text_entry_cb ()
{
    if (update_block)
	return;

    res->set_text (text_entry->get_chars (0, text_entry->get_length ()));
}

void GUI::DialogWindow::type_menu_cb (Resources::DialogType type)
{
    res->set_dialog_type (type);
}

void GUI::DialogWindow::help_combo_cb ()
{
    if (update_block)
	return;

    res->set_help_id (help_combo->get_entry ()->get_text ());
}

void GUI::DialogWindow::help_edit_cb ()
{
    g_return_if_fail (help_res != NULL);

    help_res->show_ui ();
}

void GUI::DialogWindow::dialog_add_cb (string name)
{
    if (name != "")
    {
	vector<string> buttons = res->get_buttons ();	
	buttons.push_back (name);
	res->set_buttons (buttons);
    }
}

void GUI::DialogWindow::button_add_cb ()
{
    Gnome::Dialog *dialog = Gnome::Dialogs::request
	(*window, false,
	 _("Enter name of new button"),
	 "", 0,
	 slot (this, &DialogWindow::dialog_add_cb));

    dialog->run_and_close ();
}

void GUI::DialogWindow::button_remove_cb ()
{
    g_return_if_fail (button_list->selection ().size () == 1);
    
    Gtk::List::SelectionList::iterator sel = button_list->selection ().begin ();  
    unsigned int index = GPOINTER_TO_UINT ((*sel)->get_user_data ());

    vector<string> buttons = res->get_buttons ();
    buttons.erase (buttons.begin () + index);
    res->set_buttons (buttons);

    if (index == res->get_default_button ())
	res->set_default_button (0);
}

void GUI::DialogWindow::button_default_cb ()
{
    g_return_if_fail (button_list->selection ().size () == 1);

    Gtk::List::SelectionList::iterator sel = button_list->selection ().begin ();
    unsigned int index = GPOINTER_TO_UINT ((*sel)->get_user_data ());

    res->set_default_button (index);
}

void GUI::DialogWindow::button_up_cb ()
{
    Gtk::List::SelectionList sel_list = button_list->selection ();
    g_return_if_fail (sel_list.size () == 1);

    unsigned int index = GPOINTER_TO_UINT ((*(sel_list.begin ()))->get_user_data ());
    g_return_if_fail (index != 0);

    vector<string> buttons = res->get_buttons ();
    vector<string>::iterator curr = buttons.begin () + index;
    vector<string>::iterator prev = curr - 1;
    iter_swap (curr, prev);    
    res->set_buttons (buttons);

    unsigned int old_default = res->get_default_button ();

    if (index == old_default)
	res->set_default_button (index - 1);
    if (index - 1 == old_default)
	res->set_default_button (index);

    button_list->select_item (index - 1);
}

void GUI::DialogWindow::button_down_cb ()
{
    Gtk::List::SelectionList sel_list = button_list->selection ();
    g_return_if_fail (sel_list.size () != 0);

    unsigned int index = GPOINTER_TO_UINT ((*(sel_list.begin ()))->get_user_data ());
    g_return_if_fail (index < button_list->items ().size () - 1);

    vector<string> buttons = res->get_buttons ();
    vector<string>::iterator curr = buttons.begin () + index;
    vector<string>::iterator next = curr + 1;
    iter_swap (curr, next);
    res->set_buttons (buttons);

    unsigned int old_default = res->get_default_button ();

    if (index == old_default)
	res->set_default_button (index + 1);
    if (index + 1 == old_default)
	res->set_default_button (index);

    button_list->select_item (index + 1);
}

void GUI::DialogWindow::button_sel_change_cb ()
{
    Gtk::List::SelectionList sel_list = button_list->selection ();

    if (!sel_list.size ())
    {
	
	btnRemove->set_sensitive (false);
	btnDefault->set_sensitive (false);
	btnUp->set_sensitive (false);
	btnDown->set_sensitive (false);
	
    } else {
	
	g_return_if_fail (sel_list.size() == 1);

	unsigned int index = GPOINTER_TO_UINT ((*(sel_list.begin()))->get_user_data ());
    
	btnRemove->set_sensitive (true);
	btnDefault->set_sensitive (true);
	btnUp->set_sensitive ((index > 0));
	btnDown->set_sensitive ((index < (button_list->items ().size () - 1)));
    }
}
