//$Id: resource-manager.cc,v 1.12 2001/10/20 16:06:50 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "resource-manager.h"

#include "mainwin.h"
#include "resource-util.h"

#include "dialog-res.h"
#include "form-res.h"
#include "menu-res.h"
#include "string-res.h"
#include "blob-res.h"

#include <fstream>
#include <gnome-xml/parser.h>

using namespace Guikachu;

map<string_case, Resources::Resource*> ResourceManager::resource_map;
set<string_case> ResourceManager::id_set;
Resources::Application *ResourceManager::app = 0;
SigC::Signal1<void, Resources::Resource*> ResourceManager::resource_created;
SigC::Signal1<void, Resources::Resource*> ResourceManager::resource_removed;

bool ResourceManager::dirty = false;

string ResourceManager::create_id (const string& prefix)
{
    unsigned int num = 0;
    string ret_val;
    gchar *curr_name;

    for (num = 1; num != 0; num++)
    {
	curr_name = g_strdup_printf ("%s%d", prefix.c_str (), num);

	if (id_set.find (curr_name) == id_set.end ())
	{
	    ret_val = curr_name;
	    g_free (curr_name);
	    break;
	} else {
	    g_free (curr_name);
	}
    }

    if (!num)
	return "";

    return ret_val;
}

bool ResourceManager::register_id (const string& id)
{
    if (id_set.find (id) != id_set.end ())
    {
	char *error_msg = g_strdup_printf (_("Non-unique identifier `%s'"), id.c_str ());
	GUI::show_error (error_msg);
	g_free (error_msg);
	return false;
    }

    id_set.insert (id);
    return true;
}

void ResourceManager::unregister_id (const string& id)
{
    id_set.erase (id);
}

Resources::Resource* ResourceManager::create_resource (Resources::Type  type,
						       const string    &id_ = "")
{
    string id = id_;
    
    if (id == "")
	id = create_id (Resources::type_id_from_type (type));
    
    if (!register_id (id))
	return 0;

    string real_id = id;
    for (string::iterator i = real_id.begin ();
	 i != real_id.end (); i++)
    {
	if (*i == ' ')
	    *i = '_';
    }
    
    Resources::Resource *res;

    switch (type)
    {
    case Resources::RESOURCE_DIALOG:
	res = new Resources::Dialog (real_id);
	break;
    case Resources::RESOURCE_STRING:
	res = new Resources::String (real_id);
	break;
    case Resources::RESOURCE_BLOB:
	res = new Resources::Blob (real_id);
	break;
    case Resources::RESOURCE_MENU:
	res = new Resources::Menu (real_id);
	break;
    case Resources::RESOURCE_FORM:
	Resources::Form *form = new Resources::Form (real_id);
	form->widget_changed.connect (SigC::slot (&ResourceManager::resource_changed_cb));
	res = form;
	break;
    }
    
    resource_map.insert (pair<string_case, Resources::Resource*> (real_id, res));

    res->changed.connect (SigC::slot (&ResourceManager::resource_changed_cb));

    dirty = true;
    
    resource_created.emit (res);
    
    return res;
}

bool ResourceManager::rename_resource (const string& old_id,
				       const string& new_id)
{
    // No-ops: old_id == new_id
    if (old_id == new_id)
	return true;

    if (string_case (old_id) == string_case (new_id))
	return true;

    // Error: empty id
    if (new_id == "")
	return false;

    // Error: old_id does not exist
    if (id_set.find (old_id) == id_set.end ())
	return false;

    if (id_set.find (new_id) != id_set.end ())
    {
	char *error_msg = g_strdup_printf (_("Non-unique identifier `%s'"), new_id.c_str ());
	GUI::show_error (error_msg);
	g_free (error_msg);

	return false;
    }
    
    id_set.erase (old_id);
    id_set.insert (new_id);
    
    map<string_case, Resources::Resource*>::iterator map_i = resource_map.find (old_id);
    
    if (map_i != resource_map.end ())
    {
	Resources::Resource *res = map_i->second;
	resource_map.erase (map_i);
	resource_map.insert (pair<string_case, Resources::Resource*> (new_id, res));
    }

    return true;
}

void ResourceManager::remove_resource (Resources::Resource* resource)
{
    resource_removed.emit (resource);

    unregister_id (resource->id);
    resource_map.erase (resource->id ());
    
    delete resource;

    dirty = true;
}

bool ResourceManager::is_dirty ()
{
    return dirty;
}

void ResourceManager::resource_changed_cb ()
{
    dirty = true;
}

set<Resources::Resource*> ResourceManager::get_resources ()
{
    set<Resources::Resource*> ret;

    for (map<string_case, Resources::Resource*>::const_iterator i =
	     resource_map.begin (); i != resource_map.end (); i++)
    {
	ret.insert (i->second);
    }

    return ret;
}

Resources::Resource* ResourceManager::get_resource (const string& id)
{
    map<string_case, Resources::Resource*>::const_iterator found;

    found = resource_map.find (id);
    if (found == resource_map.end ())
	return NULL;
    else
	return found->second;
}
