//$Id: resource-manager-io.cc,v 1.12 2001/10/19 20:17:56 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "resource-manager.h"

#include "config.h"
#include <libgnome/libgnome.h>

#include <fstream>
#include <time.h>
#include <gnome-xml/parser.h>

#include "mainwin.h"
#include "storage.h"

using namespace Guikachu;

void ResourceManager::reset ()
{
    map<string_case, Resources::Resource*>::iterator i;
    
    for (i = resource_map.begin ();
	 i != resource_map.end ();
	 i++)
    {
	remove_resource (i->second);
    }

    if (!app)
	app = new Resources::Application ();
    else
	app->reset ();
    
    dirty = false;
}

void ResourceManager::load (const string& filename)
{
    reset ();

    Storage storage;

    if (!storage.load (filename))
    {
	gchar *buffer = g_strdup_printf (_("Unable to open `%s'"),
					 g_basename (filename.c_str ()));
	GUI::show_error (buffer);
	g_free (buffer);

	return;
    }
	
    StorageNode root_node = storage.get_root ();
    if (root_node.name () != "guikachu")
    {
	gchar *buffer = g_strdup_printf (_("Unable to open `%s': "
					   "Not a valid Guikachu document"),
					 g_basename (filename.c_str ()));
	GUI::show_error (buffer);
	g_free (buffer);
        return;
    }

    /*
     * Walk the tree
     */
    for (StorageNode curr_node = root_node.children ();
         curr_node; curr_node++)
    {
	if (curr_node.name () == "application")
	    app->load (curr_node);
	else
	    load_resource (curr_node);
    }
    
    dirty = false;
}

Resources::Resource* ResourceManager::load_resource (StorageNode &node)
{
    Resources::Resource *resource = 0;

    string node_name = node.name ();
    string id = node.get_prop<string> ("id");

    if (id == "")
    {
	g_warning ("`%s': Missing ID field", node_name.c_str ());
	return NULL;
    }

    if (node_name == "dialog")
	resource = create_resource (Resources::RESOURCE_DIALOG, id);
    else if (node_name == "form")
	resource = create_resource (Resources::RESOURCE_FORM, id);
    else if (node_name == "menu")
	resource = create_resource (Resources::RESOURCE_MENU, id);
    else if (node_name == "string")
	resource = create_resource (Resources::RESOURCE_STRING, id);
    else if (node_name == "blob")
	resource = create_resource (Resources::RESOURCE_BLOB, id);
    else
	g_warning ("Unknown resource type `%s'", node_name.c_str ());

    if (resource)
	resource->load (node);
    
    return resource;
}

void ResourceManager::save (const string& filename)
{
    Storage     storage;
    StorageNode root = storage.create_root ("guikachu");
    StorageNode node;

    // Save per-application data
    node = root.add_node ("application");
    app->save (node);
    
    // Save resources
    for (map<string_case, Resources::Resource*>::iterator i = resource_map.begin ();
	 i != resource_map.end ();
	 i++)
    {
	Resources::Resource *resource = i->second;
	string curr_node_name;

	switch (resource->get_type ())
	{
	case Resources::RESOURCE_STRING:
	    curr_node_name = "string";
	    break;
	case Resources::RESOURCE_BLOB:
	    curr_node_name = "blob";
	    break;
	case Resources::RESOURCE_DIALOG:
	    curr_node_name = "dialog";
	    break;
	case Resources::RESOURCE_MENU:
	    curr_node_name = "menu";
	    break;
	case Resources::RESOURCE_FORM:
	    curr_node_name = "form";
	    break;
	}

	node = root.add_node (curr_node_name);
	node.set_prop ("id", resource->id);
	resource->save (node);
    }

    storage.save (filename);

    dirty = false;
}

void ResourceManager::export_rcp (const string& filename)
{
    ofstream fout (filename.c_str ());

    time_t curr_date = time (0);
    const char* date_str = ctime (&curr_date);
    
    fout << "/*" << endl;
    fout << " * This file was created by Guikachu. Do not edit it manually." << endl;
    fout << " * Creation date: " << date_str; // ctime ()'s return value ends in \n
    fout << " */" << endl << endl;;
    
    fout << app;
    
    for (map<string_case, Resources::Resource*>::iterator i = resource_map.begin ();
	 i != resource_map.end ();
	 i++)
    {
	fout << i->second;
    }
    
    fout.close ();
}
