//$Id: checkbox.cc,v 1.13 2001/10/23 13:05:51 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "checkbox.h"
#include "checkbox-canvas.h"

#include "form-editor.h"

#include <gnome-xml/xmlmemory.h>

using Guikachu::Widgets::Checkbox;
using namespace Guikachu::GUI::FormEditor;

Checkbox::Checkbox (Resources::Form *owner,
		    const string& id):
    Widget (owner, id),
    Resizeable (changed),
    Textual (changed, id),
    editor (0),
    usable (changed, true),
    anchor_right (changed, false),
    group (this),
    toggled (this)
{
}

Checkbox::~Checkbox ()
{
    if (editor)
	delete editor;
}

Checkbox::GroupProp::GroupProp (Checkbox *owner_):
    Property<int> (owner_->changed),
    owner (owner_)
{
}

Checkbox::ToggledProp::ToggledProp (Checkbox *owner_):
    Property<bool> (owner_->changed),
    owner (owner_)
{
}

void Checkbox::ToggledProp::set_val (const bool& value_)
{
    if (value_ == value)
	return;
    
    value = value_;
    if (value_ && owner->group)
    {
	set<Widgets::Widget*> widgets = owner->get_form ()->get_widgets ();
	for (set<Widgets::Widget*>::iterator i = widgets.begin ();
	     i != widgets.end (); i++)
	{
	    if ((*i)->get_type () == WIDGET_CHECKBOX &&
		(*i) != owner)
	    {
		Checkbox *checkbox = static_cast <Checkbox*> (*i);
		if (checkbox->group == owner->group && checkbox->toggled)
		    checkbox->toggled = false;
	    }
	}
    }
    
    changed ();
}

void Checkbox::GroupProp::set_val (const int& value_)
{   
    if (value_ == value)
	return;

    value = value_;
    
    if (value_)
    {
	set<Widgets::Widget*> widgets = owner->get_form ()->get_widgets ();
	set<Widgets::Widget*>::iterator i = widgets.begin ();
	bool found = false;

	// See if we there is another toggled widget in the new group
	while (!found && i != widgets.end ())
	{
	    if ((*i)->get_type () == WIDGET_CHECKBOX &&
		(*i) != owner)
	    {
		Checkbox *checkbox = static_cast <Checkbox*> (*i);
		if (checkbox->group == owner->group && checkbox->toggled)
		    found = true;
	    }
	    
	    i++;
	}

	// If there is, untoggle ourselves
	if (found && owner->toggled)
	    owner->toggled = false;
    }
    
    changed ();
}

Gtk::Widget *Checkbox::create_editor ()
{
    if (!editor)
	editor = new CheckboxProperties (this);
    
    return editor->get_editor ();
}

CanvasItem* Checkbox::create_canvas_item (GUI::FormEditor::Form *form)
{
    return new CheckboxCanvasItem (this, form);
}

bool Checkbox::load (StorageNode &node)
{
    /* Non-NULL default values */
    usable = false;
    
    for (StorageNode curr_node  = node.children ();
	 curr_node; curr_node++)
    {
	if (curr_node.name () == "pos")
	{
	    x = curr_node.get_prop<int> ("x");
	    y = curr_node.get_prop<int> ("y");

	    string width_str = curr_node.get_prop<string> ("width");
	    if (width_str == "auto")
	    {
		manual_width = false;
		width = 0;
	    } else {
		manual_width = true;
		width = atoi (width_str.c_str ());
	    }

	    string height_str = curr_node.get_prop<string> ("height");
	    if (height_str == "auto")
	    {
		manual_height = false;
		height = 0;
	    } else {
		manual_height = true;
		height = atoi (height_str.c_str ());
	    }
	}
	else if (curr_node.name () == "text")
	{
	    text = curr_node.get_content ();
	    font = curr_node.get_prop<int> ("font");
	}
	else if (curr_node.name () == "usable")
	    usable = true;
	else if (curr_node.name () == "anchor_right")
	    anchor_right = true;
	else if (curr_node.name () == "group")
	    group = curr_node.get_prop<int> ("id");
	else if (curr_node.name () == "toggled")
	    toggled = true;
	else
	    g_warning ("Error parsing `checkbox' widget: "
		       "unexpected element `%s'",
		       curr_node.name ().c_str ());
    }

    changed ();

    return true;
}

void Checkbox::save (StorageNode &node) const
{
    StorageNode my_node;

    /* Position */
    my_node = node.add_node ("pos");
    my_node.set_prop ("x", x);
    my_node.set_prop ("y", y);

    /* Size */
    if (manual_width)
	my_node.set_prop ("width", width);
    else
	my_node.set_prop ("width", "auto");

    if (manual_height)
	my_node.set_prop ("height", height);
    else
	my_node.set_prop ("height", "auto");
    
    /* Usable */
    if (usable)
	node.add_node ("usable");

    /* Group ID */
    my_node = node.add_node ("group");
    my_node.set_prop ("id", group ());

    /* Toggled */
    if (toggled)
	node.add_node ("toggled");
    
    /* Anchor_right */
    if (anchor_right)
	node.add_node ("anchor_right");
    
    /* Label and font */
    my_node = node.add_node ("text", text);
    my_node.set_prop ("font", font);
}

void Checkbox::save_rcp (ostream &ostr) const
{
    gchar *width_str, *height_str;

    if (manual_width)
	width_str = g_strdup_printf ("%d", width ());
    else
	width_str = g_strdup ("AUTO");
    
    if (manual_height)
	height_str = g_strdup_printf ("%d", height ());
    else
	height_str = g_strdup ("AUTO");

    ostr << "  CHECKBOX \"" << text << "\" ID " << id;
    ostr << " AT (" << x << " " << y << " " << width_str << " " << height_str << ")";
    ostr << " FONT " << font;
    ostr << " GROUP " << group;
    if (usable)
	ostr << " USABLE";
    else
	ostr << " NONUSABLE";
    if (anchor_right)
	ostr << " RIGHTANCHOR";
    else
	ostr << " LEFTANCHOR";
    if (toggled)
	ostr << " CHECKED";
    
    ostr << endl;

    g_free (height_str);
    g_free (width_str);
}

int Checkbox::get_auto_width ()
{
    return 18 + get_string_width (font, text);
}

int Checkbox::get_auto_height ()
{
    return 3 + get_font_height (font);
}
