//$Id: form-editor.cc,v 1.12 2001/09/22 18:54:11 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "form-editor.h"

#include <gnome--/canvas.h>
#include <gnome--/canvas-group.h>
#include <gnome--/canvas-text.h>
#include <gnome--/canvas-polygon.h>

#include <gdk-pixbuf/gnome-canvas-pixbuf.h>

using namespace Guikachu::GUI;
using Guikachu::GUI::FormEditor::Font;

#define GUIKACHU_TITLE_FONT 1

static GdkPixbuf* clip_pixbuf (GdkPixbuf    *src,
			       GtkAnchorType anchor,
			       int           clip_width,
			       int           clip_height);

const Font &FormEditor::get_font (int font_num)
{
    static Font standard_fonts[3];
    static bool init[3] = {false, false, false};

    if (!init[font_num])
    {
	string font_filename;
	
	switch (font_num)
	{
	case 0:
	    font_filename = "stdfont.txt";
	    break;
	case 1:
	    font_filename = "boldfont.txt";
	    break;
	case 2:
	    font_filename = "bigfont.txt";
	    break;
	default:
	    g_assert_not_reached ();
	    break;
	}
	
	standard_fonts[font_num].parse (font_filename);
	init[font_num] = true;
    }
	
    return standard_fonts[font_num];
}

const Font &FormEditor::get_title_font ()
{
    return get_font (GUIKACHU_TITLE_FONT);
}

unsigned int FormEditor::get_font_height (int font_num)
{
    const Font &font = get_font (font_num);
    return font.get_height ();
}

unsigned int FormEditor::get_string_width (int font_num, const string& text)
{
    const Font &font = get_font (font_num);
    return font.string_width (text);
}

Gnome::CanvasItem* FormEditor::draw_text (Gnome::CanvasGroup &group,
					  const string &text, int font_num,
					  int x, int y,
					  bool usable = true,
					  GtkAnchorType anchor = GTK_ANCHOR_NW,
					  int clip_width = 0, int clip_height = 0)
{
    Font font = get_font (font_num);
    Gdk_Color color;

    if (usable)
	color = Gdk_Color (GUIKACHU_FGC);
    else
	color = Gdk_Color (GUIKACHU_FGC_DISABLED);

    return render_text (group, text, font, color,
			x, y,
			anchor,
			clip_width, clip_height);
}

Gnome::CanvasItem* FormEditor::render_text (Gnome::CanvasGroup &group,
					    const string       &text,
					    const Font         &font,
					    const Gdk_Color    &color,
					    int x, int y,
					    GtkAnchorType anchor = GTK_ANCHOR_NW,
					    int clip_width = 0, int clip_height = 0)
{
    Gnome::CanvasGroup *text_group = new Gnome::CanvasGroup (group, 0, 0);
    
    GdkPixbuf *pixbuf = font.render (text, color);

    if (!pixbuf)
	return 0;
    
    GdkPixbuf *clipped_pixbuf;
    if (clip_width || clip_height)
	clipped_pixbuf = clip_pixbuf (pixbuf, anchor, clip_width, clip_height);
    else
	clipped_pixbuf = gdk_pixbuf_ref (pixbuf);
    gdk_pixbuf_unref (pixbuf);
    
    int width = gdk_pixbuf_get_width (clipped_pixbuf);
    int height = gdk_pixbuf_get_height (clipped_pixbuf);

    // Horizontal offset
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_SOUTH_WEST:
	break;
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_SOUTH:
	x -= width / 2;
	break;
    case GTK_ANCHOR_NORTH_EAST:
    case GTK_ANCHOR_EAST:
    case GTK_ANCHOR_SOUTH_EAST:
	x -= width;
    }

    // Vertical offset
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_NORTH_EAST:
	break;
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_EAST:
	y -= height / 2;
	break;
    case GTK_ANCHOR_SOUTH_WEST:
    case GTK_ANCHOR_SOUTH:
    case GTK_ANCHOR_SOUTH_EAST:
	y -= height;
	break;
    }

    // Draw the text
    gnome_canvas_item_new (text_group->gtkobj (),
			   gnome_canvas_pixbuf_get_type (),
			   "pixbuf", clipped_pixbuf,
			   "x", (double)x,
			   "y", (double)y,
			   0);
    gdk_pixbuf_unref (clipped_pixbuf);

    // Draw a transparent background block (to get mouse events for
    // the whole area)
    {
	Gnome::CanvasPoints bg_points;
	using Gnome::Art::Point;
	
	bg_points.push_back (Point (x, y));
	bg_points.push_back (Point (x + width, y));
	bg_points.push_back (Point (x + width, y + height));
	bg_points.push_back (Point (x, y + height));

	new Gnome::CanvasPolygon (*text_group, bg_points);
    }
    
    return text_group;
}

GdkPixbuf* clip_pixbuf (GdkPixbuf     *src,
			GtkAnchorType  anchor,
			int            clip_width,
			int            clip_height)
{
    GdkPixbuf *dest;
    
    int width = gdk_pixbuf_get_width (src);
    int height = gdk_pixbuf_get_height (src);

    if (clip_width <= 0 || clip_width > width)
	clip_width = width;

    if (clip_height <= 0 || clip_height > height)
	clip_height = height;

    int src_x = 0;
    int src_y = 0;

    /* Set src_x */
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_SOUTH_WEST:
	src_x = 0;
	break;

    case GTK_ANCHOR_NORTH_EAST:
    case GTK_ANCHOR_EAST:
    case GTK_ANCHOR_SOUTH_EAST:
	src_x = width - clip_width;
	break;

    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_SOUTH:
	src_x = (width / 2) - (clip_width / 2);
	break;
    }

    /* Set src_y */
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_NORTH_EAST:
	src_y = 0;
	break;

    case GTK_ANCHOR_SOUTH_WEST:
    case GTK_ANCHOR_SOUTH:
    case GTK_ANCHOR_SOUTH_EAST:
	src_y = height - clip_height;
	break;

    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_EAST:
	src_y = (height / 2) - (clip_height / 2);
	break;
    }

    dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, true, 8,
			   clip_width, clip_height);

    gdk_pixbuf_copy_area (src,
			  src_x, src_y,
			  clip_width, clip_height,
			  dest, 0, 0);

    return dest;
}

