//$Id: menu-win.cc,v 1.3 2001/02/25 19:10:11 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "menu-win.h"

#include <gtk--.h>
#include <sigc++/retbind.h>

using namespace Guikachu;

GUI::MenuWindow::MenuWindow (Guikachu::Resources::Menu *res_):
    res (res_)
{
    gui = glade_xml_new (GLADEDIR "/guikachu.glade", NULL);

    using Gtk::wrap;
    using namespace SigC;
    
    window = wrap (GTK_WINDOW (glade_xml_get_widget (gui, "res_menu")));
    window->delete_event.connect (slot (this, &MenuWindow::delete_event_impl));

    id_entry = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "menu_id")));

    // Menu items list
    clist = wrap (GTK_CLIST (glade_xml_get_widget (gui, "menu_tree")));
    btnUp = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_up")));
    btnDown = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_down")));
    btnRemove = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_remove")));

    // New item buttons
    btnSubmenu = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_submenu")));
    btnItem = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_item")));
    btnSeparator = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_separator")));

    // Menu item properties
    item_label = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "menuitem_label")));
    lbl_item_label = wrap (GTK_LABEL (glade_xml_get_widget (gui, "menuitem_label_label")));

    item_id = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "menuitem_id")));
    lbl_item_id = wrap (GTK_LABEL (glade_xml_get_widget (gui, "menuitem_id_label")));

    item_shortcut = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "menuitem_shortcut")));
    lbl_item_shortcut = wrap (GTK_LABEL (glade_xml_get_widget (gui, "menuitem_shortcut_label")));

    id_entry->activate.connect (bind (slot (this, &MenuWindow::id_entry_cb), (GdkEventFocus*) 0));
    id_entry->focus_out_event.connect (retbind (slot (this, &MenuWindow::id_entry_cb), 0));

    item_id->activate.connect (bind (slot (this, &MenuWindow::item_id_entry_cb), (GdkEventFocus*) 0));
    item_label->activate.connect (bind (slot (this, &MenuWindow::item_label_entry_cb), (GdkEventFocus*) 0));
    item_shortcut->activate.connect (bind (slot (this, &MenuWindow::item_shortcut_entry_cb), (GdkEventFocus*) 0));

    btnUp->clicked.connect (slot (this, &MenuWindow::btn_up_cb));
    btnDown->clicked.connect (slot (this, &MenuWindow::btn_down_cb));
    btnRemove->clicked.connect (slot (this, &MenuWindow::btn_remove_cb));

    btnSubmenu->clicked.connect (slot (this, &MenuWindow::btn_submenu_cb));
    btnItem->clicked.connect (bind (slot (this, &MenuWindow::btn_item_cb), false));
    btnSeparator->clicked.connect (bind (slot (this, &MenuWindow::btn_item_cb), true));
    
    clist->select_row.connect (slot (this, &MenuWindow::clist_sel_row));
    
    res->changed.connect (slot (this, &MenuWindow::update));
    update ();
}

GUI::MenuWindow::~MenuWindow ()
{
    window->destroy ();
    gtk_object_unref (GTK_OBJECT (gui));
}

int GUI::MenuWindow::delete_event_impl (GdkEventAny *e)
{
    window->hide ();
    return true;
}

void GUI::MenuWindow::show ()
{
    window->show_all ();
}

void GUI::MenuWindow::update ()
{
    window->set_title ("Menu: " + res->get_id ());

    id_entry->set_text (res->get_id ());

    submenus = res->get_submenus ();

    int selected_row = 0;
    if (clist->selection ().size ())
	selected_row = clist->selection ().begin ()->get_row_num ();

    clist->clear ();
    clist_rows.clear ();
    reset_controls ();
    
    unsigned int submenu_num = 0, menuitem_num = 0;
    
    for (vector<Resources::Submenu>::iterator i = submenus.begin ();
	 i != submenus.end (); i++, submenu_num++)
    {
	RowData row_data;
	row_data.is_submenu = true;
	row_data.index = submenu_num;
	clist_rows.push_back (row_data);
	
	vector<string> row;
	row.push_back (i->label);
	row.push_back ("");
	row.push_back ("");
	
	clist->rows ().push_back (row);
	
	menuitem_num = 0;
	for (vector<Resources::MenuItem>::iterator j = i->items.begin ();
	     j != i->items.end (); j++, menuitem_num++)
	{
	    vector<string> row;
	    
	    if (!j->separator)
	    {
		gchar *shortcut_str = g_strdup_printf ("%c", j->shortcut);
		
		row.push_back ("    " + j->label);
		row.push_back (j->id);
		row.push_back (shortcut_str);

		g_free (shortcut_str);
		
	    } else {
		
		row.push_back ("    ------");
		row.push_back ("");
		row.push_back ("");
	    }

	    RowData row_data;
	    row_data.is_submenu = false;
	    row_data.index = menuitem_num;
	    row_data.parent_index = submenu_num;
	    clist_rows.push_back (row_data);

	    clist->rows ().push_back (row);
	}
    }

    clist->select_row (selected_row);
}

void GUI::MenuWindow::reset_controls ()
{
    btnDown->set_sensitive (false);
    btnRemove->set_sensitive (false);
    btnUp->set_sensitive (false);

    btnItem->set_sensitive (false);
    btnSeparator->set_sensitive (false);
    
    item_label->set_text ("");
    item_label->set_sensitive (false);
    lbl_item_label->set_sensitive (false);

    item_id->set_text ("");
    item_id->set_sensitive (false);
    lbl_item_id->set_sensitive (false);

    item_shortcut->set_text ("");
    item_shortcut->set_sensitive (false);
    lbl_item_shortcut->set_sensitive (false);
}

void GUI::MenuWindow::clist_sel_row (gint row, gint column,
				     GdkEvent *event)
{
    RowData& row_data = clist_rows[row];

    if (clist->selection ().size () != 1)
	return;
    
    reset_controls ();
    
    btnRemove->set_sensitive (true);
    btnItem->set_sensitive (true);
    btnSeparator->set_sensitive (true);
    
    if (row_data.is_submenu)
    {
	// This is a submenu
	
	Resources::Submenu& submenu = submenus[row_data.index];

	btnUp->set_sensitive (row_data.index != 0);
	btnDown->set_sensitive (row_data.index < (submenus.size () - 1));

	item_label->set_text (submenu.label);
	item_label->set_sensitive (true);
	lbl_item_label->set_sensitive (true);

    } else {
	// This is a menu item

	Resources::Submenu& submenu = submenus[row_data.parent_index];
	Resources::MenuItem& menuitem = submenu.items[row_data.index];

	btnUp->set_sensitive (row_data.index != 0);
	btnDown->set_sensitive (row_data.index < (submenu.items.size () - 1));
	
	if (!menuitem.separator)
	{
	    item_label->set_text (menuitem.label);
	    item_label->set_sensitive (true);
	    lbl_item_label->set_sensitive (true);
	    
	    item_id->set_text (menuitem.id);
	    item_id->set_sensitive (true);
	    lbl_item_id->set_sensitive (true);

	    gchar *shortcut_str = g_strdup_printf ("%c", menuitem.shortcut);
	    item_shortcut->set_text (shortcut_str);
	    item_shortcut->set_sensitive (true);
	    lbl_item_shortcut->set_sensitive (true);
	    g_free (shortcut_str);
	}
    }
}

void GUI::MenuWindow::id_entry_cb (GdkEventFocus *e)
{
    res->set_id (id_entry->get_text ());
}

void GUI::MenuWindow::item_id_entry_cb (GdkEventFocus *e)
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    if (clist_rows[row].is_submenu)
	return;

    vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
    items[index].id = item_id->get_text ();

    res->set_submenus (submenus);
}

void GUI::MenuWindow::item_label_entry_cb (GdkEventFocus *e)
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    if (clist_rows[row].is_submenu)
    {
	submenus[index].label = item_label->get_text ();

	res->set_submenus (submenus);

    } else {

	vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
	items[index].label = item_label->get_text ();
	
	res->set_submenus (submenus);
    }
}

void GUI::MenuWindow::item_shortcut_entry_cb (GdkEventFocus *e)
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    if (clist_rows[row].is_submenu)
	return;

    vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
    string shortcut_str = item_shortcut->get_text ();

    if (shortcut_str == "")
	items[index].shortcut = 0;
    else
	items[index].shortcut = shortcut_str[0];

    res->set_submenus (submenus);    
}

void GUI::MenuWindow::btn_up_cb ()
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;
    g_return_if_fail (index != 0);
    
    if (clist_rows[row].is_submenu)
    {
	vector<Resources::Submenu>::iterator curr = submenus.begin() + index;
	vector<Resources::Submenu>::iterator prev = curr - 1;
	int offset = prev->items.size() + 1;

	iter_swap (curr, prev);
	res->set_submenus (submenus);

	clist->select_row (row - offset);
	
    } else {
	
	vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
	vector<Resources::MenuItem>::iterator curr = items.begin() + index;
	vector<Resources::MenuItem>::iterator prev = curr - 1;

	iter_swap (curr, prev);
	res->set_submenus (submenus);

	clist->select_row (row - 1);
    }
}

void GUI::MenuWindow::btn_down_cb ()
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    
    if (clist_rows[row].is_submenu)
    {
	g_return_if_fail (index < (submenus.size() - 1));
    
	vector<Resources::Submenu>::iterator curr = submenus.begin() + index;
	vector<Resources::Submenu>::iterator next = curr + 1;
	int offset = next->items.size() + 1;

	iter_swap (curr, next);
	res->set_submenus (submenus);
    
	clist->select_row (row + offset);

    } else {
	vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
	vector<Resources::MenuItem>::iterator curr = items.begin() + index;
	vector<Resources::MenuItem>::iterator next = curr + 1;

	g_return_if_fail (index < (items.size () - 1));
	
	iter_swap (curr, next);
	res->set_submenus (submenus);

	clist->select_row (row + 1);
    }
}

void GUI::MenuWindow::btn_remove_cb ()
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    if (clist_rows[row].is_submenu)
    {
	submenus.erase (submenus.begin () + index);
	res->set_submenus (submenus);
	
    } else {

	vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
	items.erase (items.begin () + index);

	res->set_submenus (submenus);	
    }
}

void GUI::MenuWindow::btn_submenu_cb ()
{
    if (clist->selection ().size () == 0)
    {
	Resources::Submenu submenu ("New submenu");
	submenus.push_back (submenu);

	res->set_submenus (submenus);

	return;
    }

    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;
    vector<Resources::Submenu>::iterator new_index;
    Resources::Submenu submenu ("New submenu");
    
    if (clist_rows[row].is_submenu)
    {
	new_index = submenus.begin () + index + 1;
    } else {
	new_index = submenus.begin () + clist_rows[row].parent_index + 1;
    }

    submenus.insert (new_index, submenu);
    
    res->set_submenus (submenus);
    
}

void GUI::MenuWindow::btn_item_cb (bool separator)
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;
    unsigned int parent_index = clist_rows[row].parent_index;
    vector<Resources::MenuItem>::iterator pos;
    
    Resources::MenuItem menuitem;
    menuitem.separator = separator;
    
    if (!separator)
    {
	menuitem.id = ResourceManager::create_id ("MenuItem_");
	menuitem.label = menuitem.id;
	menuitem.shortcut = 0;
    }
    
    if (clist_rows[row].is_submenu)
    {
	pos = submenus[index].items.begin ();	
	submenus[index].items.insert (pos, menuitem);
	
    } else {
	
	pos = submenus[parent_index].items.begin () + index + 1;
	submenus[parent_index].items.insert (pos, menuitem);
    }
    
    res->set_submenus (submenus);    
}
