#!/usr/bin/env python
# $Id: goblin.py 63 2007-07-08 08:33:54Z cactus $

###########################################################################
# Copyright (C) 2006 Gergő Érdi
# http://cactus.rulez.org/projects/goblin/
#
# Published under the terms of the GNU General Public License 2.0
###########################################################################

import cairo
import gtk
import gtk.glade

import math # for math.pi

class Player:
    def __init__ (self, name, color):
        self.name = name
        self.color = color

    def __str__ (self):
        return self.name
                
class Players:
    Red  = Player ("Red", (1, 0, 0))
    Blue = Player ("Blue", (0, 0, 1))

class Piece:
    def __init__ (self, player, size):
        self.player = player
        self.size = size

    def draw (self, ctx, picked):
        if picked: a = 0.2
        else: a = 1
        
        # Draw perimeter
        ctx.set_source_rgba (0, 0, 0, a)
        ctx.set_line_width (9)
        ctx.arc(50, 50, 8 * self.size - 4, 0, 2 * math.pi)
        ctx.close_path()
        ctx.stroke_preserve ()
        
        # Draw contents
        (r, g, b) = self.player.color
        ctx.set_source_rgba (r, g, b, a)
        ctx.arc(50, 50, 8 * self.size, 0, 2 * math.pi)
        ctx.close_path ()
        ctx.fill ()

class Position:
    def __init__ (self, reserve = False, pieces = []):
        self.reserve = reserve
        self.pieces = pieces[:]
        self.picked = False

    def empty (self):
        return not self.pieces

    def top (self):
        return self.pieces[-1]

    def pop (self):
        return self.pieces.pop()

    def push (self, piece):
        self.pieces.append (piece)

    def can_push (self, piece):
        if self.reserve: return False
        return self.empty() or self.top().size < piece.size
    
    def draw (self, ctx):
        if not self.reserve:
            ctx.set_source_rgb (0, 0, 0)
            ctx.arc (50, 50, 2, 0, 2 * math.pi)
            ctx.close_path ()
            ctx.fill ()

        for piece in self.pieces:
            picked = self.picked and piece == self.top ()
            piece.draw (ctx, picked)

dash_offset = 0
def timeout ():
    global dash_offset
    dash_offset = (dash_offset + 1) % 17;

import gobject
gobject.timeout_add (30, lambda *args: timeout () or True)
    
class PositionWidget (gtk.DrawingArea):
    def __init__ (self, board):
        gtk.DrawingArea.__init__ (self)

        self.board = board
        self.position = Position ()
        self.highlight = False

        def redraw_highlight ():
            if self.highlight:
                self.queue_draw ()
            
        def expose ():
            ctx = self.window.cairo_create()

            self.position.draw (ctx)

            if (self.highlight):
                ctx.set_source_rgb (1, 0.95, 0)
                ctx.set_line_width (4)
                ctx.set_dash ([10, 7], -dash_offset)
                ctx.arc (50, 50, 8 * 5 - 2, 0, 2 * math.pi)
                ctx.close_path ()
                ctx.stroke ()

        self.connect ('expose_event', lambda *args: expose ())
        gobject.timeout_add (30, lambda *args: redraw_highlight () or True)
        
        self.add_events (gtk.gdk.ENTER_NOTIFY_MASK | gtk.gdk.LEAVE_NOTIFY_MASK | gtk.gdk.BUTTON_PRESS_MASK)
        self.set_size_request (100, 100)

    def begin_pop (self):
        self.position.picked = True
        self.queue_draw ()
        return self.position.top ()

    def finish_pop (self):
        self.position.picked = False
        self.queue_draw ()
        return self.position.pop ()

    def set_highlight (self):
        if not self.highlight:
            self.highlight = True
            self.queue_draw ()

    def clear_highlight (self):
        if self.highlight:
            self.highlight = False
            self.queue_draw ()    

class ReserveWidget (PositionWidget):
    def __init__ (self, board, player):
        PositionWidget.__init__ (self, board)
        self.position = Position (True, [Piece(player, 1), Piece(player, 2), Piece(player, 3), Piece(player, 4)])

class Board:            
    def __init__ (self, table, box_red, box_blue, statusbar):
        self.table = table
        self.boxes = [box_red, box_blue]
        self.statusbar = statusbar

        self.reset ()

    def game_message (self, msg):
        statusbar.push (statusbar.get_context_id ('game-state'), msg)
        
    def finish_round (self):
        def get_winner ():
            rows = [[(x, y) for x in range (0, 4)] for y in range (0, 4)]
            cols = [[(x, y) for y in range (0, 4)] for x in range (0, 4)]
            diags = [[(xy, xy) for xy in range (0, 4)], [(xy, 3 - xy) for xy in range (0, 4)]]
        
            for player in (self.other_player, self.current_player):
                def owns_position (position):
                    return not position.empty() and position.top().player == player

                def forall (pred, list):
                    return not filter (lambda i: not pred (i), list)
            
                for structure in (rows, cols, diags):
                    for region in structure:
                        if forall (owns_position, [self.positions[x][y].position for (x,y) in region]):
                            return (player, region)

            return (None, None)

        (winner, winning_pieces) = get_winner ()
        if winner:
            self.finished = True
            
            for (x, y) in winning_pieces:
                self.positions[x][y].set_highlight ()

            self.game_message ('Player %s has won.' % winner)
            win_dialog (winner)
        else:
            self.current_player, self.other_player = self.other_player, self.current_player
            self.game_message ('%s\'s turn' % self.current_player)

    def reset (self):
        def setup_position_widget (position_widget):
            def can_pick (position):
                if self.finished: return False
                
                if self.picked_widget:
                    return position != self.picked_widget.position and position.can_push (self.picked_widget.position.top ())
                else:
                    return not position.empty () and position.top ().player == self.current_player
                
            def button_press_cb ():
                if not can_pick (position_widget.position): return
                
                if not self.picked_widget:
                    self.picked_widget = position_widget
                    self.picked_widget.begin_pop ()
                else:
                    position_widget.position.push (self.picked_widget.finish_pop ())
                    position_widget.clear_highlight()
                    self.picked_widget = None
                    self.finish_round ()
                        
            def hover_enter_cb ():
                if self.finished: return
                if can_pick (position_widget.position):
                    position_widget.set_highlight ()
                        
            def hover_leave_cb ():
                if self.finished: return
                position_widget.clear_highlight ()

            position_widget.connect ('button_press_event', lambda *args: button_press_cb ())
            position_widget.connect ('enter_notify_event', lambda *args: hover_enter_cb ())
            position_widget.connect ('leave_notify_event', lambda *args: hover_leave_cb ())

            position_widget.show ()
            return position_widget
        
        # Create game board
        self.positions = []
        self.table.foreach (self.table.remove)
        for x in range (0, 4):
            self.positions.append ([])
            for y in range (0, 4):
                position_widget = PositionWidget (self)
                setup_position_widget (position_widget)
                self.table.attach (position_widget, x, x + 1, y, y + 1)
                self.positions[x].append (position_widget)
            
        # Create reserves
        for (box, player) in zip (self.boxes, [Players.Red, Players.Blue]):
            box.foreach (box.remove)
            for x in range (0, 3):
                position_widget = ReserveWidget (self, player)
                setup_position_widget (position_widget)
                box.pack_start (position_widget, False, False)

        # Reset state
        self.current_player = Players.Red
        self.other_player = Players.Blue
        self.picked_widget = None
        self.finished = False

        # Start game
        self.game_message ('Game started')

import sys
import os.path

gladepath = os.path.join (os.path.dirname (sys.argv[0]), 'goblin.glade')
glade = gtk.glade.XML (gladepath)

def about():
    about = glade.get_widget('about')
    about.show ()
    #about.hide ()

def win_dialog (player):
    global win
    dialog = gtk.MessageDialog (win, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                gtk.MESSAGE_INFO, gtk.BUTTONS_OK,
                                "Player %s has won the game." % player)
    dialog.run ()
    dialog.hide ()

table = glade.get_widget ('table')
box_red = glade.get_widget ('box-red')
box_blue = glade.get_widget ('box-blue')
statusbar = glade.get_widget('statusbar')

board = Board (table, box_red, box_blue, statusbar)

# Set up signals
glade.signal_autoconnect ({'on_window_delete_event':      gtk.main_quit,
                           'on_about_delete_event':       lambda about, *args: about.hide () or True,
                           'on_menuQuit_activate':        lambda *args: gtk.main_quit (),
                           'on_menuNew_activate':         lambda *args: board.reset (),
                           'on_menuAbout_activate':       lambda *args: about ()})

win = glade.get_widget ('window')
win.show_all ()
gtk.main ()
