\usepackage[magyar]{babel}
\usepackage[mathletters]{ucs}
\usepackage{amsmath}
\usepackage{amsfonts}
\usepackage{booktabs}
\usepackage{fontspec}
\usepackage{hyperref}

\mode<presentation>{\usecolortheme{crane}}
                
\newcommand{\vektor}[1]{#1}        
\newcommand{\dx}{\ensuremath{\Delta x}}
\newcommand{\dy}{\ensuremath{\Delta y}}
\newcommand{\ds}{\ensuremath{\Delta s}}
                                                                  
\newcommand{\xx}{\ensuremath{x + \dx}} 
\newcommand{\yy}{\ensuremath{y + \dy}}
\renewcommand{\ss}{\ensuremath{s + \ds}}

\hypersetup{colorlinks=true,urlcolor=blue,citecolor=black,linkcolor=black}
                                           
%include polycode.fmt
\arrayhs %% Otherwise beamerarticle mangles up \\'s in code inside frames

%%format * = " \cdot "
%%format (recip u) = u "^{-1}"
%format (dot (x) (y)) = "\langle" x " , " y "\rangle"
%format (dot x y) = x "^T " y

%% %format = = ":="        
        
%format delta x s mu = "\delta (" x "," s "," mu ")"

%format PNorm1 = 1
%format PNorm2 = 2
%format Infinity = "\infty"
%format (pnorm p (v)) = "\norm{" v "}_" "{" p "}"

%format (sqrt (x)) = "\sqrt{" x "}"

%format xi = "x_{i}"
%format yi = "y_{i}"
%format si = "s_{i}"

%format x0 = "x_0"
%format y0 = "y_0"
%format s0 = "s_0"

%format mu = "\mu"        
%format mu' = mu "^{\prime}"
%format nu = "\nu"
%format nu' = nu "^{\prime}"
%format xi = "\xi"
%format zeta = "\zeta"        
%format tau = "\tau"
%format theta = "\theta"
%format alpha = "\alpha"
%format lambda = "\lambda"
%format lambda_b = lambda "_{" b "}"
%format lambda_c = lambda "_{" c "}"
%format lambda_mu = lambda "_{" mu "}"
        
%format (scale lambda v) = lambda " \cdot " v
%format <> = " \cdot "
%format <> = " "
%format (mul x y) = x y

%format rb = "r_{b}"
%format rc = "r_{c}"

%format =~= = "\approx"        
        
%format (trans m) = m "^{T}"

%format (inv m) = m "^{-1}"        
                        
\hypersetup{%
    pdftitle={Lineáris belsőpontos Newton-iteráció},
    pdfauthor={Dr. Érdi Gergő},
    pdfkeywords={lineáris programozás, Newton-iteráció, Newton-barrier,
                 belsőpontos iteráció, perturbált lineáris programozási feladat,
                 Haskell}
}%        
                
\newcommand{\comment}[1]{}         

\newcommand{\norm}[1]{\lVert#1\rVert}

\title{Lineáris belsőpontos Newton-iteráció}
\subtitle{\large{Implementáció Haskellben}}
\date{}
\author{Dr. Érdi Gergő \\ \url{http://gergo.erdi.hu/}}        
                                                          
\begin{document}
\begin{frame}        
\maketitle
\end{frame}        

Az alábbiakban összeállítunk egy Haskell modult, amely a belsőpontos Newton-iteráció
algoritmusával old meg lineáris programozási feladatokat.
\begin{frame}<presentation>{Haskell}
  \begin{itemize}
    \item Funkcionális programozási nyelv
    \item Specifikáció szinte a kész program is egyben
    \item Erős statikus típusellenőrzés
    \item Lusta kiértékelés
    \item Párhuzamosság támogatása
  \end{itemize}
\end{frame}

\begin{code}
module InteriorNewton where
\end{code}

\section{Reprezentáció}        

A lineáris algebrai műveletek megvalósítását a |HMatrix| csomagból vesszük át:
        
\begin{code}
import Foreign.Storable
import Data.Packed.Vector
import Data.Packed.Matrix
import Numeric.LinearAlgebra.Algorithms
import Numeric.LinearAlgebra.Linear
import Numeric.LinearAlgebra.Interface
import Numeric.LinearAlgebra.Instances
\end{code}

A megoldási folyamat naplózásához a |Writer| monádot használjuk:
        
\begin{code}
import qualified Control.Monad.Writer as W
import Control.Monad.Loops (dropWhileM)
import Control.Monad (liftM)
\end{code}

\begin{frame}<presentation>{Segédkönyvtárak}
  \begin{itemize}
    \item \emph{HMatrix}: Lineáris algebrai könyvtár, a \emph{GNU
      Scientific Library} programcsomag Haskell-es interface-e
    \item \emph{mtl} (\emph{Monad Transformer Library}):
      |Control.Monad.Writer| monád a naplózáshoz
    \item \emph{Parsec}: Monadikus parser-generátor a tesztesetekhez
  \end{itemize}
\end{frame}
                               
%format (nicediv (p) (q)) = "\dfrac{" p "}{" q "}"
%format (nicepow x (y)) = x "^{" y "}"
%format idx v (i) = "{" v "}" "_{" i "}"
\comment{
\begin{code}
-- LHS2TeX kludges
nicediv = (/)
nicepow = (^^)
idx = (@>)
\end{code}
}                
                        
Kétszeres pontosságú, lebegőpontos számábrázolást fogunk használni,
ezért minden vektorunk és mátrixunk |Double| elemeket tárol. Az
egyszerűség kedvéért bevezetjük az alábbi típus-szinonímákat.  A
|toVec| segédfüggvényre azért van szükségünk, mert a |HMatrix|
csomag megkülönbözteti a vektorokat és az egyoszlopos mátrixokat.
        
\begin{code}
type Vec = Vector Double
type Mtx = Matrix Double
{-"\vspace{1em}"-}

toVec :: Mtx -> Vec
toVec = head . toColumns             
\end{code}

\begin{frame}
  \frametitle<presentation>{A kitűzött feladat}
A megoldandó lineáris programozási feladatot az alábbi formában várjuk el:

\begin{align*}
        \min\, c^Tx \\
        A\,x = b \\
        x \ge 0
\end{align*}
\end{frame}

Egy ilyen feladatot az alábbi algebrai adattípussal reprezentálunk:

\begin{code}
data Problem = Problem {  prob_A :: Mtx,
                          prob_b :: Vec,
                          prob_c :: Vec }
             deriving Show
\end{code}

\section{$\varepsilon$-egyenlőség} 

\begin{frame}\frametitle<presentation>{$\varepsilon$-egyenlőség}
  
A lebegőpontos számábrázolás korlátai, illetve a numerikus hibák
feltorlódása miatt számok, illetve vektorok egyenlőségét nem érdemes
szigorúan néznünk, ezért bevezetjük az alábbi |=~=| relációt:

%format epsilon = "\varepsilon"
%format abs (x) = "|" x "|"
        
\begin{code}
infix 4 =~=
class Approx a where
    (=~=) :: a -> a -> Bool
{-"\vspace{1em}"-}
epsilon = nicepow 10 (-6)
{-"\vspace{1em}"-}    
instance Approx Double where
    x =~= y = nicediv (2 * abs(x-y)) (1 + abs(x) + abs(y)) < epsilon
{-"\vspace{1em}"-}    
instance (Storable a, Approx a) => Approx (Vector a) where
   x =~= y = and $ zipWith (=~=)  (toList x) (toList y)               
\end{code}
\end{frame}

\section{Naplózás}
        
\begin{code}
data OptimizationStep  =  Infeasible (Vec, Vec, Vec) Double
                       |  Feasible (Vec, Vec, Vec)
                         deriving Show
{-"\vspace{1em}"-}
type Logging a = W.Writer [OptimizationStep] a
{-"\vspace{1em}"-}
writeLog :: OptimizationStep -> Logging ()
writeLog x = W.tell [x]
{-"\vspace{1em}"-}
iterateUntil :: (a -> Bool) -> (a -> OptimizationStep) -> (a -> a) -> a -> Logging a
iterateUntil isFinished logger step initial = liftM head $ dropWhileM notFinished (iterate step initial)
    where notFinished x = do writeLog (logger x)
                             return (not (isFinished x))
\end{code}
        
\section{A feladat perturbálása}
\begin{frame}\frametitle<presentation>{Perturbálás}
Ahhoz, hogy kezdeti belső megoldásunk legyen, a feladatot az alábbi
formában perturbáljuk, azzal a céllal, hogy $\nu = 1, \zeta = 1$
esetén választhassuk az $(\vektor{e},\vektor{0},\vektor{e})$
vektor-hármast kezdeti megoldásnak, és $\nu \to 0$ esetén az eredeti
feladathoz tartsunk:

\begin{displaymath}
\begin{array}{llrcl}
             & A\,x &    &=& b - \nu\,r_b \\
A^{T}\,y \,+ &      & s  &=& c - \nu\,r_c 
\end{array}        
\end{displaymath}

ahol

\begin{displaymath}
\begin{array}{lcl}
        r_b &:=& b - \zeta A\,e \vspace{0.5em} \\
        r_c &:=& c - \zeta e
\end{array}        
\end{displaymath}

A megvalósítás során a |zeta| paraméter értékét fixen 1-en tartjuk.        
\end{frame}
        
%format a = "A"        
%format aT = "A^T"
        
\begin{code}
perturbation :: Problem -> (Vec, Vec)
perturbation (Problem a b c) = (rb, rc)
    where  rb = b - (scale zeta a<>e)
           rc = c - (scale zeta e)
           {-"\vspace{1em}"-}
           zeta = 1
           (n, m) = (cols a, rows a)
           e = constant 1 n
\end{code}
        
\section{Centrális út}        

A centrális utat az alábbi, $\mu$-vel paraméterezett egyenletrendszer
$(x,y,s)$ megoldásaival definiáljuk:

\begin{displaymath}      
\begin{array}{llrclc}
             & A\,x &    &=& b \\
A^{T}\,y \,+ &      & s  &=& c \\
             &      & xs &=& \mu\,e
\end{array}
\end{displaymath}

\section{Haladás az eredeti feladatba, illetve a centrális úton}
\begin{frame}\frametitle<presentation>{A Newton-rendszer}

\only<article>{
Adott |nu| és |(x,y,s)| belső, perturbált megoldás esetén olyan
|(x',y',s')| belső megoldát keresünk, amelyre teljesülnek a feltételek
valamilyen |nu'| és |mu| paraméterek esetén:
}

\begin{displaymath}      
\begin{array}{llrclc}
              & A\,x' &      &=& b - \nu'\,r_b \\
A^{T}\,y' \,+ &       & s'   &=& c - \nu'\,r_c \\
              &       & x's' &=& \mu\,e
\end{array}
\end{displaymath}

Ehhez az alábbi formában állítjuk elő őket:

\begin{align*}
 x' := x + \alpha\,\dx \\ 
 y' := y + \alpha\,\dy \\
 s' := s + \alpha\,\ds 
\end{align*}                     

\only<article>{
Ahol $\alpha \in (0, 1]$-et úgy választjuk, hogy a megengedettség és a
pozitivitás megmaradjon, $(\dx, \dy, \ds)$-et pedig $(x,y,s)$
megoldás-voltát kihasználva, az előjel-kötöttséget elhagyva kapjuk:
}
  
\begin{displaymath}      
\begin{array}{llrclc}
                 & A\, x    &            &=& b - \nu\,r_b \\    
                 & A\,(\xx) &            &=& b - \nu'\,r_b\\
  A^T\,y\, +     &          &         s  &=& c - \nu\,r_c \\                 
  A^T\,(\yy)\, + &          &      (\ss) &=& c - \nu'\,r_c\\
                 &          & (\xx)(\ss) &=& \mu\,e
\end{array}
\end{displaymath}
\end{frame}

\begin{frame}\frametitle<presentation>{A Newton-rendszer (folyt.)}
Az egyenletrendszert átalakítva, a kvadratikus tagot elhagyva a
következő lineáris egyenletrendszer alapján számítható ki
$(\dx,\dy,\ds)$:

\begin{displaymath}      
\begin{array}{llrclc}
                 & A\,\dx     &        &=& (\nu - \nu')\,r_b \\
  A^T\,\dy\, +   &            &    \ds &=& (\nu - \nu')\,r_c \\
                 & s\,\dx\, + & x\,\ds &=& \mu\,e - x\,s
\end{array}
\end{displaymath}     

\only<article>{
Ennek az iterációs lépésnek az irányát számítja ki az alábbi függvény:
                
%format dx = "\Delta " x
%format dy = "\Delta " y
%format ds = "\Delta " s
%format dnu = "\Delta " nu
%format null n m = "0_{" n "," m "}"
%format i = "I"
                          
\begin{code}
dirNewton :: Problem -> Double -> Double -> (Vec, Vec, Vec) -> (Vec, Vec, Vec)
dirNewton prob@(Problem a b c) mu dnu (x, y, s) = (dx, dy, ds)
    where  dx = subVector 0 n xi
           dy = subVector n m xi
           ds = subVector (n + m) n xi
           {-"\vspace{1em}"-}
           xi = toVec (linearSolve mtx (asColumn lambda))
           {-"\vspace{1em}"-}
           mtx = fromBlocks [  [a,   0,        0],
                               [0,   trans a,  i],
                               [mS,  0,        mX]]
               where  mS = diag s
                      mX = diag x
                      i = ident n
           {-"\vspace{1em}"-}               
           lambda = join [lambda_b, lambda_c, lambda_mu]
           lambda_b = scale dnu rb
           lambda_c = scale dnu rc
           lambda_mu = (scale mu e) - (mul x s)
           {-"\vspace{1em}"-}
           (rb, rc) = perturbation prob
           {-"\vspace{1em}"-}
           (n, m) = (cols a, rows a)
           e = mapVector (const 1) x
\end{code}
}

Az irány ismeretében már csak az $\alpha$ lépéshossz meghatározása van
hátra. Ehhez hányadostesztet végzünk az irány alapján, az eredményt
eltolva |epsilon|-nal a szigorú pozitivitás érdekében.

%%format alphaMax = "\lceil{}" alpha " \rceil{}"
%format alphaMax = "\alpha{}Max"
%format alphaMaxFromRatios = alphaMax "FromRatios"

\only<article>{
\begin{code}
alphaMaxFromRatios :: (Vec, Vec, Vec) -> (Vec, Vec, Vec) -> Double
alphaMaxFromRatios (x,y,s) (dx,dy,ds) = min 1 ((minimum ratios) - epsilon)
    where  ratios = ratiosX ++ ratiosS
           ratiosX = filter (> 0) $ zipWith ratio (toList x) (toList dx)
           ratiosS = filter (> 0) $ zipWith ratio (toList s) (toList ds)
           ratio z dz = (nicediv (- z) dz)
\end{code}
}
\end{frame}

\section{Kezdő belső pont keresése}
\begin{frame}<presentation>{Kezdő belső pont keresése}
  \begin{itemize}
    \item Cél: $\nu: 1 \to 0$ mellett |(x,y,s)| megoldás maradjon
    \item Kezdőpont: |(e, 0, e)|
    \item Lépésenként |dnu = nu| választással megoldjuk a
      Newton-rendszert
    \item Lépéshossz: pozitivitás megtartásával
    \item Leállási feltétel: $A\,x \approx b$
  \end{itemize}
\end{frame}

Az optimalizálás első fázisában $\nu$-t 1-ről 0-ra lenyomva keresünk
olyan pontot, amely az eredeti feladatnak belső pontja. Ehhez a
perturbált feladat $(e, 0, e)$ belső kezdőpontjából indítjuk a fent
definiált Newton-iterációt, úgy, hogy minden lépésben 0-ra próbáljuk
csökkenteni |nu'|-t (azaz |dnu = nu|). Az alábbi függvény az iteráció
egy lépését számítja ki úgy, hogy a tényleges |nu'|-t a Newton-lépés
megtétele után, egyszerű behelyettesítéssel végzi.
                           
\begin{code}
stepToFeasible :: Problem -> Double -> ((Vec, Vec, Vec), Double) -> ((Vec, Vec, Vec), Double)
stepToFeasible prob@(Problem a b c) mu ((x, y, s), nu) = ((x', y', s'), nu')
    where  (dx, dy, ds) = dirNewton prob mu nu (x, y, s)
           alpha = alphaMaxFromRatios (x, y, s) (dx, dy, ds)
           x' = x + (scale alpha dx)
           y' = y + (scale alpha dy)
           s' = s + (scale alpha ds)
\end{code}
%           {-"\vspace{1em}"-}
\begin{code}
           b' = a <> (x + (scale alpha dx))                
           (rb, _) = perturbation prob
           nu' = nicediv (idx (b - b') 0) (idx rb 0)
\end{code}

Minden adott tehát, hogy elkészíthessük azt a kezdőpont-kereső
függvényt, amelyik addig iterál, amíg az eredeti feladatnak egy
(|epsilon|-)megengedett belső pontját kapja. A |Writer| monádot
használjuk a számítási sorozat naplózásához.
                  
\begin{code}
{-"\vspace{1em}"-}               
initialInfeasible :: Problem -> (Vec, Vec, Vec)
initialInfeasible (Problem a b c) = (x, y, s)
    where  x = constant 1 n
           y = constant 0 m
           s = constant 1 n
           (n, m) = (cols a, rows a)                    
{-"\vspace{1em}"-}
solveToFeasible :: Problem -> Logging (Vec, Vec, Vec)
solveToFeasible prob@(Problem a b c) = liftM fst $ iterateUntil isFeasible toLog (stepToFeasible prob mu) ((x, y, s), nu)
    where  (x, y, s) = initialInfeasible prob
           mu = 1
           nu = 1           
           isFeasible ((x', y' ,s'), nu') = a <> x' =~= b
           toLog ((x, y, s), nu) = Infeasible (x, y, s) nu
\end{code}

\section{Optimalizálás}

\begin{frame}<presentation>{Megengedett optimalizálás}
  \begin{itemize}
    \item Cél: $\mu: 1 \to 0$ mellett |(x, y, s)| a centrális út
      közelében maradjon
    \item Kezdőpont: |(x0, y0, s0)| megengedett megoldás
    \item Irány: Newton-rendszer adott $\mu$-vel
    \item Lépéshossz: pozitivitás \alert{és dualitási rés}
    \item Leállási feltétel: dualitási rés $\approx 0$
  \end{itemize}
\end{frame}

Egy |(x, y, s)| belső pont birtokában végre elkezdhetjük a tényleges
belsőpontos optimalizálást, vagyis a centrális úton haladást $\mu$
csökkentésével. Lépésenként kiszámítjuk a javító irányt és
lépéshosszt, és ha $\mathcal{C}(\mu)$ megfelelő közelébe kerültünk,
akkor $\mu$-t is csökkentjük. A centrális úttól való távolságot az
alábbi függvénnyel értelmezzük:

\begin{code}
delta :: Vec -> Vec -> Double -> Double                     
delta x s mu = 0.5 * (pnorm PNorm2 (u - recip u))
    where  u = zipVector f x s
           f xi si = sqrt (nicediv (xi * si) mu)
\end{code}

A lépéshosszt pedig úgy választjuk meg, hogy amellett, hogy az $x, s >
0$ feltétel teljesüljön, ne „ugorjunk ki” az optimális megoldáson
átlépve a megengedett pontokból, vagyis a dualitási rés nemnegatív
maradjon.

\begin{code}
target (Problem a b c) (x, y, s) = dot x c

shadow (Problem a b c) (x, y, s) = dot y b

gap prob (x, y, s) = target prob (x, y, s) - shadow prob (x, y, s)
\end{code}
%{-"\vspace{1em}"-}
\begin{code}
                     
stepNewton :: Problem -> ((Vec, Vec, Vec), Double) -> ((Vec, Vec, Vec), Double)
stepNewton prob ((x, y, s), mu) = ((x', y', s'), mu')
    where  (dx, dy, ds) = dirNewton prob mu 0 (x, y, s)
           {-"\vspace{1em}"-}
           x' = x + (scale alpha dx)
           y' = y + (scale alpha dy)
           s' = s + (scale alpha ds)
\end{code}
%           {-"\vspace{1em}"-}
\begin{code}
           alphaMax = alphaMaxFromRatios (x, y, s) (dx, dy, ds)
           alpha = until isPosGap (*0.95) alphaMax
               where isPosGap alpha = gap prob (x', y', s') >= 0
                         where  x' = x + (scale alpha dx)
                                y' = y + (scale alpha dy)
                                s' = s + (scale alpha ds)                                     
           {-"\vspace{1em}"-}
           mu' = if delta x' s' mu < tau then (1 - theta) * mu else mu
           {-"\vspace{1em}"-}
           tau = 0.1
           theta = 0.8
\end{code}

Ezekután nincs más dolgunk, mint addig iterálni, amig optimális, azaz
|epsilon|-nál kiesebb dualitási résű megoldást nem találunk:

         
\begin{code}
solveNewton :: Problem -> (Vec, Vec, Vec) -> Logging (Vec, Vec, Vec)
solveNewton prob (x, y, s) = liftM fst $ iterateUntil optimal toLog (stepNewton prob) ((x, y, s), mu)
    where  mu = nicediv (dot x s) n
           n = fromIntegral (dim x)
           optimal ((x', y', s'), mu') = gap prob (x', y', s') =~= 0
           toLog ((x, y, s), mu) = Feasible (x, y, s)
{-"\vspace{1em}"-}                                         
optimize :: Problem -> Logging (Vec, Vec, Vec)
optimize prob = do (x, y, s) <- solveToFeasible prob
                   solveNewton prob (x, y, s)
\end{code}

\begin{frame}<presentation>{Teszt-feladatok}
  \begin{center}
    \usebeamercolor{block title}
    \only<presentation>{\rowcolors{1}{bg!40}{bg!60}}
    \begin{tabular}{||l||rr||rr||}\hline
      Név & $n$ & $m$ & Perturbált & Megengedett \\
      \hline
      \texttt{ADLITTLE} & 138 &  56 &  52 & 18 \\
      \texttt{AFIRO}    &  51 &  27 &  20 & 16 \\
      \texttt{BLEND}    & 114 &  74 &   8 & 24 \\
      \texttt{KB2}      &  77 &  52 &  38 & 26 \\
      \texttt{SC105}    & 163 & 105 &  18 & 24 \\
      \texttt{SC50A}    &  78 &  50 &  15 & 21 \\
      \texttt{SC50B}    &  78 &  50 &  16 & 19 \\
      \texttt{SCAGR7}   & 185 & 129 & 106 & 20 \\
      \texttt{SHARE2B}  & 162 &  96 &  17 & 26 \\
      \texttt{STOCFOR1} & 165 & 117 &  45 & 18 \\
      \hline
    \end{tabular}
  \end{center}
\end{frame}

\begin{frame}<presentation>{Köszönöm a figyelmet!}
  A program és a teljes dokumentáció letölthető az alábbi címről:

  \begin{itemize}
    \item \url{http://gergo.erdi.hu/elte/2009-2-nlp/}
  \end{itemize}
\end{frame}

\end{document}
